# frozen_string_literal: true

module Emails
  module Issues
    def new_issue_email(recipient_id, issue_id, reason = nil)
      setup_issue_mail(issue_id, recipient_id)

      mail_new_thread(@issue, issue_thread_options(@issue.author_id, recipient_id, reason))
    end

    def issue_due_email(recipient_id, issue_id, reason = nil)
      setup_issue_mail(issue_id, recipient_id)

      mail_new_thread(@issue, issue_thread_options(@issue.author_id, recipient_id, reason))
    end

    def new_mention_in_issue_email(recipient_id, issue_id, updated_by_user_id, reason = nil)
      setup_issue_mail(issue_id, recipient_id)
      mail_answer_thread(@issue, issue_thread_options(updated_by_user_id, recipient_id, reason))
    end

    # rubocop: disable CodeReuse/ActiveRecord
    def reassigned_issue_email(recipient_id, issue_id, previous_assignee_ids, updated_by_user_id, reason = nil)
      setup_issue_mail(issue_id, recipient_id)

      @previous_assignees = []
      @previous_assignees = User.where(id: previous_assignee_ids) if previous_assignee_ids.any?

      mail_answer_thread(@issue, issue_thread_options(updated_by_user_id, recipient_id, reason))
    end
    # rubocop: enable CodeReuse/ActiveRecord

    def closed_issue_email(recipient_id, issue_id, updated_by_user_id, reason = nil)
      setup_issue_mail(issue_id, recipient_id)

      @updated_by = User.find(updated_by_user_id)
      mail_answer_thread(@issue, issue_thread_options(updated_by_user_id, recipient_id, reason))
    end

    def relabeled_issue_email(recipient_id, issue_id, label_names, updated_by_user_id, reason = nil)
      setup_issue_mail(issue_id, recipient_id)

      @label_names = label_names
      @labels_url = project_labels_url(@project)
      mail_answer_thread(@issue, issue_thread_options(updated_by_user_id, recipient_id, reason))
    end

    def removed_milestone_issue_email(recipient_id, issue_id, updated_by_user_id, reason = nil)
      setup_issue_mail(issue_id, recipient_id)

      mail_answer_thread(@issue, issue_thread_options(updated_by_user_id, recipient_id, reason))
    end

    def changed_milestone_issue_email(recipient_id, issue_id, milestone, updated_by_user_id, reason = nil)
      setup_issue_mail(issue_id, recipient_id)

      @milestone = milestone
      @milestone_url = milestone_url(@milestone)
      mail_answer_thread(@issue, issue_thread_options(updated_by_user_id, recipient_id, reason).merge({
        template_name: 'changed_milestone_email'
      }))
    end

    def issue_status_changed_email(recipient_id, issue_id, status, updated_by_user_id, reason = nil)
      setup_issue_mail(issue_id, recipient_id)

      @issue_status = status
      @updated_by = User.find(updated_by_user_id)
      mail_answer_thread(@issue, issue_thread_options(updated_by_user_id, recipient_id, reason))
    end

    def issue_moved_email(recipient, issue, new_issue, updated_by_user, reason = nil)
      setup_issue_mail(issue.id, recipient.id)

      @new_issue = new_issue
      @new_project = new_issue.project
      mail_answer_thread(issue, issue_thread_options(updated_by_user.id, recipient.id, reason))
    end

    def import_issues_csv_email(user_id, project_id, results)
      @user = User.find(user_id)
      @project = Project.find(project_id)
      @results = results

      mail(to: @user.notification_email, subject: subject('Imported issues')) do |format|
        format.html { render layout: 'mailer' }
        format.text { render layout: 'mailer' }
      end
    end

    private

    def setup_issue_mail(issue_id, recipient_id)
      @issue = Issue.find(issue_id)
      @project = @issue.project
      @target_url = project_issue_url(@project, @issue)

      @sent_notification = SentNotification.record(@issue, recipient_id, reply_key)
    end

    def issue_thread_options(sender_id, recipient_id, reason)
      {
        from: sender(sender_id),
        to: recipient(recipient_id),
        subject: subject("#{@issue.title} (##{@issue.iid})"),
        'X-GitLab-NotificationReason' => reason
      }
    end
  end
end
