# frozen_string_literal: true

require 'spec_helper'

RSpec.describe Gitlab::BackgroundMigration::PopulateVulnerabilityReads, :migration, schema: 20220326161803 do
  let(:vulnerabilities) { table(:vulnerabilities) }
  let(:vulnerability_reads) { table(:vulnerability_reads) }
  let(:vulnerabilities_findings) { table(:vulnerability_occurrences) }
  let(:vulnerability_issue_links) { table(:vulnerability_issue_links) }
  let(:namespace) { table(:namespaces).create!(name: 'user', path: 'user') }
  let(:user) { table(:users).create!(email: 'author@example.com', username: 'author', projects_limit: 10) }
  let(:project) { table(:projects).create!(namespace_id: namespace.id) }
  let(:scanner) { table(:vulnerability_scanners).create!(project_id: project.id, external_id: 'test 1', name: 'test scanner 1') }
  let(:sub_batch_size) { 1000 }

  before do
    vulnerabilities_findings.connection.execute 'ALTER TABLE vulnerability_occurrences DISABLE TRIGGER "trigger_insert_or_update_vulnerability_reads_from_occurrences"'
    vulnerabilities.connection.execute 'ALTER TABLE vulnerabilities DISABLE TRIGGER "trigger_update_vulnerability_reads_on_vulnerability_update"'
    vulnerability_issue_links.connection.execute 'ALTER TABLE vulnerability_issue_links DISABLE TRIGGER "trigger_update_has_issues_on_vulnerability_issue_links_update"'

    10.times.each do |x|
      vulnerability = create_vulnerability!(
        project_id: project.id,
        report_type: 7,
        author_id: user.id
      )
      identifier = table(:vulnerability_identifiers).create!(
        project_id: project.id,
        external_type: 'uuid-v5',
        external_id: 'uuid-v5',
        fingerprint: Digest::SHA1.hexdigest(vulnerability.id.to_s),
        name: 'Identifier for UUIDv5')

      create_finding!(
        vulnerability_id: vulnerability.id,
        project_id: project.id,
        scanner_id: scanner.id,
        primary_identifier_id: identifier.id
      )
    end
  end

  it 'creates vulnerability_reads for the given records' do
    described_class.new.perform(vulnerabilities.first.id, vulnerabilities.last.id, sub_batch_size)

    expect(vulnerability_reads.count).to eq(10)
  end

  it 'does not create new records when records already exists' do
    described_class.new.perform(vulnerabilities.first.id, vulnerabilities.last.id, sub_batch_size)
    described_class.new.perform(vulnerabilities.first.id, vulnerabilities.last.id, sub_batch_size)

    expect(vulnerability_reads.count).to eq(10)
  end

  private

  def create_vulnerability!(project_id:, author_id:, title: 'test', severity: 7, confidence: 7, report_type: 0)
    vulnerabilities.create!(
      project_id: project_id,
      author_id: author_id,
      title: title,
      severity: severity,
      confidence: confidence,
      report_type: report_type
    )
  end

  # rubocop:disable Metrics/ParameterLists
  def create_finding!(
    project_id:, scanner_id:, primary_identifier_id:, vulnerability_id: nil,
    name: "test", severity: 7, confidence: 7, report_type: 0,
    project_fingerprint: '123qweasdzxc', location: { "image" => "alpine:3.4" }, location_fingerprint: 'test',
    metadata_version: 'test', raw_metadata: 'test', uuid: SecureRandom.uuid)
    vulnerabilities_findings.create!(
      vulnerability_id: vulnerability_id,
      project_id: project_id,
      name: name,
      severity: severity,
      confidence: confidence,
      report_type: report_type,
      project_fingerprint: project_fingerprint,
      scanner_id: scanner_id,
      primary_identifier_id: primary_identifier_id,
      location: location,
      location_fingerprint: location_fingerprint,
      metadata_version: metadata_version,
      raw_metadata: raw_metadata,
      uuid: uuid
    )
  end
  # rubocop:enable Metrics/ParameterLists
end
