---
stage: Verify
group: Pipeline Authoring
info: To determine the technical writer assigned to the Stage/Group associated with this page, see https://about.gitlab.com/handbook/product/ux/technical-writing/#assignments
type: reference
---

# CI/CD Components (Experimental)

> Introduced in GitLab 16.0 as an [experimental feature](../../policy/alpha-beta-support.md).

FLAG:
On self-managed GitLab, by default this feature is not available.
To make it available, ask an administrator to enable the feature flag named `ci_namespace_catalog_experimental`.
On GitLab.com, this feature is not available. This feature is not ready for production use.

This feature is an experimental feature and [an epic exists](https://gitlab.com/groups/gitlab-org/-/epics/9897)
to track future work. Tell us about your use case by leaving comments in the epic.

## Components Repository

A components repository is a GitLab project with a repository that hosts one or more pipeline components. A pipeline component is a reusable single pipeline configuration unit. You can use them to compose an entire pipeline configuration or a small part of a larger pipeline. It can optionally take [input parameters](../yaml/includes.md#define-input-parameters-with-specinputs).

### Create a components repository

To create a components repository, you must:

1. [Create a new project](../../user/project/index.md#create-a-blank-project) with a `README.md` file.

1. Create a `template.yml` file inside the project's root directory that contains the configuration you want to provide as a component. For example:

   ```yaml
   spec:
     inputs:
       stage:
         default: test
   ---
   component-job:
     script: echo job 1
     stage: $[[ inputs.stage ]]
   ```

### Directory structure

A components repository can host one or more components.

Components repositories must follow a mandatory file structure, containing:

- `template.yml`: The component configuration, one file per component. If there is
  only one component, this file can be in the root of the project. If there are multiple
  components, each file must be in a separate subdirectory.
- `README.md`: A documentation file explaining the details of the all the components in the repository.

For example, if the project is on GitLab.com, named `my-component`, and in a personal
namespace named `my-username`:

- Containing a single component and a simple pipeline to test the component, then
  the file structure might be:

  ```plaintext
  ├── template.yml
  ├── README.md
  └── .gitlab-ci.yml
  ```

  This component is referenced with the path `gitlab.com/my-username/my-component@<version>`.

- Containing one default component and multiple sub-components, then the file structure
  might be:

  ```plaintext
  ├── template.yml
  ├── README.md
  ├── .gitlab-ci.yml
  ├── unit/
  │   └── template.yml
  └── integration/
      └── template.yml
  ```

  These components are identified by these paths:

  - `gitlab.com/my-username/my-component`
  - `gitlab.com/my-username/my-component/unit`
  - `gitlab.com/my-username/my-component/integration`

It is possible to have a components repository with no default component, by having
no `template.yml` in the root directory.

**Additional notes:**

Nesting of components is not possible. For example:

```plaintext
├── unit/
│   └── template.yml
│   └── another_folder/
│       └── nested_template.yml
```

### Test a component

Testing components as part of the development workflow to ensure that quality maintains high standards is strongly recommended.

Testing changes in a CI/CD pipeline can be done, like any other project, by creating a `.gitlab-ci.yml` in the root directory.

For example:

```yaml
include:
  # include the component located in the current project from the current SHA
  - component: gitlab.com/$CI_PROJECT_PATH@$CI_COMMIT_SHA
    inputs:
      stage: build

stages: [build, test, release]

# Expect `component-job` is added.
# This is an example of testing that the included component works as expected.
# You can leverage GitLab API endpoints or 3rd party tools to inspect data generated by the component.
ensure-job-added:
  stage: test
  image: badouralix/curl-jq
  script:
    - |
      route="https://gitlab.com/api/v4/projects/$CI_PROJECT_ID/pipelines/$CI_PIPELINE_ID/jobs"
      count=`curl --silent --header "PRIVATE-TOKEN: $API_TOKEN" $route | jq 'map(select(.name | contains("component-job"))) | length'`
      if [ "$count" != "1" ]; then
        exit 1
      fi

# If we are tagging a release with a specific convention ("v" + number) and all
# previous checks succeeded, we proceed with creating a release automatically.
create-release:
  stage: release
  image: registry.gitlab.com/gitlab-org/release-cli:latest
  rules:
    - if: $CI_COMMIT_TAG =~ /^v\d+/
  script: echo "Creating release $CI_COMMIT_TAG"
  release:
    tag_name: $CI_COMMIT_TAG
    description: "Release $CI_COMMIT_TAG of components repository $CI_PROJECT_PATH"
```

After committing and pushing changes, the pipeline tests the component then releases it if the test passes.

### Release a component

Component repositories are released using the [`release`](../yaml/index.md#release) keyword within a CI pipeline.

Like in the [example above](#test-a-component), after all tests pass in a pipeline running for a tag ref, we can release a new version of the components repository.

All released versions of the components repository are displayed in the Components Catalog page for the given resource, providing users with information about official releases.

### Use a component in a CI/CD configuration

A pipeline component is identified by a unique address in the form `<fully-qualified-doman-name>/<component-path>@<version>`
containing:

- **A fully qualified domain name (FQDN)**: The FQDN must match the GitLab host.
- **A specific version**: The version of the component can be (in order of highest priority first):
  - A commit SHA, for example `gitlab.com/gitlab-org/dast@e3262fdd0914fa823210cdb79a8c421e2cef79d8`.
  - A tag. for example: `gitlab.com/gitlab-org/dast@1.0`.
  - `~latest`, which is a special version that always points to the most recent released tag,
     for example `gitlab.com/gitlab-org/dast@~latest`.
  - A branch name, for example `gitlab.com/gitlab-org/dast@main`.
- **A component path**: Contains the project's full path and the directory where the component YAML file `template.yml` is located.

For example, for a component repository located at `gitlab-org/dast` on `gitlab.com`:

- The path `gitlab.com/gitlab-org/dast` tries to load the `template.yml` from the root directory.
- The path `gitalb.com/gitlab-org/dast/api-scan` tries to load the `template.yml` from the `/api-scan` directory.

**Additional notes:**

- You can only reference components in the same GitLab instance as your project.
- If a tag and branch exist with the same name, the tag takes precedence over the branch.
- If a tag is named the same as a commit SHA that exists, like `e3262fdd0914fa823210cdb79a8c421e2cef79d8`,
  the commit SHA takes precedence over the tag.

## CI/CD Catalog

The CI/CD Catalog is a list of [components repositories](#components-repository),
each containing resources that you can add to your CI/CD pipelines.

### Mark the project as a catalog resource

After components are added to a components repository, they can immediately be [used](#use-a-component-in-a-cicd-configuration) to build pipelines in other projects.

However, this repository is not discoverable. You must mark this project as a catalog resource to allow it to be visible in the CI Catalog
so other users can discover it.

To mark a project as a catalog resource, run the following [graphQL](../../api/graphql/index.md)
mutation:

```graphql
mutation {
    catalogResourcesCreate(input: { projectPath: "path-to-project"}) {
      errors
  }
}
```
