# frozen_string_literal: true

require 'spec_helper'

RSpec.describe Issue, feature_category: :team_planning do
  include ExternalAuthorizationServiceHelpers

  using RSpec::Parameterized::TableSyntax

  let_it_be(:user) { create(:user) }
  let_it_be_with_reload(:reusable_project) { create(:project) }

  describe "Associations" do
    it { is_expected.to belong_to(:milestone) }
    it { is_expected.to belong_to(:project) }
    it { is_expected.to belong_to(:work_item_type).class_name('WorkItems::Type') }
    it { is_expected.to belong_to(:moved_to).class_name('Issue') }
    it { is_expected.to have_one(:moved_from).class_name('Issue') }
    it { is_expected.to belong_to(:duplicated_to).class_name('Issue') }
    it { is_expected.to belong_to(:closed_by).class_name('User') }
    it { is_expected.to have_many(:assignees) }
    it { is_expected.to have_many(:user_mentions).class_name("IssueUserMention") }
    it { is_expected.to have_many(:designs) }
    it { is_expected.to have_many(:design_versions) }
    it { is_expected.to have_one(:sentry_issue) }
    it { is_expected.to have_one(:alert_management_alert) }
    it { is_expected.to have_many(:alert_management_alerts).validate(false) }
    it { is_expected.to have_many(:resource_milestone_events) }
    it { is_expected.to have_many(:resource_state_events) }
    it { is_expected.to have_and_belong_to_many(:prometheus_alert_events) }
    it { is_expected.to have_and_belong_to_many(:self_managed_prometheus_alert_events) }
    it { is_expected.to have_many(:prometheus_alerts) }
    it { is_expected.to have_many(:issue_email_participants) }
    it { is_expected.to have_one(:email) }
    it { is_expected.to have_many(:timelogs).autosave(true) }
    it { is_expected.to have_one(:incident_management_issuable_escalation_status) }
    it { is_expected.to have_many(:issue_customer_relations_contacts) }
    it { is_expected.to have_many(:customer_relations_contacts) }
    it { is_expected.to have_many(:incident_management_timeline_events) }
    it { is_expected.to have_many(:assignment_events).class_name('ResourceEvents::IssueAssignmentEvent').inverse_of(:issue) }

    describe 'versions.most_recent' do
      it 'returns the most recent version' do
        issue = create(:issue, project: reusable_project)
        create_list(:design_version, 2, issue: issue)
        last_version = create(:design_version, issue: issue)

        expect(issue.design_versions.most_recent).to eq(last_version)
      end
    end
  end

  describe 'modules' do
    subject { described_class }

    it { is_expected.to include_module(Issuable) }
    it { is_expected.to include_module(Referable) }
    it { is_expected.to include_module(Sortable) }
    it { is_expected.to include_module(Taskable) }
    it { is_expected.to include_module(MilestoneEventable) }
    it { is_expected.to include_module(StateEventable) }

    it_behaves_like 'AtomicInternalId' do
      let(:internal_id_attribute) { :iid }
      let(:instance) { build(:issue) }
      let(:scope) { :namespace }
      let(:scope_attrs) { { namespace: instance.project.project_namespace } }
      let(:usage) { :issues }
    end
  end

  describe 'validations' do
    it { is_expected.not_to allow_value(nil).for(:confidential) }
    it { is_expected.to allow_value(true, false).for(:confidential) }
  end

  describe 'custom validations' do
    subject(:valid?) { issue.valid? }

    describe 'due_date_after_start_date' do
      let(:today) { Date.today }

      context 'when both values are not present' do
        let(:issue) { build(:issue) }

        it { is_expected.to be_truthy }
      end

      context 'when start date is present and due date is not' do
        let(:issue) { build(:work_item, start_date: today) }

        it { is_expected.to be_truthy }
      end

      context 'when due date is present and start date is not' do
        let(:issue) { build(:work_item, due_date: today) }

        it { is_expected.to be_truthy }
      end

      context 'when both date values are present' do
        context 'when due date is greater than start date' do
          let(:issue) { build(:work_item, start_date: today, due_date: 1.week.from_now) }

          it { is_expected.to be_truthy }
        end

        context 'when due date is equal to start date' do
          let(:issue) { build(:work_item, start_date: today, due_date: today) }

          it { is_expected.to be_truthy }
        end

        context 'when due date is before start date' do
          let(:issue) { build(:work_item, due_date: today, start_date: 1.week.from_now) }

          it { is_expected.to be_falsey }

          it 'adds an error message' do
            valid?

            expect(issue.errors.full_messages).to contain_exactly(
              'Due date must be greater than or equal to start date'
            )
          end
        end
      end
    end

    describe 'issue_type' do
      let(:issue) { build(:issue, issue_type: issue_type) }

      context 'when a valid type' do
        let(:issue_type) { :issue }

        it { is_expected.to eq(true) }
      end

      context 'empty type' do
        let(:issue_type) { nil }

        it { is_expected.to eq(false) }
      end
    end

    describe '#allowed_work_item_type_change' do
      where(:old_type, :new_type, :is_valid) do
        :issue     | :incident  | true
        :incident  | :issue     | true
        :test_case | :issue     | true
        :issue     | :test_case | true
        :issue     | :task      | false
        :test_case | :task      | false
        :incident  | :task      | false
        :task      | :issue     | false
        :task      | :incident  | false
        :task      | :test_case | false
      end

      with_them do
        it 'is possible to change type only between selected types' do
          issue = create(:issue, old_type, project: reusable_project)

          issue.assign_attributes(work_item_type: WorkItems::Type.default_by_type(new_type), issue_type: new_type)

          expect(issue.valid?).to eq(is_valid)
        end
      end
    end

    describe 'confidentiality' do
      let_it_be(:project) { create(:project) }

      context 'when parent and child are confidential' do
        let_it_be(:parent) { create(:work_item, confidential: true, project: project) }
        let_it_be(:child) { create(:work_item, :task, confidential: true, project: project) }
        let_it_be(:link) { create(:parent_link, work_item: child, work_item_parent: parent) }

        it 'does not allow to make child not-confidential' do
          issue = Issue.find(child.id)
          issue.confidential = false

          expect(issue).not_to be_valid
          expect(issue.errors[:base])
            .to include(_('A non-confidential issue cannot have a confidential parent.'))
        end

        it 'allows to make parent not-confidential' do
          issue = Issue.find(parent.id)
          issue.confidential = false

          expect(issue).to be_valid
        end
      end

      context 'when parent and child are not-confidential' do
        let_it_be(:parent) { create(:work_item, project: project) }
        let_it_be(:child) { create(:work_item, :task, project: project) }
        let_it_be(:link) { create(:parent_link, work_item: child, work_item_parent: parent) }

        it 'does not allow to make parent confidential' do
          issue = Issue.find(parent.id)
          issue.confidential = true

          expect(issue).not_to be_valid
          expect(issue.errors[:base])
            .to include(_('A confidential issue cannot have a parent that already has non-confidential children.'))
        end

        it 'allows to make child confidential' do
          issue = Issue.find(child.id)
          issue.confidential = true

          expect(issue).to be_valid
        end
      end
    end
  end

  subject { create(:issue, project: reusable_project) }

  describe 'callbacks' do
    describe '#ensure_metrics!' do
      it 'creates metrics after saving' do
        expect(subject.metrics).to be_persisted
        expect(Issue::Metrics.count).to eq(1)
      end

      it 'does not create duplicate metrics for an issue' do
        subject.close!

        expect(subject.metrics).to be_persisted
        expect(Issue::Metrics.count).to eq(1)
      end

      it 'records current metrics' do
        expect(Issue::Metrics).to receive(:record!)

        create(:issue, project: reusable_project)
      end

      context 'when metrics record is missing' do
        before do
          subject.metrics.delete
          subject.reload
        end

        it 'creates the metrics record' do
          subject.update!(title: 'title')

          expect(subject.metrics).to be_present
        end
      end
    end

    describe '#ensure_work_item_type' do
      let_it_be(:issue_type) { create(:work_item_type, :issue, :default) }
      let_it_be(:incident_type) { create(:work_item_type, :incident, :default) }
      let_it_be(:project) { create(:project) }

      context 'when a type was already set' do
        let_it_be(:issue, refind: true) { create(:issue, project: project) }

        it 'does not fetch a work item type from the DB' do
          expect(issue.work_item_type_id).to eq(issue_type.id)
          expect(WorkItems::Type).not_to receive(:default_by_type)

          expect(issue).to be_valid
        end

        it 'does not fetch a work item type from the DB when updating the type' do
          expect(issue.work_item_type_id).to eq(issue_type.id)
          expect(WorkItems::Type).not_to receive(:default_by_type)

          issue.update!(work_item_type: incident_type, issue_type: :incident)

          expect(issue.work_item_type_id).to eq(incident_type.id)
        end

        it 'ensures a work item type if updated to nil' do
          expect(issue.work_item_type_id).to eq(issue_type.id)

          expect do
            issue.update!(work_item_type: nil)
          end.to not_change(issue, :work_item_type).from(issue_type)
        end
      end

      context 'when no type was set' do
        let(:issue) { build(:issue, project: project, work_item_type: nil) }

        it 'sets a work item type before validation' do
          expect(issue.work_item_type_id).to be_nil

          issue.save!

          expect(issue.work_item_type_id).to eq(issue_type.id)
        end

        it 'does not fetch type from DB if provided during update' do
          expect(issue.work_item_type_id).to be_nil
          expect(WorkItems::Type).not_to receive(:default_by_type)

          issue.update!(work_item_type: incident_type, issue_type: :incident)

          expect(issue.work_item_type_id).to eq(incident_type.id)
        end
      end
    end

    describe '#check_issue_type_in_sync' do
      it 'raises an error if issue_type is out of sync' do
        issue = build(:issue, issue_type: :issue, work_item_type: WorkItems::Type.default_by_type(:task))

        expect do
          issue.save!
        end.to raise_error(Issue::IssueTypeOutOfSyncError)
      end

      it 'uses attributes to compare both issue_type values' do
        issue_type = WorkItems::Type.default_by_type(:issue)
        issue = build(:issue, issue_type: :issue, work_item_type: issue_type)

        attributes = double(:attributes)
        allow(issue).to receive(:attributes).and_return(attributes)

        expect(attributes).to receive(:[]).with('issue_type').twice.and_return('issue')
        expect(issue_type).to receive(:base_type).and_call_original

        issue.save!
      end
    end

    describe '#record_create_action' do
      it 'records the creation action after saving' do
        expect(Gitlab::UsageDataCounters::IssueActivityUniqueCounter).to receive(:track_issue_created_action)

        create(:issue)
      end

      it_behaves_like 'issue_edit snowplow tracking' do
        let(:issue) { create(:issue) }
        let(:project) { issue.project }
        let(:property) { Gitlab::UsageDataCounters::IssueActivityUniqueCounter::ISSUE_CREATED }
        let(:user) { issue.author }
        subject(:service_action) { issue }
      end
    end

    context 'issue namespace' do
      let(:issue) { build(:issue, project: reusable_project) }

      it 'sets the namespace_id' do
        expect(issue).to be_valid
        expect(issue.namespace).to eq(reusable_project.project_namespace)
      end

      context 'when issue is created' do
        it 'sets the namespace_id' do
          issue.save!

          expect(issue.reload.namespace).to eq(reusable_project.project_namespace)
        end
      end

      context 'when existing issue is saved' do
        let(:issue) { create(:issue) }

        before do
          issue.update!(namespace_id: nil)
        end

        it 'sets the namespace id' do
          issue.update!(title: "#{issue.title} and something extra")

          expect(issue.namespace).to eq(issue.project.project_namespace)
        end
      end
    end
  end

  context 'order by upvotes' do
    let!(:issue) { create(:issue) }
    let!(:issue2) { create(:issue) }
    let!(:award_emoji) { create(:award_emoji, :upvote, awardable: issue2) }

    describe '.order_upvotes_desc' do
      it 'orders on upvotes' do
        expect(described_class.order_upvotes_desc.to_a).to eq [issue2, issue]
      end
    end

    describe '.order_upvotes_asc' do
      it 'orders on upvotes' do
        expect(described_class.order_upvotes_asc.to_a).to eq [issue, issue2]
      end
    end
  end

  describe '.with_alert_management_alerts' do
    subject { described_class.with_alert_management_alerts }

    it 'gets only issues with alerts' do
      alert = create(:alert_management_alert, project: reusable_project, issue: create(:issue, project: reusable_project))
      issue = create(:issue, project: reusable_project)

      expect(subject).to contain_exactly(alert.issue)
      expect(subject).not_to include(issue)
    end
  end

  describe '.simple_sorts' do
    it 'includes all keys' do
      expect(described_class.simple_sorts.keys).to include(
        *%w(created_asc created_at_asc created_date created_desc created_at_desc
            closest_future_date closest_future_date_asc due_date due_date_asc due_date_desc
            id_asc id_desc relative_position relative_position_asc updated_desc updated_asc
            updated_at_asc updated_at_desc title_asc title_desc))
    end
  end

  describe '.with_issue_type' do
    let_it_be(:issue) { create(:issue, project: reusable_project) }
    let_it_be(:incident) { create(:incident, project: reusable_project) }

    it 'gives issues with the given issue type' do
      expect(described_class.with_issue_type('issue'))
        .to contain_exactly(issue)
    end

    it 'gives issues with the given issue type' do
      expect(described_class.with_issue_type(%w(issue incident)))
        .to contain_exactly(issue, incident)
    end
  end

  describe '.order_severity' do
    let_it_be(:issue_high_severity) { create(:issuable_severity, severity: :high).issue }
    let_it_be(:issue_low_severity) { create(:issuable_severity, severity: :low).issue }
    let_it_be(:issue_no_severity) { create(:incident) }

    context 'sorting ascending' do
      subject { described_class.order_severity_asc }

      it { is_expected.to eq([issue_no_severity, issue_low_severity, issue_high_severity]) }
    end

    context 'sorting descending' do
      subject { described_class.order_severity_desc }

      it { is_expected.to eq([issue_high_severity, issue_low_severity, issue_no_severity]) }
    end
  end

  describe '.order_title' do
    let_it_be(:issue1) { create(:issue, title: 'foo') }
    let_it_be(:issue2) { create(:issue, title: 'bar') }
    let_it_be(:issue3) { create(:issue, title: 'baz') }
    let_it_be(:issue4) { create(:issue, title: 'Baz 2') }

    context 'sorting ascending' do
      subject { described_class.order_title_asc }

      it { is_expected.to eq([issue2, issue3, issue4, issue1]) }
    end

    context 'sorting descending' do
      subject { described_class.order_title_desc }

      it { is_expected.to eq([issue1, issue4, issue3, issue2]) }
    end
  end

  describe '#order_by_relative_position' do
    let(:project) { reusable_project }
    let!(:issue1) { create(:issue, project: project) }
    let!(:issue2) { create(:issue, project: project) }
    let!(:issue3) { create(:issue, project: project, relative_position: -200) }
    let!(:issue4) { create(:issue, project: project, relative_position: -100) }

    it 'returns ordered list' do
      expect(project.issues.order_by_relative_position)
        .to match [issue3, issue4, issue1, issue2]
    end
  end

  context 'order by escalation status' do
    let_it_be(:triggered_incident) { create(:incident_management_issuable_escalation_status, :triggered).issue }
    let_it_be(:resolved_incident) { create(:incident_management_issuable_escalation_status, :resolved).issue }
    let_it_be(:issue_no_status) { create(:issue) }

    describe '.order_escalation_status_asc' do
      subject { described_class.order_escalation_status_asc }

      it { is_expected.to eq([triggered_incident, resolved_incident, issue_no_status]) }
    end

    describe '.order_escalation_status_desc' do
      subject { described_class.order_escalation_status_desc }

      it { is_expected.to eq([resolved_incident, triggered_incident, issue_no_status]) }
    end
  end

  describe '#sort' do
    let(:project) { reusable_project }

    context "by relative_position" do
      let!(:issue)  { create(:issue, project: project, relative_position: nil) }
      let!(:issue2) { create(:issue, project: project, relative_position: 2) }
      let!(:issue3) { create(:issue, project: project, relative_position: 1) }

      it "sorts asc with nulls at the end" do
        issues = project.issues.sort_by_attribute('relative_position')
        expect(issues).to eq([issue3, issue2, issue])
      end
    end
  end

  describe '#card_attributes' do
    it 'includes the author name' do
      allow(subject).to receive(:author).and_return(double(name: 'Robert'))
      allow(subject).to receive(:assignees).and_return([])

      expect(subject.card_attributes)
        .to eq({ 'Author' => 'Robert', 'Assignee' => '' })
    end

    it 'includes the assignee name' do
      allow(subject).to receive(:author).and_return(double(name: 'Robert'))
      allow(subject).to receive(:assignees).and_return([double(name: 'Douwe')])

      expect(subject.card_attributes)
        .to eq({ 'Author' => 'Robert', 'Assignee' => 'Douwe' })
    end
  end

  describe '#close' do
    subject(:issue) { create(:issue, project: reusable_project, state: 'opened') }

    it 'sets closed_at to Time.current when an issue is closed' do
      expect { issue.close }.to change { issue.closed_at }.from(nil)
    end

    it 'changes the state to closed' do
      open_state = described_class.available_states[:opened]
      closed_state = described_class.available_states[:closed]

      expect { issue.close }.to change { issue.state_id }.from(open_state).to(closed_state)
    end

    context 'when an argument is provided' do
      context 'and the argument is a User' do
        it 'changes closed_by to the given user' do
          expect { issue.close(user) }.to change { issue.closed_by }.from(nil).to(user)
        end
      end

      context 'and the argument is a not a User' do
        it 'does not change closed_by' do
          expect { issue.close("test") }.not_to change { issue.closed_by }
        end
      end
    end

    context 'when an argument is not provided' do
      it 'does not change closed_by' do
        expect { issue.close }.not_to change { issue.closed_by }
      end
    end
  end

  describe '#reopen' do
    let_it_be_with_reload(:issue) { create(:issue, project: reusable_project, state: 'closed', closed_at: Time.current, closed_by: user) }

    it 'sets closed_at to nil when an issue is reopened' do
      expect { issue.reopen }.to change { issue.closed_at }.to(nil)
    end

    it 'sets closed_by to nil when an issue is reopened' do
      expect { issue.reopen }.to change { issue.closed_by }.from(user).to(nil)
    end

    it 'clears moved_to_id for moved issues' do
      moved_issue = create(:issue)

      issue.update!(moved_to_id: moved_issue.id)

      expect { issue.reopen }.to change { issue.moved_to_id }.from(moved_issue.id).to(nil)
    end

    it 'clears duplicated_to_id for duplicated issues' do
      duplicate_issue = create(:issue)

      issue.update!(duplicated_to_id: duplicate_issue.id)

      expect { issue.reopen }.to change { issue.duplicated_to_id }.from(duplicate_issue.id).to(nil)
    end

    it 'changes the state to opened' do
      expect { issue.reopen }.to change { issue.state_id }.from(described_class.available_states[:closed]).to(described_class.available_states[:opened])
    end
  end

  describe '#to_reference' do
    let_it_be(:namespace) { create(:namespace) }
    let_it_be(:project)   { create(:project, namespace: namespace) }
    let_it_be(:issue)     { create(:issue, project: project) }

    context 'when nil argument' do
      it 'returns issue id' do
        expect(issue.to_reference).to eq "##{issue.iid}"
      end

      it 'returns complete path to the issue with full: true' do
        expect(issue.to_reference(full: true)).to eq "#{project.full_path}##{issue.iid}"
      end
    end

    context 'when argument is a project' do
      context 'when same project' do
        it 'returns issue id' do
          expect(issue.to_reference(project)).to eq("##{issue.iid}")
        end

        it 'returns full reference with full: true' do
          expect(issue.to_reference(project, full: true)).to eq "#{project.full_path}##{issue.iid}"
        end
      end

      context 'when cross-project in same namespace' do
        let(:another_project) do
          create(:project, namespace: project.namespace)
        end

        it 'returns a cross-project reference' do
          expect(issue.to_reference(another_project)).to eq "#{project.path}##{issue.iid}"
        end
      end

      context 'when cross-project in different namespace' do
        let(:another_namespace) { build(:namespace, id: non_existing_record_id, path: 'another-namespace') }
        let(:another_namespace_project) { build(:project, namespace: another_namespace) }

        it 'returns complete path to the issue' do
          expect(issue.to_reference(another_namespace_project)).to eq "#{project.full_path}##{issue.iid}"
        end
      end
    end

    context 'when argument is a namespace' do
      context 'when same as issue' do
        it 'returns path to the issue with the project name' do
          expect(issue.to_reference(namespace)).to eq "#{project.path}##{issue.iid}"
        end

        it 'returns full reference with full: true' do
          expect(issue.to_reference(namespace, full: true)).to eq "#{project.full_path}##{issue.iid}"
        end
      end

      context 'when different to issue namespace' do
        let(:group) { build(:group, name: 'Group', path: 'sample-group') }

        it 'returns full path to the issue with full: true' do
          expect(issue.to_reference(group)).to eq "#{project.full_path}##{issue.iid}"
        end
      end
    end
  end

  describe '#to_reference with table syntax' do
    using RSpec::Parameterized::TableSyntax

    let_it_be(:user) { create(:user) }
    let_it_be(:user_namespace) { user.namespace }

    let_it_be(:parent) { create(:group) }
    let_it_be(:group) { create(:group, parent: parent) }
    let_it_be(:another_group) { create(:group) }

    let_it_be(:project) { create(:project, namespace: group) }
    let_it_be(:project_namespace) { project.project_namespace }
    let_it_be(:same_namespace_project) { create(:project, namespace: group) }
    let_it_be(:same_namespace_project_namespace) { same_namespace_project.project_namespace }

    let_it_be(:another_namespace_project) { create(:project) }
    let_it_be(:another_namespace_project_namespace) { another_namespace_project.project_namespace }

    let_it_be(:project_issue) { build(:issue, project: project, iid: 123) }
    let_it_be(:project_issue_full_reference) { "#{project.full_path}##{project_issue.iid}" }

    let_it_be(:group_issue) { build(:issue, namespace: group, iid: 123) }
    let_it_be(:group_issue_full_reference) { "#{group.full_path}##{group_issue.iid}" }

    # this one is just theoretically possible, not smth to be supported for real
    let_it_be(:user_issue) { build(:issue, namespace: user_namespace, iid: 123) }
    let_it_be(:user_issue_full_reference) { "#{user_namespace.full_path}##{user_issue.iid}" }

    # namespace would be group, project namespace or user namespace
    where(:issue, :full, :from, :result) do
      ref(:project_issue) | false | nil                                       | lazy { "##{issue.iid}" }
      ref(:project_issue) | true  | nil                                       | ref(:project_issue_full_reference)
      ref(:project_issue) | false | ref(:group)                               | lazy { "#{project.path}##{issue.iid}" }
      ref(:project_issue) | true  | ref(:group)                               | ref(:project_issue_full_reference)
      ref(:project_issue) | false | ref(:parent)                              | ref(:project_issue_full_reference)
      ref(:project_issue) | true  | ref(:parent)                              | ref(:project_issue_full_reference)
      ref(:project_issue) | false | ref(:project)                             | lazy { "##{issue.iid}" }
      ref(:project_issue) | true  | ref(:project)                             | ref(:project_issue_full_reference)
      ref(:project_issue) | false | ref(:project_namespace)                   | lazy { "##{issue.iid}" }
      ref(:project_issue) | true  | ref(:project_namespace)                   | ref(:project_issue_full_reference)
      ref(:project_issue) | false | ref(:same_namespace_project)              | lazy { "#{project.path}##{issue.iid}" }
      ref(:project_issue) | true  | ref(:same_namespace_project)              | ref(:project_issue_full_reference)
      ref(:project_issue) | false | ref(:same_namespace_project_namespace)    | lazy { "#{project.path}##{issue.iid}" }
      ref(:project_issue) | true  | ref(:same_namespace_project_namespace)    | ref(:project_issue_full_reference)
      ref(:project_issue) | false | ref(:another_group)                       | ref(:project_issue_full_reference)
      ref(:project_issue) | true  | ref(:another_group)                       | ref(:project_issue_full_reference)
      ref(:project_issue) | false | ref(:another_namespace_project)           | ref(:project_issue_full_reference)
      ref(:project_issue) | true  | ref(:another_namespace_project)           | ref(:project_issue_full_reference)
      ref(:project_issue) | false | ref(:another_namespace_project_namespace) | ref(:project_issue_full_reference)
      ref(:project_issue) | true  | ref(:another_namespace_project_namespace) | ref(:project_issue_full_reference)
      ref(:project_issue) | false | ref(:user_namespace)                      | ref(:project_issue_full_reference)
      ref(:project_issue) | true  | ref(:user_namespace)                      | ref(:project_issue_full_reference)

      ref(:group_issue) | false | nil                                         | lazy { "##{issue.iid}" }
      ref(:group_issue) | true  | nil                                         | ref(:group_issue_full_reference)
      ref(:group_issue) | false | ref(:user_namespace)                        | ref(:group_issue_full_reference)
      ref(:group_issue) | true  | ref(:user_namespace)                        | ref(:group_issue_full_reference)
      ref(:group_issue) | false | ref(:group)                                 | lazy { "##{issue.iid}" }
      ref(:group_issue) | true  | ref(:group)                                 | ref(:group_issue_full_reference)
      ref(:group_issue) | false | ref(:parent)                                | lazy { "#{group.path}##{issue.iid}" }
      ref(:group_issue) | true  | ref(:parent)                                | ref(:group_issue_full_reference)
      ref(:group_issue) | false | ref(:project)                               | lazy { "#{group.path}##{issue.iid}" }
      ref(:group_issue) | true  | ref(:project)                               | ref(:group_issue_full_reference)
      ref(:group_issue) | false | ref(:project_namespace)                     | lazy { "#{group.path}##{issue.iid}" }
      ref(:group_issue) | true  | ref(:project_namespace)                     | ref(:group_issue_full_reference)
      ref(:group_issue) | false | ref(:another_group)                         | ref(:group_issue_full_reference)
      ref(:group_issue) | true  | ref(:another_group)                         | ref(:group_issue_full_reference)
      ref(:group_issue) | false | ref(:another_namespace_project)             | ref(:group_issue_full_reference)
      ref(:group_issue) | true  | ref(:another_namespace_project)             | ref(:group_issue_full_reference)
      ref(:group_issue) | false | ref(:another_namespace_project_namespace)   | ref(:group_issue_full_reference)
      ref(:group_issue) | true  | ref(:another_namespace_project_namespace)   | ref(:group_issue_full_reference)

      ref(:user_issue) | false | nil                                          | lazy { "##{issue.iid}" }
      ref(:user_issue) | true  | nil                                          | ref(:user_issue_full_reference)
      ref(:user_issue) | false | ref(:user_namespace)                         | lazy { "##{issue.iid}" }
      ref(:user_issue) | true  | ref(:user_namespace)                         | ref(:user_issue_full_reference)
      ref(:user_issue) | false | ref(:group)                                  | ref(:user_issue_full_reference)
      ref(:user_issue) | true  | ref(:group)                                  | ref(:user_issue_full_reference)
      ref(:user_issue) | false | ref(:parent)                                 | ref(:user_issue_full_reference)
      ref(:user_issue) | true  | ref(:parent)                                 | ref(:user_issue_full_reference)
      ref(:user_issue) | false | ref(:project)                                | ref(:user_issue_full_reference)
      ref(:user_issue) | true  | ref(:project)                                | ref(:user_issue_full_reference)
      ref(:user_issue) | false | ref(:project_namespace)                      | ref(:user_issue_full_reference)
      ref(:user_issue) | true  | ref(:project_namespace)                      | ref(:user_issue_full_reference)
      ref(:user_issue) | false | ref(:another_group)                          | ref(:user_issue_full_reference)
      ref(:user_issue) | true  | ref(:another_group)                          | ref(:user_issue_full_reference)
      ref(:user_issue) | false | ref(:another_namespace_project)              | ref(:user_issue_full_reference)
      ref(:user_issue) | true  | ref(:another_namespace_project)              | ref(:user_issue_full_reference)
      ref(:user_issue) | false | ref(:another_namespace_project_namespace)    | ref(:user_issue_full_reference)
      ref(:user_issue) | true  | ref(:another_namespace_project_namespace)    | ref(:user_issue_full_reference)
    end

    with_them do
      it 'returns correct reference' do
        expect(issue.to_reference(from, full: full)).to eq(result)
      end
    end
  end

  describe '#assignee_or_author?' do
    let(:issue) { create(:issue, project: reusable_project) }

    it 'returns true for a user that is assigned to an issue' do
      issue.assignees << user

      expect(issue.assignee_or_author?(user)).to be_truthy
    end

    it 'returns true for a user that is the author of an issue' do
      issue.update!(author: user)

      expect(issue.assignee_or_author?(user)).to be_truthy
    end

    it 'returns false for a user that is not the assignee or author' do
      expect(issue.assignee_or_author?(user)).to be_falsey
    end
  end

  describe '#related_issues to relate incidents and issues' do
    let_it_be(:authorized_project) { create(:project) }
    let_it_be(:authorized_project2) { create(:project) }
    let_it_be(:unauthorized_project) { create(:project) }

    let_it_be(:authorized_issue_a) { create(:issue, project: authorized_project) }
    let_it_be(:authorized_issue_b) { create(:issue, project: authorized_project) }
    let_it_be(:authorized_issue_c) { create(:issue, project: authorized_project2) }
    let_it_be(:authorized_incident_a) { create(:incident, project: authorized_project) }

    let_it_be(:unauthorized_issue) { create(:issue, project: unauthorized_project) }

    let_it_be(:issue_link_a) { create(:issue_link, source: authorized_issue_a, target: authorized_issue_b) }
    let_it_be(:issue_link_b) { create(:issue_link, source: authorized_issue_a, target: unauthorized_issue) }
    let_it_be(:issue_link_c) { create(:issue_link, source: authorized_issue_a, target: authorized_issue_c) }
    let_it_be(:issue_incident_link_a) { create(:issue_link, source: authorized_issue_a, target: authorized_incident_a) }

    before_all do
      authorized_project.add_developer(user)
      authorized_project2.add_developer(user)
    end

    it 'returns only authorized related issues for given user' do
      expect(authorized_issue_a.related_issues(user))
        .to contain_exactly(authorized_issue_b, authorized_issue_c, authorized_incident_a)
    end

    it 'returns issues with valid issue_link_type' do
      link_types = authorized_issue_a.related_issues(user).map(&:issue_link_type)

      expect(link_types).not_to be_empty
      expect(link_types).not_to include(nil)
    end

    it 'returns issues including the link creation time' do
      dates = authorized_issue_a.related_issues(user).map(&:issue_link_created_at)

      expect(dates).not_to be_empty
      expect(dates).not_to include(nil)
    end

    it 'returns issues including the link update time' do
      dates = authorized_issue_a.related_issues(user).map(&:issue_link_updated_at)

      expect(dates).not_to be_empty
      expect(dates).not_to include(nil)
    end

    describe 'when a user cannot read cross project' do
      it 'only returns issues within the same project' do
        expect(Ability).to receive(:allowed?).with(user, :read_all_resources, :global).at_least(:once).and_call_original
        expect(Ability).to receive(:allowed?).with(user, :read_cross_project).and_return(false)

        expect(authorized_issue_a.related_issues(user))
          .to contain_exactly(authorized_issue_b, authorized_incident_a)
      end
    end
  end

  describe '#can_move?' do
    let(:issue) { create(:issue) }

    subject { issue.can_move?(user) }

    context 'user is not a member of project issue belongs to' do
      it { is_expected.to eq false }
    end

    context 'user is reporter in project issue belongs to' do
      let(:issue) { create(:issue, project: reusable_project) }

      before_all do
        reusable_project.add_reporter(user)
      end

      it { is_expected.to eq true }

      context 'issue not persisted' do
        let(:issue) { build(:issue, project: reusable_project) }

        it { is_expected.to eq false }
      end

      context 'checking destination project also' do
        subject { issue.can_move?(user, to_project) }

        let_it_be(:to_project) { create(:project) }

        context 'destination project allowed' do
          before do
            to_project.add_reporter(user)
          end

          it { is_expected.to eq true }
        end

        context 'destination project not allowed' do
          before do
            to_project.add_guest(user)
          end

          it { is_expected.to eq false }
        end
      end
    end
  end

  describe '#moved?' do
    context 'when issue has not been moved' do
      subject { build_stubbed(:issue) }

      it { is_expected.not_to be_moved }
    end

    context 'when issue has already been moved' do
      subject { build_stubbed(:issue, moved_to: build_stubbed(:issue)) }

      it { is_expected.to be_moved }
    end
  end

  describe '#duplicated?' do
    let(:issue) { create(:issue, project: reusable_project) }

    subject { issue.duplicated? }

    context 'issue not duplicated' do
      it { is_expected.to eq false }
    end

    context 'issue already duplicated' do
      let(:duplicated_to_issue) { create(:issue, project: reusable_project) }
      let(:issue) { create(:issue, duplicated_to: duplicated_to_issue) }

      it { is_expected.to eq true }
    end
  end

  describe '#from_service_desk?' do
    subject { issue.from_service_desk? }

    context 'when issue author is support bot' do
      let(:issue) { create(:issue, project: reusable_project, author: ::User.support_bot) }

      it { is_expected.to be_truthy }
    end

    context 'when issue author is not support bot' do
      let(:issue) { create(:issue, project: reusable_project) }

      it { is_expected.to be_falsey }
    end
  end

  describe '#suggested_branch_name' do
    let(:repository) { double }

    subject { build(:issue) }

    before do
      allow(subject.project).to receive(:repository).and_return(repository)
    end

    describe '#to_branch_name does not exists' do
      before do
        allow(repository).to receive(:branch_exists?).and_return(false)
      end

      it 'returns #to_branch_name' do
        expect(subject.suggested_branch_name).to eq(subject.to_branch_name)
      end
    end

    describe '#to_branch_name exists not ending with -index' do
      before do
        allow(repository).to receive(:branch_exists?).and_return(true)
        allow(repository).to receive(:branch_exists?).with(/#{subject.to_branch_name}-\d/).and_return(false)
      end

      it 'returns #to_branch_name ending with -2' do
        expect(subject.suggested_branch_name).to eq("#{subject.to_branch_name}-2")
      end
    end

    describe '#to_branch_name exists ending with -index' do
      it 'returns #to_branch_name ending with max index + 1' do
        allow(repository).to receive(:branch_exists?).and_return(true)
        allow(repository).to receive(:branch_exists?).with("#{subject.to_branch_name}-3").and_return(false)

        expect(subject.suggested_branch_name).to eq("#{subject.to_branch_name}-3")
      end

      context 'when branch name still exists after 5 attempts' do
        it 'returns #to_branch_name ending with random characters' do
          allow(repository).to receive(:branch_exists?).with(subject.to_branch_name).and_return(true)
          allow(repository).to receive(:branch_exists?).with(/#{subject.to_branch_name}-\d/).and_return(true)
          allow(repository).to receive(:branch_exists?).with(/#{subject.to_branch_name}-\h{8}/).and_return(false)

          expect(subject.suggested_branch_name).to match(/#{subject.to_branch_name}-\h{8}/)
        end
      end
    end
  end

  it_behaves_like 'an editable mentionable' do
    subject { create(:issue, project: create(:project, :repository)) }

    let(:backref_text) { "issue #{subject.to_reference}" }
    let(:set_mentionable_text) { ->(txt) { subject.description = txt } }
  end

  it_behaves_like 'a Taskable' do
    let(:subject) { create :issue }
  end

  describe '.to_branch_name' do
    it 'parameterizes arguments and joins with dashes' do
      expect(described_class.to_branch_name(123, 'foo bar!@#$%f!o@o#b$a%r^')).to eq('123-foo-bar-f-o-o-b-a-r')
    end

    it 'preserves the case in the first argument' do
      expect(described_class.to_branch_name('ACME-!@#$-123', 'FoO BaR')).to eq('ACME-123-foo-bar')
    end

    it 'truncates branch name to at most 100 characters' do
      expect(described_class.to_branch_name('a' * 101, 'a')).to eq('a' * 100)
    end

    it 'truncates dangling parts of the branch name' do
      branch_name = described_class.to_branch_name(
        999,
        'Lorem ipsum dolor sit amet consectetur adipiscing elit Mauris sit amet ipsum id lacus custom fringilla convallis'
      )

      # 100 characters would've got us "999-lorem...lacus-custom-fri".
      expect(branch_name).to eq('999-lorem-ipsum-dolor-sit-amet-consectetur-adipiscing-elit-mauris-sit-amet-ipsum-id-lacus-custom')
    end

    it 'takes issue branch template into account' do
      project = create(:project)
      project.project_setting.update!(issue_branch_template: 'feature-%{id}-%{title}')

      expect(described_class.to_branch_name(123, 'issue title', project: project)).to eq('feature-123-issue-title')
    end
  end

  describe '#to_branch_name' do
    let_it_be(:issue) { create(:issue, project: reusable_project, iid: 123, title: 'Testing Issue') }

    it 'returns a branch name with the issue title if not confidential' do
      expect(issue.to_branch_name).to eq('123-testing-issue')
    end

    it 'returns a generic branch name if confidential' do
      issue.confidential = true
      expect(issue.to_branch_name).to eq('123-confidential-issue')
    end
  end

  describe '#can_be_worked_on?' do
    let(:project) { build(:project) }

    subject { build(:issue, :opened, project: project) }

    context 'is closed' do
      subject { build(:issue, :closed) }

      it { is_expected.not_to be_can_be_worked_on }
    end

    context 'project is forked' do
      before do
        allow(project).to receive(:forked?).and_return(true)
      end

      it { is_expected.not_to be_can_be_worked_on }
    end

    it { is_expected.to be_can_be_worked_on }
  end

  describe '#participants' do
    it_behaves_like 'issuable participants' do
      let_it_be(:issuable_parent) { create(:project, :public) }
      let_it_be_with_refind(:issuable) { create(:issue, project: issuable_parent) }

      let(:params) { { noteable: issuable, project: issuable_parent } }
    end

    context 'using a private project' do
      it 'does not include mentioned users that do not have access to the project' do
        project = create(:project)
        issue = create(:issue, project: project)
        user = create(:user)

        create(:note_on_issue,
               noteable: issue,
               project: project,
               note: user.to_reference)

        expect(issue.participants).not_to include(user)
      end
    end
  end

  describe 'cached counts' do
    it 'updates when assignees change' do
      user1 = create(:user)
      user2 = create(:user)
      issue = create(:issue, assignees: [user1], project: reusable_project)
      reusable_project.add_developer(user1)
      reusable_project.add_developer(user2)

      expect(user1.assigned_open_issues_count).to eq(1)
      expect(user2.assigned_open_issues_count).to eq(0)

      issue.assignees = [user2]
      issue.save!

      expect(user1.assigned_open_issues_count).to eq(0)
      expect(user2.assigned_open_issues_count).to eq(1)
    end
  end

  describe '#visible_to_user?' do
    let(:project) { reusable_project }
    let(:issue)   { build(:issue, project: project) }

    subject { issue.visible_to_user?(user) }

    context 'with a project' do
      it 'returns false when feature is disabled' do
        project.add_developer(user)
        project.project_feature.update_attribute(:issues_access_level, ProjectFeature::DISABLED)

        is_expected.to eq(false)
      end
    end

    context 'without a user' do
      let(:user) { nil }

      context 'with issue available as public' do
        before do
          project.project_feature.update_attribute(:issues_access_level, ProjectFeature::PUBLIC)
        end

        it 'returns true when the issue is publicly visible' do
          expect(issue).to receive(:publicly_visible?).and_return(true)

          is_expected.to eq(true)
        end

        it 'returns false when the issue is not publicly visible' do
          expect(issue).to receive(:publicly_visible?).and_return(false)

          is_expected.to eq(false)
        end
      end

      context 'with issues available only to team members in a public project' do
        let(:public_project) { create(:project, :public) }
        let(:issue) { build(:issue, project: public_project) }

        before do
          public_project.project_feature.update_attribute(:issues_access_level, ProjectFeature::PRIVATE)
        end

        it 'returns false' do
          is_expected.to eq(false)
        end
      end
    end

    context 'with a user' do
      shared_examples 'issue readable by user' do
        it { is_expected.to eq(true) }
      end

      shared_examples 'issue not readable by user' do
        it { is_expected.to eq(false) }
      end

      shared_examples 'confidential issue readable by user' do
        specify do
          issue.confidential = true

          is_expected.to eq(true)
        end
      end

      shared_examples 'confidential issue not readable by user' do
        specify do
          issue.confidential = true

          is_expected.to eq(false)
        end
      end

      shared_examples 'hidden issue readable by user' do
        before do
          issue.author.ban!
        end

        specify do
          is_expected.to eq(true)
        end

        after do
          issue.author.activate!
        end
      end

      shared_examples 'hidden issue not readable by user' do
        before do
          issue.author.ban!
        end

        specify do
          is_expected.to eq(false)
        end

        after do
          issue.author.activate!
        end
      end

      context 'with an admin user' do
        let(:user) { build(:admin) }

        context 'when admin mode is enabled', :enable_admin_mode do
          it_behaves_like 'issue readable by user'
          it_behaves_like 'confidential issue readable by user'
          it_behaves_like 'hidden issue readable by user'
        end

        context 'when admin mode is disabled' do
          it_behaves_like 'issue not readable by user'
          it_behaves_like 'confidential issue not readable by user'
          it_behaves_like 'hidden issue not readable by user'
        end
      end

      # TODO update when we have multiple owners of a project
      # https://gitlab.com/gitlab-org/gitlab/-/issues/350605
      context 'with an owner' do
        before do
          project.add_maintainer(user)
        end

        it_behaves_like 'issue readable by user'
        it_behaves_like 'confidential issue readable by user'
        it_behaves_like 'hidden issue not readable by user'
      end

      context 'with a reporter user' do
        before do
          project.add_reporter(user)
        end

        it_behaves_like 'issue readable by user'
        it_behaves_like 'confidential issue readable by user'
        it_behaves_like 'hidden issue not readable by user'
      end

      context 'with a guest user' do
        before do
          project.add_guest(user)
        end

        it_behaves_like 'issue readable by user'
        it_behaves_like 'confidential issue not readable by user'
        it_behaves_like 'hidden issue not readable by user'

        context 'when user is an assignee' do
          before do
            issue.update!(assignees: [user])
          end

          it_behaves_like 'issue readable by user'
          it_behaves_like 'confidential issue readable by user'
          it_behaves_like 'hidden issue not readable by user'
        end

        context 'when user is the author' do
          before do
            issue.update!(author: user)
          end

          it_behaves_like 'issue readable by user'
          it_behaves_like 'confidential issue readable by user'
          it_behaves_like 'hidden issue not readable by user'
        end
      end

      context 'with a user that is not a member' do
        context 'using a public project' do
          let(:project) { build(:project, :public) }

          it_behaves_like 'issue readable by user'
          it_behaves_like 'confidential issue not readable by user'
          it_behaves_like 'hidden issue not readable by user'
        end

        context 'using an internal project' do
          let(:project) { build(:project, :internal) }

          context 'using an internal user' do
            before do
              allow(user).to receive(:external?).and_return(false)
            end

            it_behaves_like 'issue readable by user'
            it_behaves_like 'confidential issue not readable by user'
            it_behaves_like 'hidden issue not readable by user'
          end

          context 'using an external user' do
            before do
              allow(user).to receive(:external?).and_return(true)
            end

            it_behaves_like 'issue not readable by user'
            it_behaves_like 'confidential issue not readable by user'
            it_behaves_like 'hidden issue not readable by user'
          end
        end

        context 'using an external user' do
          before do
            allow(user).to receive(:external?).and_return(true)
          end

          it_behaves_like 'issue not readable by user'
          it_behaves_like 'confidential issue not readable by user'
          it_behaves_like 'hidden issue not readable by user'
        end
      end

      context 'with an external authentication service' do
        before do
          enable_external_authorization_service_check
        end

        it 'is `false` when an external authorization service is enabled' do
          issue = build(:issue, project: build(:project, :public))

          expect(issue).not_to be_visible_to_user
        end

        it 'checks the external service to determine if an issue is readable by a user' do
          project = build(:project, :public,
                          external_authorization_classification_label: 'a-label')
          issue = build(:issue, project: project)
          user = build(:user)

          allow(::Gitlab::ExternalAuthorization).to receive(:access_allowed?).with(user, 'a-label', project.full_path).and_call_original
          expect(::Gitlab::ExternalAuthorization).to receive(:access_allowed?).with(user, 'a-label') { false }
          expect(issue.visible_to_user?(user)).to be_falsy
        end

        it 'does not check the external service if a user does not have access to the project' do
          project = build(:project, :private,
                          external_authorization_classification_label: 'a-label')
          issue = build(:issue, project: project)
          user = build(:user)

          expect(::Gitlab::ExternalAuthorization).not_to receive(:access_allowed?)
          expect(issue.visible_to_user?(user)).to be_falsy
        end

        context 'with an admin' do
          context 'when admin mode is enabled', :enable_admin_mode do
            it 'does not check the external webservice' do
              issue = build(:issue)
              user = build(:admin)

              expect(::Gitlab::ExternalAuthorization).not_to receive(:access_allowed?)

              issue.visible_to_user?(user)
            end
          end

          context 'when admin mode is disabled' do
            it 'checks the external service to determine if an issue is readable by the admin' do
              project = build(:project, :public,
                              external_authorization_classification_label: 'a-label')
              issue = build(:issue, project: project)
              user = build(:admin)

              allow(::Gitlab::ExternalAuthorization).to receive(:access_allowed?).with(user, 'a-label', project.full_path).and_call_original
              expect(::Gitlab::ExternalAuthorization).to receive(:access_allowed?).with(user, 'a-label') { false }
              expect(issue.visible_to_user?(user)).to be_falsy
            end
          end
        end
      end

      context 'when issue is moved to a private project' do
        let(:private_project) { build(:project, :private) }

        before do
          issue.update!(project: private_project) # move issue to private project
        end

        shared_examples 'issue visible if user has guest access' do
          context 'when user is not a member' do
            it_behaves_like 'issue not readable by user'
            it_behaves_like 'confidential issue not readable by user'
          end

          context 'when user is a guest' do
            before do
              private_project.add_guest(user)
            end

            it_behaves_like 'issue readable by user'
            it_behaves_like 'confidential issue readable by user'
          end
        end

        context 'when user is the author of the original issue' do
          before do
            issue.update!(author: user)
          end

          it_behaves_like 'issue visible if user has guest access'
        end

        context 'when user is an assignee in the original issue' do
          before do
            issue.update!(assignees: [user])
          end

          it_behaves_like 'issue visible if user has guest access'
        end

        context 'when user is not the author or an assignee in original issue' do
          context 'when user is a guest' do
            before do
              private_project.add_guest(user)
            end

            it_behaves_like 'issue readable by user'
            it_behaves_like 'confidential issue not readable by user'
          end

          context 'when user is a reporter' do
            before do
              private_project.add_reporter(user)
            end

            it_behaves_like 'issue readable by user'
            it_behaves_like 'confidential issue readable by user'
          end
        end
      end
    end
  end

  describe '#publicly_visible?' do
    context 'using a public project' do
      let(:project) { create(:project, :public) }

      it 'returns true for a regular issue' do
        issue = build(:issue, project: project)

        expect(issue).to be_truthy
      end

      it 'returns false for a confidential issue' do
        issue = build(:issue, :confidential, project: project)

        expect(issue).not_to be_falsy
      end
    end

    context 'using an internal project' do
      let(:project) { create(:project, :internal) }

      it 'returns false for a regular issue' do
        issue = build(:issue, project: project)

        expect(issue).not_to be_falsy
      end

      it 'returns false for a confidential issue' do
        issue = build(:issue, :confidential, project: project)

        expect(issue).not_to be_falsy
      end
    end

    context 'using a private project' do
      let(:project) { create(:project, :private) }

      it 'returns false for a regular issue' do
        issue = build(:issue, project: project)

        expect(issue).not_to be_falsy
      end

      it 'returns false for a confidential issue' do
        issue = build(:issue, :confidential, project: project)

        expect(issue).not_to be_falsy
      end
    end
  end

  describe '#check_for_spam?' do
    let_it_be(:support_bot) { ::User.support_bot }

    where(:support_bot?, :visibility_level, :confidential, :new_attributes, :check_for_spam?) do
      ### non-support-bot cases
      # spammable attributes changing
      false | Gitlab::VisibilityLevel::PUBLIC   | false | { description: 'new' } | true
      false | Gitlab::VisibilityLevel::PUBLIC   | false | { title: 'new' } | true
      # confidential to non-confidential
      false | Gitlab::VisibilityLevel::PUBLIC   | true  | { confidential: false } | true
      # non-confidential to confidential
      false | Gitlab::VisibilityLevel::PUBLIC   | false | { confidential: true } | false
      # spammable attributes changing on confidential
      false | Gitlab::VisibilityLevel::PUBLIC   | true  | { description: 'new' } | false
      # spammable attributes changing while changing to confidential
      false | Gitlab::VisibilityLevel::PUBLIC   | false | { title: 'new', confidential: true } | false
      # spammable attribute not changing
      false | Gitlab::VisibilityLevel::PUBLIC   | false | { description: 'original description' } | false
      # non-spammable attribute changing
      false | Gitlab::VisibilityLevel::PUBLIC   | false | { weight: 3 } | false
      # spammable attributes changing on non-public
      false | Gitlab::VisibilityLevel::INTERNAL | false | { description: 'new' } | false
      false | Gitlab::VisibilityLevel::PRIVATE  | false | { description: 'new' } | false

      ### support-bot cases
      # confidential to non-confidential
      true | Gitlab::VisibilityLevel::PUBLIC    | true  | { confidential: false } | true
      # non-confidential to confidential
      true | Gitlab::VisibilityLevel::PUBLIC    | false | { confidential: true } | false
      # spammable attributes changing on confidential
      true  | Gitlab::VisibilityLevel::PUBLIC   | true  | { description: 'new' } | true
      # spammable attributes changing while changing to confidential
      true  | Gitlab::VisibilityLevel::PUBLIC   | false | { title: 'new', confidential: true } | true
      # spammable attributes changing on non-public
      true  | Gitlab::VisibilityLevel::INTERNAL | false | { description: 'new' } | true
      true  | Gitlab::VisibilityLevel::PRIVATE  | false | { title: 'new' } | true
      # spammable attribute not changing
      true  | Gitlab::VisibilityLevel::PUBLIC   | false | { description: 'original description' } | false
      # non-spammable attribute changing
      true  | Gitlab::VisibilityLevel::PRIVATE  | true  | { weight: 3 } | false
    end

    with_them do
      it 'checks for spam when necessary' do
        active_user = support_bot? ? support_bot : user
        project = reusable_project
        project.update!(visibility_level: visibility_level)
        issue = create(:issue, project: project, confidential: confidential, description: 'original description', author: support_bot)

        issue.assign_attributes(new_attributes)

        expect(issue.check_for_spam?(user: active_user)).to eq(check_for_spam?)
      end
    end
  end

  describe 'removing an issue' do
    it 'refreshes the number of open issues of the project' do
      project = subject.project

      expect do
        subject.destroy!

        BatchLoader::Executor.clear_current
      end.to change { project.open_issues_count }.from(1).to(0)
    end
  end

  describe '.public_only' do
    it 'only returns public issues' do
      public_issue = create(:issue, project: reusable_project)
      create(:issue, project: reusable_project, confidential: true)

      expect(described_class.public_only).to eq([public_issue])
    end
  end

  describe '.confidential_only' do
    it 'only returns confidential_only issues' do
      create(:issue, project: reusable_project)
      confidential_issue = create(:issue, project: reusable_project, confidential: true)

      expect(described_class.confidential_only).to eq([confidential_issue])
    end
  end

  describe '.without_hidden' do
    let_it_be(:banned_user) { create(:user, :banned) }
    let_it_be(:public_issue) { create(:issue, project: reusable_project) }
    let_it_be(:hidden_issue) { create(:issue, project: reusable_project, author: banned_user) }

    it 'only returns without_hidden issues' do
      expect(described_class.without_hidden).to eq([public_issue])
    end
  end

  describe '.by_project_id_and_iid' do
    let_it_be(:issue_a) { create(:issue, project: reusable_project) }
    let_it_be(:issue_b) { create(:issue, iid: issue_a.iid) }
    let_it_be(:issue_c) { create(:issue, project: issue_a.project) }
    let_it_be(:issue_d) { create(:issue, project: issue_a.project) }

    it_behaves_like 'a where_composite scope', :by_project_id_and_iid do
      let(:all_results) { [issue_a, issue_b, issue_c, issue_d] }
      let(:first_result) { issue_a }

      let(:composite_ids) do
        all_results.map { |issue| { project_id: issue.project_id, iid: issue.iid } }
      end
    end
  end

  describe '.service_desk' do
    it 'returns the service desk issue' do
      service_desk_issue = create(:issue, project: reusable_project, author: ::User.support_bot)
      regular_issue = create(:issue, project: reusable_project)

      expect(described_class.service_desk).to include(service_desk_issue)
      expect(described_class.service_desk).not_to include(regular_issue)
    end
  end

  it_behaves_like 'throttled touch' do
    subject { create(:issue, updated_at: 1.hour.ago) }
  end

  context "relative positioning" do
    let_it_be(:group) { create(:group) }
    let_it_be(:project) { create(:project, group: group) }
    let_it_be(:issue1) { create(:issue, project: project, relative_position: nil) }
    let_it_be(:issue2) { create(:issue, project: project, relative_position: nil) }

    it_behaves_like "a class that supports relative positioning" do
      let_it_be(:project) { reusable_project }
      let(:factory) { :issue }
      let(:default_params) { { project: project } }
    end

    it 'is not blocked for repositioning by default' do
      expect(issue1.blocked_for_repositioning?).to eq(false)
    end

    context 'when block_issue_repositioning flag is enabled for group' do
      before do
        stub_feature_flags(block_issue_repositioning: group)
      end

      it 'is blocked for repositioning' do
        expect(issue1.blocked_for_repositioning?).to eq(true)
      end

      it 'does not move issues with null position' do
        payload = [issue1, issue2]

        expect { described_class.move_nulls_to_end(payload) }.to raise_error(Gitlab::RelativePositioning::IssuePositioningDisabled)
        expect { described_class.move_nulls_to_start(payload) }.to raise_error(Gitlab::RelativePositioning::IssuePositioningDisabled)
      end
    end
  end

  it_behaves_like 'versioned description'

  describe "#previous_updated_at" do
    let_it_be(:updated_at) { Time.zone.local(2012, 01, 06) }
    let_it_be(:issue) { create(:issue, project: reusable_project, updated_at: updated_at) }

    it 'returns updated_at value if updated_at did not change at all' do
      allow(issue).to receive(:previous_changes).and_return({})

      expect(issue.previous_updated_at).to eq(updated_at)
    end

    it 'returns updated_at value if `previous_changes` has nil value for `updated_at`' do
      allow(issue).to receive(:previous_changes).and_return({ 'updated_at' => nil })

      expect(issue.previous_updated_at).to eq(updated_at)
    end

    it 'returns updated_at value if previous updated_at value is not present' do
      allow(issue).to receive(:previous_changes).and_return({ 'updated_at' => [nil, Time.zone.local(2013, 02, 06)] })

      expect(issue.previous_updated_at).to eq(updated_at)
    end

    it 'returns previous updated_at when present' do
      allow(issue).to receive(:previous_changes).and_return({ 'updated_at' => [Time.zone.local(2013, 02, 06), Time.zone.local(2013, 03, 06)] })

      expect(issue.previous_updated_at).to eq(Time.zone.local(2013, 02, 06))
    end
  end

  describe '#design_collection' do
    it 'returns a design collection' do
      issue = build(:issue)
      collection = issue.design_collection

      expect(collection).to be_a(DesignManagement::DesignCollection)
      expect(collection.issue).to eq(issue)
    end
  end

  describe 'current designs' do
    let(:issue) { create(:issue, project: reusable_project) }

    subject { issue.designs.current }

    context 'an issue has no designs' do
      it { is_expected.to be_empty }
    end

    context 'an issue only has current designs' do
      let!(:design_a) { create(:design, :with_file, issue: issue) }
      let!(:design_b) { create(:design, :with_file, issue: issue) }
      let!(:design_c) { create(:design, :with_file, issue: issue) }

      it { is_expected.to include(design_a, design_b, design_c) }
    end

    context 'an issue only has deleted designs' do
      let!(:design_a) { create(:design, :with_file, issue: issue, deleted: true) }
      let!(:design_b) { create(:design, :with_file, issue: issue, deleted: true) }
      let!(:design_c) { create(:design, :with_file, issue: issue, deleted: true) }

      it { is_expected.to be_empty }
    end

    context 'an issue has a mixture of current and deleted designs' do
      let!(:design_a) { create(:design, :with_file, issue: issue) }
      let!(:design_b) { create(:design, :with_file, issue: issue, deleted: true) }
      let!(:design_c) { create(:design, :with_file, issue: issue) }

      it { is_expected.to contain_exactly(design_a, design_c) }
    end
  end

  describe 'banzai_render_context' do
    let(:project) { build(:project_empty_repo) }
    let(:issue) { build :issue, project: project }

    subject(:context) { issue.banzai_render_context(:title) }

    it 'sets the label_url_method in the context' do
      expect(context[:label_url_method]).to eq(:project_issues_url)
    end
  end

  describe 'scheduling rebalancing' do
    before do
      allow_next_instance_of(RelativePositioning::Mover) do |mover|
        allow(mover).to receive(:move) { raise RelativePositioning::NoSpaceLeft }
      end
    end

    shared_examples 'schedules issues rebalancing' do
      let(:issue) { build_stubbed(:issue, relative_position: 100, project: project) }

      it 'schedules rebalancing if there is no space left' do
        lhs = build_stubbed(:issue, relative_position: 99, project: project)
        to_move = build(:issue, project: project)
        expect(Issues::RebalancingWorker).to receive(:perform_async).with(nil, project_id, namespace_id)

        expect { to_move.move_between(lhs, issue) }.to raise_error(RelativePositioning::NoSpaceLeft)
      end
    end

    context 'when project in user namespace' do
      let(:project_namespace) { build_stubbed(:project_namespace) }
      let(:project) { build_stubbed(:project_empty_repo, project_namespace: project_namespace) }
      let(:project_id) { project.id }
      let(:namespace_id) { nil }

      it_behaves_like 'schedules issues rebalancing'
    end

    context 'when project in a group namespace' do
      let(:group) { create(:group) }
      let(:project_namespace) { build_stubbed(:project_namespace) }
      let(:project) { build_stubbed(:project_empty_repo, group: group, project_namespace: project_namespace) }
      let(:project_id) { nil }
      let(:namespace_id) { group.id }

      it_behaves_like 'schedules issues rebalancing'
    end
  end

  describe '#allows_reviewers?' do
    it 'returns false as we do not support reviewers on issues yet' do
      issue = build_stubbed(:issue)

      expect(issue.allows_reviewers?).to be(false)
    end
  end

  describe '#issue_type' do
    let_it_be(:issue) { create(:issue) }

    context 'when the issue_type_uses_work_item_types_table feature flag is enabled' do
      it 'gets the type field from the work_item_types table' do
        expect(issue).to receive_message_chain(:work_item_type, :base_type)

        issue.issue_type
      end

      context 'when the issue is not persisted' do
        it 'uses the default work item type' do
          non_persisted_issue = build(:issue, work_item_type: nil)

          expect(non_persisted_issue.issue_type).to eq(described_class::DEFAULT_ISSUE_TYPE.to_s)
        end
      end
    end

    context 'when the issue_type_uses_work_item_types_table feature flag is disabled' do
      before do
        stub_feature_flags(issue_type_uses_work_item_types_table: false)
      end

      it 'does not get the value from the work_item_types table' do
        expect(issue).not_to receive(:work_item_type)

        issue.issue_type
      end

      context 'when the issue is not persisted' do
        it 'uses the default work item type' do
          non_persisted_issue = build(:issue, work_item_type: nil)

          expect(non_persisted_issue.issue_type).to eq(described_class::DEFAULT_ISSUE_TYPE.to_s)
        end
      end
    end
  end

  describe '#issue_type_supports?' do
    let_it_be(:issue) { create(:issue) }

    it 'raises error when feature is invalid' do
      expect { issue.issue_type_supports?(:unkown_feature) }.to raise_error(ArgumentError)
    end
  end

  describe '#supports_assignee?' do
    Gitlab::DatabaseImporters::WorkItems::BaseTypeImporter::WIDGETS_FOR_TYPE.each_pair do |base_type, widgets|
      specify do
        issue = build(:issue, base_type)
        supports_assignee = widgets.include?(:assignees)

        expect(issue.supports_assignee?).to eq(supports_assignee)
      end
    end
  end

  describe '#supports_time_tracking?' do
    let_it_be(:project) { create(:project) }
    let_it_be_with_refind(:issue) { create(:incident, project: project) }

    where(:issue_type, :supports_time_tracking) do
      :issue | true
      :incident | true
    end

    with_them do
      before do
        issue.update!(issue_type: issue_type, work_item_type: WorkItems::Type.default_by_type(issue_type))
      end

      specify do
        expect(issue.supports_time_tracking?).to eq(supports_time_tracking)
      end
    end
  end

  describe '#supports_move_and_clone?' do
    let_it_be(:project) { create(:project) }
    let_it_be_with_refind(:issue) { create(:incident, project: project) }

    where(:issue_type, :supports_move_and_clone) do
      :issue | true
      :incident | true
    end

    with_them do
      before do
        issue.update!(issue_type: issue_type, work_item_type: WorkItems::Type.default_by_type(issue_type))
      end

      specify do
        expect(issue.supports_move_and_clone?).to eq(supports_move_and_clone)
      end
    end
  end

  describe '#email_participants_emails' do
    let_it_be(:issue) { create(:issue) }

    it 'returns a list of emails' do
      participant1 = issue.issue_email_participants.create!(email: 'a@gitlab.com')
      participant2 = issue.issue_email_participants.create!(email: 'b@gitlab.com')

      expect(issue.email_participants_emails).to contain_exactly(participant1.email, participant2.email)
    end
  end

  describe '#email_participants_downcase' do
    it 'returns a list of emails with all uppercase letters replaced with their lowercase counterparts' do
      participant = create(:issue_email_participant, email: 'SomEoNe@ExamPLe.com')

      expect(participant.issue.email_participants_emails_downcase).to match([participant.email.downcase])
    end
  end

  describe '#escalation_status' do
    it 'returns the incident_management_issuable_escalation_status association' do
      escalation_status = create(:incident_management_issuable_escalation_status)
      issue = escalation_status.issue

      expect(issue.escalation_status).to eq(escalation_status)
    end
  end

  describe '#expire_etag_cache' do
    let_it_be(:issue) { create(:issue) }

    subject(:expire_cache) { issue.expire_etag_cache }

    it 'touches the etag cache store' do
      key = Gitlab::Routing.url_helpers.realtime_changes_project_issue_path(issue.project, issue)

      expect_next_instance_of(Gitlab::EtagCaching::Store) do |cache_store|
        expect(cache_store).to receive(:touch).with(key)
      end

      expire_cache
    end
  end

  describe '#link_reference_pattern' do
    let(:match_data) { described_class.link_reference_pattern.match(link_reference_url) }

    context 'with issue url' do
      let(:link_reference_url) { 'http://localhost/namespace/project/-/issues/1' }

      it 'matches with expected attributes' do
        expect(match_data['namespace']).to eq('namespace')
        expect(match_data['project']).to eq('project')
        expect(match_data['issue']).to eq('1')
      end
    end

    context 'with incident url' do
      let(:link_reference_url) { 'http://localhost/namespace1/project1/-/issues/incident/2' }

      it 'matches with expected attributes' do
        expect(match_data['namespace']).to eq('namespace1')
        expect(match_data['project']).to eq('project1')
        expect(match_data['issue']).to eq('2')
      end
    end
  end

  context 'order by closed_at' do
    let!(:issue_a) { create(:issue, closed_at: 1.day.ago) }
    let!(:issue_b) { create(:issue, closed_at: 5.days.ago) }
    let!(:issue_c_nil) { create(:issue, closed_at: nil) }
    let!(:issue_d) { create(:issue, closed_at: 3.days.ago) }
    let!(:issue_e_nil) { create(:issue, closed_at: nil) }

    describe '.order_closed_at_asc' do
      it 'orders on closed at' do
        expect(described_class.order_closed_at_asc.to_a).to eq([issue_b, issue_d, issue_a, issue_c_nil, issue_e_nil])
      end
    end

    describe '.order_closed_at_desc' do
      it 'orders on closed at' do
        expect(described_class.order_closed_at_desc.to_a).to eq([issue_a, issue_d, issue_b, issue_c_nil, issue_e_nil])
      end
    end
  end

  describe '#full_search' do
    context 'when searching non-english terms' do
      [
        'abc 中文語',
        '中文語cn',
        '中文語',
        'Привет'
      ].each do |term|
        it 'adds extra where clause to match partial index' do
          expect(described_class.full_search(term).to_sql).to include(
            "AND (issues.title NOT SIMILAR TO '[\\u0000-\\u02FF\\u1E00-\\u1EFF\\u2070-\\u218F]*' " \
            "OR issues.description NOT SIMILAR TO '[\\u0000-\\u02FF\\u1E00-\\u1EFF\\u2070-\\u218F]*')"
          )
        end
      end
    end
  end

  describe '#work_item_type_with_default' do
    subject { Issue.new.work_item_type_with_default }

    it { is_expected.to eq(WorkItems::Type.default_by_type(::Issue::DEFAULT_ISSUE_TYPE)) }
  end

  describe '#unsubscribe_email_participant' do
    let_it_be(:email) { 'email@example.com' }

    let_it_be(:issue1) do
      create(:issue, project: reusable_project, external_author: email) do |issue|
        issue.issue_email_participants.create!(email: email)
      end
    end

    let_it_be(:issue2) do
      create(:issue, project: reusable_project, external_author: email) do |issue|
        issue.issue_email_participants.create!(email: email)
      end
    end

    it 'deletes email for issue1' do
      expect { issue1.unsubscribe_email_participant(email) }.to change { issue1.issue_email_participants.count }.by(-1)
    end

    it 'does not delete email for issue2 when issue1 is used' do
      expect { issue1.unsubscribe_email_participant(email) }.not_to change { issue2.issue_email_participants.count }
    end
  end

  describe 'issue_type enum generated methods' do
    using RSpec::Parameterized::TableSyntax

    let_it_be(:issue) { create(:issue, project: reusable_project) }

    where(issue_type: WorkItems::Type.base_types.keys)

    with_them do
      it 'raises an error if called' do
        expect { issue.public_send("#{issue_type}?".to_sym) }.to raise_error(Issue::ForbiddenColumnUsed)
      end
    end
  end
end
