require 'spec_helper'

describe API::Labels do
  let(:user) { create(:user) }
  let(:project) { create(:project, creator_id: user.id, namespace: user.namespace) }
  let!(:label1) { create(:label, title: 'label1', project: project) }
  let!(:priority_label) { create(:label, title: 'bug', project: project, priority: 3) }

  shared_examples 'label update API' do
    it 'returns 200 if name is changed' do
      request_params = {
        new_name: 'New Label'
      }.merge(spec_params)

      put api("/projects/#{project.id}/labels", user),
          params: request_params

      expect(response).to have_gitlab_http_status(200)
      expect(json_response['name']).to eq('New Label')
      expect(json_response['color']).to eq(label1.color)
    end

    it 'returns 200 if colors is changed' do
      request_params = {
        color: '#FFFFFF'
      }.merge(spec_params)

      put api("/projects/#{project.id}/labels", user),
          params: request_params

      expect(response).to have_gitlab_http_status(200)
      expect(json_response['name']).to eq(label1.name)
      expect(json_response['color']).to eq('#FFFFFF')
    end

    it 'returns 200 if a priority is added' do
      request_params = {
        priority: 3
      }.merge(spec_params)

      put api("/projects/#{project.id}/labels", user),
          params: request_params

      expect(response.status).to eq(200)
      expect(json_response['name']).to eq(label1.name)
      expect(json_response['priority']).to eq(3)
    end

    it 'returns 400 if no new parameters given' do
      put api("/projects/#{project.id}/labels", user), params: spec_params

      expect(response).to have_gitlab_http_status(400)
      expect(json_response['error']).to eq('new_name, color, description, priority are missing, '\
                                           'at least one parameter must be provided')
    end

    it 'returns 400 when color code is too short' do
      request_params = {
        color: '#FF'
      }.merge(spec_params)

      put api("/projects/#{project.id}/labels", user),
          params: request_params

      expect(response).to have_gitlab_http_status(400)
      expect(json_response['message']['color']).to eq(['must be a valid color code'])
    end

    it 'returns 400 for too long color code' do
      request_params = {
        color: '#FFAAFFFF'
      }.merge(spec_params)

      put api("/projects/#{project.id}/labels", user),
          params: request_params

      expect(response).to have_gitlab_http_status(400)
      expect(json_response['message']['color']).to eq(['must be a valid color code'])
    end

    it 'returns 400 for invalid priority' do
      request_params = {
        priority: 'foo'
      }.merge(spec_params)

      put api("/projects/#{project.id}/labels", user),
          params: request_params

      expect(response).to have_gitlab_http_status(400)
    end

    it 'returns 200 if name and colors and description are changed' do
      request_params = {
        new_name: 'New Label',
        color: '#FFFFFF',
        description: 'test'
      }.merge(spec_params)

      put api("/projects/#{project.id}/labels", user),
          params: request_params

      expect(response).to have_gitlab_http_status(200)
      expect(json_response['name']).to eq('New Label')
      expect(json_response['color']).to eq('#FFFFFF')
      expect(json_response['description']).to eq('test')
    end

    it 'returns 400 for invalid name' do
      request_params = {
        new_name: ',',
        color: '#FFFFFF'
      }.merge(spec_params)

      put api("/projects/#{project.id}/labels", user),
          params: request_params

      expect(response).to have_gitlab_http_status(400)
      expect(json_response['message']['title']).to eq(['is invalid'])
    end

    it 'returns 200 if description is changed' do
      request_params = {
        description: 'test'
      }.merge(spec_params)

      put api("/projects/#{project.id}/labels", user),
          params: request_params

      expect(response).to have_gitlab_http_status(200)
      expect(json_response['id']).to eq(expected_response_label_id)
      expect(json_response['description']).to eq('test')
    end

    it 'returns 200 if priority is changed' do
      request_params = {
        priority: 10
      }.merge(spec_params)

      put api("/projects/#{project.id}/labels", user),
          params: request_params

      expect(response.status).to eq(200)
      expect(json_response['id']).to eq(expected_response_label_id)
      expect(json_response['priority']).to eq(10)
    end

    it 'returns 200 if a priority is removed' do
      label = find_by_spec_params(spec_params)
      expect(label).not_to be_nil

      label.priorities.create(project: label.project, priority: 1)
      label.save!

      request_params = {
        priority: nil
      }.merge(spec_params)

      put api("/projects/#{project.id}/labels", user),
          params: request_params

      expect(response.status).to eq(200)
      expect(json_response['id']).to eq(expected_response_label_id)
      expect(json_response['priority']).to be_nil
    end

    def find_by_spec_params(params)
      if params.key?(:label_id)
        Label.find(params[:label_id])
      else
        Label.find_by(name: params[:name])
      end
    end
  end

  shared_examples 'label delete API' do
    it 'returns 204 for existing label' do
      delete api("/projects/#{project.id}/labels", user), params: spec_params

      expect(response).to have_gitlab_http_status(204)
    end
  end

  before do
    project.add_maintainer(user)
  end

  describe 'GET /projects/:id/labels' do
    let(:group) { create(:group) }
    let!(:group_label) { create(:group_label, title: 'feature', group: group) }

    before do
      project.update!(group: group)
    end

    it 'returns all available labels to the project' do
      get api("/projects/#{project.id}/labels", user)

      expect(response).to have_gitlab_http_status(200)
      expect(response).to include_pagination_headers
      expect(json_response).to all(match_schema('public_api/v4/labels/project_label'))
      expect(json_response.size).to eq(3)
      expect(json_response.map { |l| l['name'] }).to match_array([group_label.name, priority_label.name, label1.name])
    end

    context 'when the with_counts parameter is set' do
      before do
        create(:labeled_issue, project: project, labels: [group_label], author: user)
        create(:labeled_issue, project: project, labels: [label1], author: user, state: :closed)
        create(:labeled_merge_request, labels: [priority_label], author: user, source_project: project )
      end

      it 'includes counts in the response' do
        get api("/projects/#{project.id}/labels", user), params: { with_counts: true }

        expect(response).to have_gitlab_http_status(200)
        expect(response).to include_pagination_headers
        expect(json_response).to all(match_schema('public_api/v4/labels/project_label_with_counts'))
        expect(json_response.size).to eq(3)
        expect(json_response.map { |l| l['name'] }).to match_array([group_label.name, priority_label.name, label1.name])

        label1_response = json_response.find { |l| l['name'] == label1.title }
        group_label_response = json_response.find { |l| l['name'] == group_label.title }
        priority_label_response = json_response.find { |l| l['name'] == priority_label.title }

        expect(label1_response).to include('open_issues_count' => 0,
                                           'closed_issues_count' => 1,
                                           'open_merge_requests_count' => 0,
                                           'name' => label1.name,
                                           'description' => nil,
                                           'color' => a_string_matching(/^#\h{6}$/),
                                           'text_color' => a_string_matching(/^#\h{6}$/),
                                           'priority' => nil,
                                           'subscribed' => false,
                                           'is_project_label' => true)

        expect(group_label_response).to include('open_issues_count' => 1,
                                                'closed_issues_count' => 0,
                                                'open_merge_requests_count' => 0,
                                                'name' => group_label.name,
                                                'description' => nil,
                                                'color' => a_string_matching(/^#\h{6}$/),
                                                'text_color' => a_string_matching(/^#\h{6}$/),
                                                'priority' => nil,
                                                'subscribed' => false,
                                                'is_project_label' => false)

        expect(priority_label_response).to include('open_issues_count' => 0,
                                                   'closed_issues_count' => 0,
                                                   'open_merge_requests_count' => 1,
                                                   'name' => priority_label.name,
                                                   'description' => nil,
                                                   'color' => a_string_matching(/^#\h{6}$/),
                                                   'text_color' => a_string_matching(/^#\h{6}$/),
                                                   'priority' => 3,
                                                   'subscribed' => false,
                                                   'is_project_label' => true)
      end
    end
  end

  describe 'POST /projects/:id/labels' do
    it 'returns created label when all params' do
      post api("/projects/#{project.id}/labels", user),
           params: {
             name: 'Foo',
             color: '#FFAABB',
             description: 'test',
             priority: 2
           }

      expect(response).to have_gitlab_http_status(201)
      expect(json_response['name']).to eq('Foo')
      expect(json_response['color']).to eq('#FFAABB')
      expect(json_response['description']).to eq('test')
      expect(json_response['priority']).to eq(2)
    end

    it 'returns created label when only required params' do
      post api("/projects/#{project.id}/labels", user),
           params: {
             name: 'Foo & Bar',
             color: '#FFAABB'
           }

      expect(response.status).to eq(201)
      expect(json_response['name']).to eq('Foo & Bar')
      expect(json_response['color']).to eq('#FFAABB')
      expect(json_response['description']).to be_nil
      expect(json_response['priority']).to be_nil
    end

    it 'creates a prioritized label' do
      post api("/projects/#{project.id}/labels", user),
           params: {
             name: 'Foo & Bar',
             color: '#FFAABB',
             priority: 3
           }

      expect(response.status).to eq(201)
      expect(json_response['name']).to eq('Foo & Bar')
      expect(json_response['color']).to eq('#FFAABB')
      expect(json_response['description']).to be_nil
      expect(json_response['priority']).to eq(3)
    end

    it 'returns a 400 bad request if name not given' do
      post api("/projects/#{project.id}/labels", user), params: { color: '#FFAABB' }
      expect(response).to have_gitlab_http_status(400)
    end

    it 'returns a 400 bad request if color not given' do
      post api("/projects/#{project.id}/labels", user), params: { name: 'Foobar' }
      expect(response).to have_gitlab_http_status(400)
    end

    it 'returns 400 for invalid color' do
      post api("/projects/#{project.id}/labels", user),
           params: {
             name: 'Foo',
             color: '#FFAA'
           }
      expect(response).to have_gitlab_http_status(400)
      expect(json_response['message']['color']).to eq(['must be a valid color code'])
    end

    it 'returns 400 for too long color code' do
      post api("/projects/#{project.id}/labels", user),
           params: {
             name: 'Foo',
             color: '#FFAAFFFF'
           }
      expect(response).to have_gitlab_http_status(400)
      expect(json_response['message']['color']).to eq(['must be a valid color code'])
    end

    it 'returns 400 for invalid name' do
      post api("/projects/#{project.id}/labels", user),
           params: {
             name: ',',
             color: '#FFAABB'
           }
      expect(response).to have_gitlab_http_status(400)
      expect(json_response['message']['title']).to eq(['is invalid'])
    end

    it 'returns 409 if label already exists in group' do
      group = create(:group)
      group_label = create(:group_label, group: group)
      project.update(group: group)

      post api("/projects/#{project.id}/labels", user),
           params: {
             name: group_label.name,
             color: '#FFAABB'
           }

      expect(response).to have_gitlab_http_status(409)
      expect(json_response['message']).to eq('Label already exists')
    end

    it 'returns 400 for invalid priority' do
      post api("/projects/#{project.id}/labels", user),
           params: {
             name: 'Foo',
             color: '#FFAAFFFF',
             priority: 'foo'
           }

      expect(response).to have_gitlab_http_status(400)
    end

    it 'returns 409 if label already exists in project' do
      post api("/projects/#{project.id}/labels", user),
           params: {
             name: 'label1',
             color: '#FFAABB'
           }
      expect(response).to have_gitlab_http_status(409)
      expect(json_response['message']).to eq('Label already exists')
    end
  end

  describe 'DELETE /projects/:id/labels' do
    it_behaves_like 'label delete API' do
      let(:spec_params) { { name: 'label1' } }
    end

    it_behaves_like 'label delete API' do
      let(:spec_params) { { label_id: label1.id } }
    end

    it 'returns 404 for non existing label' do
      delete api("/projects/#{project.id}/labels", user), params: { name: 'label2' }

      expect(response).to have_gitlab_http_status(404)
      expect(json_response['message']).to eq('404 Label Not Found')
    end

    it 'returns 400 for wrong parameters' do
      delete api("/projects/#{project.id}/labels", user)

      expect(response).to have_gitlab_http_status(400)
    end

    it 'fails if label_id and name are given in params' do
      delete api("/projects/#{project.id}/labels", user),
          params: {
            label_id: label1.id,
            name: priority_label.name
          }

      expect(response).to have_gitlab_http_status(400)
    end

    it_behaves_like '412 response' do
      let(:request) { api("/projects/#{project.id}/labels", user) }
      let(:params) { { name: 'label1' } }
    end
  end

  describe 'PUT /projects/:id/labels' do
    context 'when using name' do
      it_behaves_like 'label update API' do
        let(:spec_params) { { name: 'label1' } }
        let(:expected_response_label_id) { label1.id }
      end
    end

    context 'when using label_id' do
      it_behaves_like 'label update API' do
        let(:spec_params) { { label_id: label1.id } }
        let(:expected_response_label_id) { label1.id }
      end
    end

    it 'returns 404 if label does not exist' do
      put api("/projects/#{project.id}/labels", user),
          params: {
            name: 'label2',
            new_name: 'label3'
          }

      expect(response).to have_gitlab_http_status(404)
    end

    it 'returns 404 if label by id does not exist' do
      put api("/projects/#{project.id}/labels", user),
          params: {
            label_id: 0,
            new_name: 'label3'
          }

      expect(response).to have_gitlab_http_status(404)
    end

    it 'returns 400 if no label name and id is given' do
      put api("/projects/#{project.id}/labels", user), params: { new_name: 'label2' }

      expect(response).to have_gitlab_http_status(400)
      expect(json_response['error']).to eq('label_id, name are missing, exactly one parameter must be provided')
    end

    it 'fails if label_id and name are given in params' do
      put api("/projects/#{project.id}/labels", user),
          params: {
            label_id: label1.id,
            name: priority_label.name,
            new_name: 'New Label'
          }

      expect(response).to have_gitlab_http_status(400)
    end
  end

  describe 'PUT /projects/:id/labels/promote' do
    let(:group) { create(:group) }

    before do
      group.add_owner(user)
      project.update!(group: group)
    end

    it 'returns 200 if label is promoted' do
      put api("/projects/#{project.id}/labels/promote", user), params: { name: label1.name }

      expect(response).to have_gitlab_http_status(200)
      expect(json_response['name']).to eq(label1.name)
      expect(json_response['color']).to eq(label1.color)
    end

    it 'returns 200 if group label already exists' do
      create(:group_label, title: label1.name, group: group)

      expect { put api("/projects/#{project.id}/labels/promote", user), params: { name: label1.name } }
        .to change(project.labels, :count).by(-1)
        .and change(group.labels, :count).by(0)

      expect(response).to have_gitlab_http_status(200)
    end

    it 'returns 403 if guest promotes label' do
      guest = create(:user)
      project.add_guest(guest)

      put api("/projects/#{project.id}/labels/promote", guest), params: { name: label1.name }

      expect(response).to have_gitlab_http_status(403)
    end

    it 'returns 404 if label does not exist' do
      put api("/projects/#{project.id}/labels/promote", user), params: { name: 'unknown' }

      expect(response).to have_gitlab_http_status(404)
    end

    it 'returns 400 if no label name given' do
      put api("/projects/#{project.id}/labels/promote", user)

      expect(response).to have_gitlab_http_status(400)
      expect(json_response['error']).to eq('name is missing')
    end
  end

  describe "POST /projects/:id/labels/:label_id/subscribe" do
    context "when label_id is a label title" do
      it "subscribes to the label" do
        post api("/projects/#{project.id}/labels/#{label1.title}/subscribe", user)

        expect(response).to have_gitlab_http_status(201)
        expect(json_response["name"]).to eq(label1.title)
        expect(json_response["subscribed"]).to be_truthy
      end
    end

    context "when label_id is a label ID" do
      it "subscribes to the label" do
        post api("/projects/#{project.id}/labels/#{label1.id}/subscribe", user)

        expect(response).to have_gitlab_http_status(201)
        expect(json_response["name"]).to eq(label1.title)
        expect(json_response["subscribed"]).to be_truthy
      end
    end

    context "when user is already subscribed to label" do
      before do
        label1.subscribe(user, project)
      end

      it "returns 304" do
        post api("/projects/#{project.id}/labels/#{label1.id}/subscribe", user)

        expect(response).to have_gitlab_http_status(304)
      end
    end

    context "when label ID is not found" do
      it "returns 404 error" do
        post api("/projects/#{project.id}/labels/1234/subscribe", user)

        expect(response).to have_gitlab_http_status(404)
      end
    end
  end

  describe "POST /projects/:id/labels/:label_id/unsubscribe" do
    before do
      label1.subscribe(user, project)
    end

    context "when label_id is a label title" do
      it "unsubscribes from the label" do
        post api("/projects/#{project.id}/labels/#{label1.title}/unsubscribe", user)

        expect(response).to have_gitlab_http_status(201)
        expect(json_response["name"]).to eq(label1.title)
        expect(json_response["subscribed"]).to be_falsey
      end
    end

    context "when label_id is a label ID" do
      it "unsubscribes from the label" do
        post api("/projects/#{project.id}/labels/#{label1.id}/unsubscribe", user)

        expect(response).to have_gitlab_http_status(201)
        expect(json_response["name"]).to eq(label1.title)
        expect(json_response["subscribed"]).to be_falsey
      end
    end

    context "when user is already unsubscribed from label" do
      before do
        label1.unsubscribe(user, project)
      end

      it "returns 304" do
        post api("/projects/#{project.id}/labels/#{label1.id}/unsubscribe", user)

        expect(response).to have_gitlab_http_status(304)
      end
    end

    context "when label ID is not found" do
      it "returns 404 error" do
        post api("/projects/#{project.id}/labels/1234/unsubscribe", user)

        expect(response).to have_gitlab_http_status(404)
      end
    end
  end
end
