require 'spec_helper'

describe 'View on environment', js: true do
  let(:branch_name) { 'feature' }
  let(:file_path) { 'files/ruby/feature.rb' }
  let(:project) { create(:project, :repository) }
  let(:user) { project.creator }

  before do
    project.add_master(user)
  end

  context 'when the branch has a route map' do
    let(:route_map) do
      <<-MAP.strip_heredoc
      - source: /files/(.*)\\..*/
        public: '\\1'
      MAP
    end

    before do
      Files::CreateService.new(
        project,
        user,
        start_branch: branch_name,
        branch_name: branch_name,
        commit_message: "Add .gitlab/route-map.yml",
        file_path: '.gitlab/route-map.yml',
        file_content: route_map
      ).execute

      # Update the file so that we still have a commit that will have a file on the environment
      Files::UpdateService.new(
        project,
        user,
        start_branch: branch_name,
        branch_name: branch_name,
        commit_message: "Update feature",
        file_path: file_path,
        file_content: "# Noop"
      ).execute
    end

    context 'and an active deployment' do
      let(:sha) { project.commit(branch_name).sha }
      let(:environment) { create(:environment, project: project, name: 'review/feature', external_url: 'http://feature.review.example.com') }
      let!(:deployment) { create(:deployment, environment: environment, ref: branch_name, sha: sha) }

      context 'when visiting the diff of a merge request for the branch' do
        let(:merge_request) { create(:merge_request, :simple, source_project: project, source_branch: branch_name) }

        before do
          login_as(user)

          visit diffs_namespace_project_merge_request_path(project.namespace, project, merge_request)

          wait_for_requests
        end

        it 'has a "View on env" button' do
          within '.diffs' do
            expect(page).to have_link('View on feature.review.example.com', href: 'http://feature.review.example.com/ruby/feature')
          end
        end
      end

      context 'when visiting a comparison for the branch' do
        before do
          login_as(user)

          visit namespace_project_compare_path(project.namespace, project, from: 'master', to: branch_name)

          wait_for_requests
        end

        it 'has a "View on env" button' do
          expect(page).to have_link('View on feature.review.example.com', href: 'http://feature.review.example.com/ruby/feature')
        end
      end

      context 'when visiting a comparison for the commit' do
        before do
          login_as(user)

          visit namespace_project_compare_path(project.namespace, project, from: 'master', to: sha)

          wait_for_requests
        end

        it 'has a "View on env" button' do
          expect(page).to have_link('View on feature.review.example.com', href: 'http://feature.review.example.com/ruby/feature')
        end
      end

      context 'when visiting a blob on the branch' do
        before do
          login_as(user)

          visit namespace_project_blob_path(project.namespace, project, File.join(branch_name, file_path))

          wait_for_requests
        end

        it 'has a "View on env" button' do
          expect(page).to have_link('View on feature.review.example.com', href: 'http://feature.review.example.com/ruby/feature')
        end
      end

      context 'when visiting a blob on the commit' do
        before do
          login_as(user)

          visit namespace_project_blob_path(project.namespace, project, File.join(sha, file_path))

          wait_for_requests
        end

        it 'has a "View on env" button' do
          expect(page).to have_link('View on feature.review.example.com', href: 'http://feature.review.example.com/ruby/feature')
        end
      end

      context 'when visiting the commit' do
        before do
          login_as(user)

          visit namespace_project_commit_path(project.namespace, project, sha)

          wait_for_requests
        end

        it 'has a "View on env" button' do
          expect(page).to have_link('View on feature.review.example.com', href: 'http://feature.review.example.com/ruby/feature')
        end
      end
    end
  end
end
