require 'rails_helper'

describe 'Milestone draggable', feature: true, js: true do
  include DragTo

  let(:milestone) { create(:milestone, project: project, title: 8.14) }
  let(:project)   { create(:empty_project, :public) }
  let(:user)      { create(:user) }

  context 'issues' do
    let(:issue)        { page.find_by_id('issues-list-unassigned').find('li') }
    let(:issue_target) { page.find_by_id('issues-list-ongoing') }

    it 'does not allow guest to drag issue' do
      create_and_drag_issue

      expect(issue_target).not_to have_selector('.issuable-row')
    end

    it 'does not allow authorized user to drag issue' do
      login_as(user)
      create_and_drag_issue

      expect(issue_target).not_to have_selector('.issuable-row')
    end

    it 'allows author to drag issue' do
      login_as(user)
      create_and_drag_issue(author: user)

      expect(issue_target).to have_selector('.issuable-row')
    end

    it 'allows admin to drag issue' do
      login_as(:admin)
      create_and_drag_issue

      expect(issue_target).to have_selector('.issuable-row')
    end

    it 'assigns issue when it has been dragged to ongoing list' do
      login_as(:admin)
      create_and_drag_issue

      expect(@issue.reload.assignees).not_to be_empty
      expect(page).to have_selector("#sortable_issue_#{@issue.iid} .assignee-icon img", count: 1)
    end
  end

  context 'merge requests' do
    let(:merge_request)        { page.find_by_id('merge_requests-list-unassigned').find('li') }
    let(:merge_request_target) { page.find_by_id('merge_requests-list-ongoing') }

    it 'does not allow guest to drag merge request' do
      create_and_drag_merge_request

      expect(merge_request_target).not_to have_selector('.issuable-row')
    end

    it 'does not allow authorized user to drag merge request' do
      login_as(user)
      create_and_drag_merge_request

      expect(merge_request_target).not_to have_selector('.issuable-row')
    end

    it 'allows author to drag merge request' do
      login_as(user)
      create_and_drag_merge_request(author: user)

      expect(merge_request_target).to have_selector('.issuable-row')
    end

    it 'allows admin to drag merge request' do
      login_as(:admin)
      create_and_drag_merge_request

      expect(merge_request_target).to have_selector('.issuable-row')
    end
  end

  def create_and_drag_issue(params = {})
    @issue = create(:issue, params.merge(title: 'Foo', project: project, milestone: milestone))

    visit namespace_project_milestone_path(project.namespace, project, milestone)
    scroll_into_view('.milestone-content')
    drag_to(selector: '.issues-sortable-list', list_to_index: 1)

    wait_for_requests
  end

  def create_and_drag_merge_request(params = {})
    create(:merge_request, params.merge(title: 'Foo', source_project: project, target_project: project, milestone: milestone))

    visit namespace_project_milestone_path(project.namespace, project, milestone)
    page.find("a[href='#tab-merge-requests']").click

    wait_for_requests

    scroll_into_view('.milestone-content')
    drag_to(selector: '.merge_requests-sortable-list', list_to_index: 1)

    wait_for_requests
  end

  def scroll_into_view(selector)
    page.evaluate_script("document.querySelector('#{selector}').scrollIntoView();")
  end
end
