# ##### BEGIN GPL LICENSE BLOCK #####
#
#  This program is free software; you can redistribute it and/or
#  modify it under the terms of the GNU General Public License
#  as published by the Free Software Foundation; either version 2
#  of the License, or (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program; if not, write to the Free Software Foundation,
#  Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
#
# ##### END GPL LICENSE BLOCK #####

from math import sin, cos, pi, sqrt, pow
from functools import reduce

import bpy
from bpy.props import FloatProperty, EnumProperty, BoolProperty, IntProperty
import bmesh
from mathutils import Vector, Matrix
from mathutils.geometry import barycentric_transform

from sverchok.node_tree import SverchCustomTreeNode
from sverchok.data_structure import (updateNode, Vector_generate,
                                     Vector_degenerate, match_long_repeat, fullList, cycle_for_length,
                                     rotate_list)

from sverchok.utils.sv_bmesh_utils import bmesh_from_pydata
from sverchok.utils.sv_mesh_utils import mesh_join
from sverchok.utils.geom import diameter, LineEquation2D, center
from sverchok.utils.logging import info, debug
# "coauthor": "Alessandro Zomparelli (sketchesofcode)"

cos_pi_6 = cos(pi/6)
sin_pi_6 = sin(pi/6)
sqrt_3 = sqrt(3)
sqrt_3_6 = sqrt_3/6
sqrt_3_3 = sqrt_3/3
sqrt_3_2 = sqrt_3/2

class SvAdaptivePolygonsNodeMk2(bpy.types.Node, SverchCustomTreeNode):
    """
    Triggers: Adaptive Polygons Tessellate Tissue
    Tooltip: Generate an adapted copy of donor object along each face of recipient object.
    """
    bl_idname = 'SvAdaptivePolygonsNodeMk2'
    bl_label = 'Adaptive Polygons Mk2'
    bl_icon = 'OUTLINER_OB_EMPTY'
    sv_icon = 'SV_ADAPTATIVE_POLS'

    axes = [
            ("X", "X", "Orient donor's X axis along normal", 0),
            ("Y", "Y", "Orient donor's Y axis along normal", 1),
            ("Z", "Z", "Orient donor's Z axis along normal", 2)
        ]

    normal_axis: EnumProperty(
        name = "Normal axis",
        description = "Donor object axis to be oriented along recipient face normal",
        items = axes,
        default = 'Z',
        update = updateNode)

    width_coef: FloatProperty(
        name='Width coeff', description='with coefficient for sverchok adaptivepols donors size',
        default=1.0, max=3.0, min=0.5, update=updateNode)
    
    z_coef: FloatProperty(
        name='Z coeff',
        default=1.0, max=3.0, min=0.0, update=updateNode)

    z_offset: FloatProperty(
        name = "Z offet",
        default = 0.0,
        update = updateNode)

    normal_interp_modes = [
            ("LINEAR", "Linear", "Exact / linear normals interpolation", 0),
            ("SMOOTH", "Unit length", "Use normals of unit length", 1)
        ]

    normal_interp_mode : EnumProperty(
        name = "Interpolate normals",
        description = "Normals interpolation mode",
        items = normal_interp_modes, default = "LINEAR",
        update = updateNode)

    normal_modes = [
            ("MAP", "Map", "Interpolate from donor vertex normals", 0),
            ("FACE", "Face", "Use donor face normals", 1)
        ]

    normal_mode : EnumProperty(
        name = "Use normals",
        description = "Normals mapping mode",
        items = normal_modes, default = "MAP",
        update = updateNode)
    
    z_scale_modes = [
            ("PROP", "Proportional", "Scale along normal proportionally with the donor object", 0),
            ("CONST", "Constant", "Constant scale along normal", 1),
            ("AUTO", "Auto", "Try to calculate the correct scale automatically", 2)
        ]

    z_scale : EnumProperty(
        name = "Z Scale",
        description = "Mode of scaling along the normals",
        items = z_scale_modes, default = "PROP",
        update = updateNode)

    z_rotation: FloatProperty(
        name = "Z Rotation",
        description = "Rotate donor object around recipient's face normal",
        min = 0, max = 2*pi, default = 0,
        update = updateNode)
    
    poly_rotation: IntProperty(
        name = "Polygons rotation",
        description = "Rotate indexes in polygons definition",
        min = 0, default = 0,
        update = updateNode)

    xy_modes = [
            ("BOUNDS", "Bounds", "Map donor object bounds to recipient face", 0),
            ("PLAIN", "As Is", "Map donor object's coordinate space to recipient face as-is", 1)
        ]

    xy_mode : EnumProperty(
        name = "Coordinates",
        description = "Donor object coordinates mapping",
        items = xy_modes, default = "BOUNDS",
        update = updateNode)

    map_modes = [
            ("QUADTRI", "Quads / Tris Auto", "Use Quads or Tris mapping automatically", 0),
            ("QUADS", "Quads Always", "Use Quads mapping even for the Tris", 1)
        ]

    map_mode : EnumProperty(
        name = "Faces mode",
        description = "Donor object mapping mode",
        items = map_modes, default = "QUADTRI",
        update = updateNode)

    skip_modes = [
            ("SKIP", "Skip", "Do not output anything", 0),
            ("ASIS", "As Is", "Output these faces as is", 1)
        ]

    mask_mode: EnumProperty(
        name = "Mask mode",
        description = "What to do with masked out faces",
        items = skip_modes, default = "SKIP",
        update = updateNode)

    ngon_modes = [
            ("QUADS", "As Quads", "Try to process as Quads", 0),
            ("SKIP", "Skip", "Do not output anything", 1),
            ("ASIS", "As Is", "Output these faces as is", 2)
        ]

    ngon_mode: EnumProperty(
        name = "NGons",
        description = "What to do with NGons",
        items = ngon_modes, default = "QUADS",
        update = updateNode)

    join : BoolProperty(
        name = "Join",
        description = "Output one joined mesh",
        default = False,
        update = updateNode)

    def sv_init(self, context):
        self.inputs.new('SvVerticesSocket', "VersR")
        self.inputs.new('SvStringsSocket', "PolsR")
        self.inputs.new('SvVerticesSocket', "VersD")
        self.inputs.new('SvStringsSocket', "PolsD")
        self.inputs.new('SvStringsSocket', "W_Coef").prop_name = 'width_coef'
        self.inputs.new('SvStringsSocket', "Z_Coef").prop_name = 'z_coef'
        self.inputs.new('SvStringsSocket', "Z_Offset").prop_name = 'z_offset'
        self.inputs.new('SvStringsSocket', "Z_Rotation").prop_name = 'z_rotation'
        self.inputs.new('SvStringsSocket', "PolyRotation").prop_name = 'poly_rotation'
        self.inputs.new('SvStringsSocket', "PolyMask")

        self.outputs.new('SvVerticesSocket', "Vertices")
        self.outputs.new('SvStringsSocket', "Polygons")

    def draw_buttons(self, context, layout):
        layout.prop(self, "join")

    def draw_buttons_ext(self, context, layout):
        self.draw_buttons(context, layout)

        layout.label(text = "Normal axis:")
        layout.prop(self, "normal_axis", expand=True)
        layout.prop(self, "z_scale")
        layout.prop(self, "normal_mode")
        if self.normal_mode == 'MAP':
            layout.prop(self, "normal_interp_mode")
        layout.prop(self, "xy_mode")
        layout.prop(self, "map_mode")
        layout.prop(self, "mask_mode")
        layout.prop(self, "ngon_mode")

    def get_triangle_directions(self):
        """
        Three normal of unit triangle's edges.
        This is not constant just because the normal can be X or Y or Z.
        """
        triangle_direction_1 = Vector((cos_pi_6, sin_pi_6, 0))
        triangle_direction_2 = Vector((-cos_pi_6, sin_pi_6, 0))
        triangle_direction_3 = Vector((0, -1, 0))

        if self.normal_axis == 'X':
            return triangle_direction_1.zxy, triangle_direction_2.zxy, triangle_direction_3.zxy
        elif self.normal_axis == 'Y':
            return triangle_direction_1.xzy, triangle_direction_2.xzy, triangle_direction_3.xzy
        else:
            return triangle_direction_1, triangle_direction_2, triangle_direction_3

    def to2d(self, v):
        """
        Convert vector to 2D.
        Remove the coordinate which is responsible for normal axis.
        """
        if self.normal_axis == 'X':
            return v.yz
        elif self.normal_axis == 'Y':
            return v.xz
        else:
            return v.xy

    def from2d(self, x, y):
        """
        Make 3D vector from X and Y.
        Add zero for the coordinate which is responsible for normal axis.
        """
        if self.normal_axis == 'X':
            return Vector((0, x, y))
        elif self.normal_axis == 'Y':
            return Vector((x, 0, y))
        else:
            return Vector((x, y, 0))

    def bounding_triangle(self, vertices):
        """
        Return three vertices of a triangle with equal sides,
        which contains all provided vertices.
        """
        X, Y = self.get_other_axes()

        triangle_direction_1, triangle_direction_2, triangle_direction_3 = self.get_triangle_directions()
        max_1 = self.to2d(max(vertices, key = lambda vertex: triangle_direction_1.dot(vertex)))
        max_2 = self.to2d(max(vertices, key = lambda vertex: triangle_direction_2.dot(vertex)))
        max_3 = self.to2d(min(vertices, key = lambda vertex: vertex[Y]))

        side_1 = LineEquation2D.from_normal_and_point(self.to2d(triangle_direction_1), max_1)
        side_2 = LineEquation2D.from_normal_and_point(self.to2d(triangle_direction_2), max_2)
        side_3 = LineEquation2D.from_normal_and_point(self.to2d(triangle_direction_3), max_3)

        p1 = side_2.intersect_with_line(side_3)
        p2 = side_1.intersect_with_line(side_3)
        p3 = side_1.intersect_with_line(side_2)

        p1 = self.from2d(p1[0], p1[1])
        p2 = self.from2d(p2[0], p2[1])
        p3 = self.from2d(p3[0], p3[1])

        return p1, p2, p3

    def interpolate_quad_2d(self, dst_vert_1, dst_vert_2, dst_vert_3, dst_vert_4, v, x_coef, y_coef):
        """
        Map the provided `v` vertex, considering only two of it's coordinates,
        from the [-1/2; 1/2] x [-1/2; 1/2] square to the face defined by
        four `dst_vert_n` vertices.
        """
        X, Y = self.get_other_axes()
        v12 = dst_vert_1 + (dst_vert_2-dst_vert_1)*v[X]*x_coef + ((dst_vert_2-dst_vert_1)/2)
        v43 = dst_vert_4 + (dst_vert_3-dst_vert_4)*v[X]*x_coef + ((dst_vert_3-dst_vert_4)/2)
        return v12 + (v43-v12)*v[Y]*y_coef + ((v43-v12)/2)

    def interpolate_quad_3d(self, dst_vert_1, dst_vert_2, dst_vert_3, dst_vert_4, face_normal, v, x_coef, y_coef, z_coef, z_offset):
        """
        Map the provided `v` vertex from the source
        [-1/2; 1/2] x [-1/2; 1/2] x [-1/2; 1/2] cube
        to the space defined by `dst_vert_n` vertices.
        """
        loc = self.interpolate_quad_2d(dst_vert_1.co, dst_vert_2.co, dst_vert_3.co, dst_vert_4.co, v, x_coef, y_coef)
        if self.normal_mode == 'MAP':
            if self.normal_interp_mode == 'SMOOTH':
                normal = self.interpolate_quad_2d(dst_vert_1.normal, dst_vert_2.normal, dst_vert_3.normal, dst_vert_4.normal, v, x_coef, y_coef)
                normal.normalize()
            else:
                normal = self.interpolate_quad_2d(dst_vert_1.co + dst_vert_1.normal, dst_vert_2.co + dst_vert_2.normal,
                                             dst_vert_3.co + dst_vert_3.normal, dst_vert_4.co + dst_vert_4.normal,
                                             v, x_coef, y_coef)
                normal = normal - loc
        else:
            #normal = (dst_vert_1.normal + dst_vert_2.normal + dst_vert_3.normal + dst_vert_4.normal) * 0.25
            normal = face_normal
        Z = self.normal_axis_idx()
        return loc + normal*(v[Z]*z_coef + z_offset)

    def interpolate_tri_2d(self, dst_vert_1, dst_vert_2, dst_vert_3, src_vert_1, src_vert_2, src_vert_3, v):
        """
        Map the provided `v` vertex, considering only two of it's coordinates,
        from the source triangle (defined by `src_vert_n` vertices) to the face defined by
        three `dst_vert_n` vertices.
        """
        X, Y = self.get_other_axes()
        v = self.from2d(v[X], v[Y])
        return barycentric_transform(v, src_vert_1, src_vert_2, src_vert_3,
                                        dst_vert_1, dst_vert_2, dst_vert_3)

    def interpolate_tri_3d(self, dst_vert_1, dst_vert_2, dst_vert_3, src_vert_1, src_vert_2, src_vert_3, face_normal, v, z_coef, z_offset):
        """
        Map the provided `v` vertex from the source triangle
        to the space defined by `dst_vert_n` vertices.
        """
        v_at_triangle = self.interpolate_tri_2d(dst_vert_1.co, dst_vert_2.co, dst_vert_3.co,
                                            src_vert_1, src_vert_2, src_vert_3, v)
        if self.normal_mode == 'MAP':
            if self.normal_interp_mode == 'SMOOTH':
                normal = self.interpolate_tri_2d(dst_vert_1.normal, dst_vert_2.normal, dst_vert_3.normal,
                                             src_vert_1, src_vert_2, src_vert_3, v)
                normal.normalize()
            else:
                normal = self.interpolate_tri_2d(dst_vert_1.co + dst_vert_1.normal, dst_vert_2.co + dst_vert_2.normal,
                                            dst_vert_3.co + dst_vert_3.normal,
                                            src_vert_1, src_vert_2, src_vert_3, v)
                normal = normal - v_at_triangle
        else:
            #normal = (dst_vert_1.normal + dst_vert_2.normal + dst_vert_3.normal) * 0.333333333
            normal = face_normal
        Z = self.normal_axis_idx()
        return v_at_triangle + normal * (v[Z] * z_coef + z_offset)

    def get_other_axes(self):
        if self.normal_axis == 'X':
            return 1, 2
        elif self.normal_axis == 'Y':
            return 0, 2
        else:
            return 0, 1

    def normal_axis_idx(self):
        return "XYZ".index(self.normal_axis)

    def map_bounds(self, min, max, x):
        c = (min + max) / 2.0
        k = 1.0 / (max - min)
        return (x - c) * k

    def rotate_z(self, verts, angle):
        if abs(angle) < 1e-6:
            return verts
        projection = [self.to2d(v) for v in verts]
        x0, y0 = center(projection)
        c = self.from2d(x0, y0)
        rot = Matrix.Rotation(angle, 4, self.normal_axis)
        result = [(rot @ (v - c)) + c for v in verts]
        return result

    def calc_z_scale(self, dst_verts, src_verts):
        src_lens = []
        for v1, v2 in zip(src_verts, src_verts[1:]):
            src_lens.append((v1 - v2).length)
        src_lens.append((src_verts[-1] - src_verts[0]).length)

        dst_lens = []
        for v1, v2 in zip(dst_verts, dst_verts[1:]):
            dst_lens.append((v1 - v2).length)
        dst_lens.append((dst_verts[-1] - dst_verts[0]).length)

        scales = [dst_len / src_len for src_len,dst_len in zip(src_lens, dst_lens) if abs(src_len) > 1e-6 and abs(dst_len) > 1e-6]
        n = len(scales)
        prod = reduce(lambda x,y: x*y, scales, 1.0)
        return pow(prod, 1.0/n)

    def _process(self, verts_recpt, faces_recpt, verts_donor, faces_donor, zcoefs, zoffsets, zrotations, wcoefs, facerots, mask):
        bm = bmesh_from_pydata(verts_recpt, None, faces_recpt, normal_update=True)
        bm.verts.ensure_lookup_table()
        # Original (unrotated) donor vertices
        donor_verts_o = [Vector(v) for v in verts_donor]

        X, Y = self.get_other_axes()
        Z = self.normal_axis_idx()

        # Vertices of the unit triangle.
        # In case xy_mode != BOUNDS, we will never
        # have to recalculate these.
        tri_vert_1 = self.from2d(-0.5, -sqrt_3_6)
        tri_vert_2 = self.from2d(0.5, -sqrt_3_6)
        tri_vert_3 = self.from2d(0, sqrt_3_3)

        # We will be rotating the donor object around Z axis,
        # so it's size along Z is not going to change.
        z_size = diameter(donor_verts_o, Z)

        verts_out = []
        faces_out = []

        tri_vert_idxs = [0, 1, 2]
        quad_vert_idxs = [0, 1, 2, -1]

        prev_angle = None
        for recpt_face, recpt_face_bm, zcoef, zoffset, angle, wcoef, facerot, m in zip(faces_recpt, bm.faces, zcoefs, zoffsets, zrotations, wcoefs, facerots, mask):

            recpt_face_vertices_bm = [bm.verts[i] for i in recpt_face]

            # We have to recalculate rotated vertices only if
            # the rotation angle have changed.
            if prev_angle is None or angle != prev_angle:
                donor_verts_v = self.rotate_z(donor_verts_o, angle)

                if self.xy_mode == 'BOUNDS' or self.z_scale == 'AUTO' :
                    max_x = max(v[X] for v in donor_verts_v)
                    min_x = min(v[X] for v in donor_verts_v)
                    max_y = max(v[Y] for v in donor_verts_v)
                    min_y = min(v[Y] for v in donor_verts_v)

                if self.xy_mode == 'BOUNDS':
                    tri_vert_1, tri_vert_2, tri_vert_3 = self.bounding_triangle(donor_verts_v)

            prev_angle = angle

            if self.z_scale == 'CONST':
                if abs(z_size) < 1e-6:
                    zcoef = 0
                else:
                    zcoef = zcoef / z_size

            # Define TRI/QUAD mode based on node settings.
            n = len(recpt_face)
            if not m:
                map_mode = self.mask_mode
            else:
                if n == 3:
                    if self.map_mode == 'QUADTRI':
                        map_mode = 'TRI'
                    else:
                        map_mode = 'QUAD'
                elif n == 4:
                    map_mode = 'QUAD'
                else:
                    if self.ngon_mode == 'QUADS':
                        map_mode = 'QUAD'
                    elif self.ngon_mode == 'ASIS':
                        map_mode = 'ASIS'
                    else:
                        map_mode = 'SKIP'

            if map_mode == 'SKIP':
                # Skip this recipient's face - do not produce any vertices/faces for it
                continue

            elif map_mode == 'ASIS':
                # Leave this recipient's face as it was - as a single face.
                verts_out.append([verts_recpt[i] for i in recpt_face])
                faces_out.append([list(range(n))])

            elif map_mode == 'TRI':
                # Tris processing mode.
                #
                # As interpolate_tri_3d is based on barycentric_transform,
                # here we do not have to manually map donor vertices to the
                # unit triangle.

                i0, i1, i2 = rotate_list(tri_vert_idxs, facerot)
                if self.z_scale == 'AUTO':
                    zcoef = self.calc_z_scale(
                                        [recpt_face_vertices_bm[i0].co,
                                         recpt_face_vertices_bm[i1].co,
                                         recpt_face_vertices_bm[i2].co],
                                        [tri_vert_1/wcoef, tri_vert_2/wcoef, tri_vert_3/wcoef]
                                    ) * zcoef
                #self.info("T: %s, %s, %s", tri_vert_1, tri_vert_2, tri_vert_3)
                new_verts = []
                for v in donor_verts_v:
                    new_verts.append(self.interpolate_tri_3d(
                                        recpt_face_vertices_bm[i0],
                                        recpt_face_vertices_bm[i1],
                                        recpt_face_vertices_bm[i2],
                                        tri_vert_1/wcoef, tri_vert_2/wcoef, tri_vert_3/wcoef,
                                        recpt_face_bm.normal,
                                        v, zcoef, zoffset))
                verts_out.append(new_verts)
                faces_out.append(faces_donor)

            elif map_mode == 'QUAD':
                # Quads processing mode.
                #
                # It can process Tris, but it will look strange:
                # triangle will be processed as degenerated Quad,
                # where third and fourth vertices coincide.
                # In Tissue addon, this is the only mode possible for Quads.
                # Someone may like that behaivour, so we allow it with setting...
                #
                # This can process NGons in even worse way:
                # it will take first three vertices and the last one
                # and consider that as a Quad.

                i0, i1, i2, i3 = rotate_list(quad_vert_idxs, facerot)
                if self.z_scale == 'AUTO':
                    corner1 = self.from2d(min_x, min_y)
                    corner2 = self.from2d(min_x, max_y)
                    corner3 = self.from2d(max_x, max_y)
                    corner4 = self.from2d(max_x, min_y)

                    zcoef = self.calc_z_scale(
                                    [recpt_face_vertices_bm[i0].co,
                                     recpt_face_vertices_bm[i1].co,
                                     recpt_face_vertices_bm[i2].co,
                                     recpt_face_vertices_bm[i3].co],
                                    [corner1, corner2, corner3, corner4]
                                ) * zcoef

                new_verts = []
                for v in donor_verts_v:
                    if self.xy_mode == 'BOUNDS':
                        # Map the `v` vertex's X, Y coordinates
                        # from it's bounding square to
                        # [-1/2; 1/2] square.
                        # Leave Z coordinate as it was.
                        x = self.map_bounds(min_x, max_x, v[X])
                        y = self.map_bounds(min_y, max_y, v[Y])
                        z = v[Z]

                        v = Vector((0, 0, 0))
                        v[X] = x
                        v[Y] = y
                        v[Z] = z

                    new_verts.append(self.interpolate_quad_3d(
                                        recpt_face_vertices_bm[i0],
                                        recpt_face_vertices_bm[i1],
                                        recpt_face_vertices_bm[i2],
                                        recpt_face_vertices_bm[i3],
                                        recpt_face_bm.normal,
                                        v,
                                        wcoef, wcoef,
                                        zcoef, zoffset))

                verts_out.append(new_verts)
                faces_out.append(faces_donor)

        bm.free()

        return verts_out, faces_out

    def process(self):
        if not any(output.is_linked for output in self.outputs):
            return

        verts_donor_s = self.inputs['VersR'].sv_get()
        faces_donor_s = self.inputs['PolsR'].sv_get()
        verts_recpt_s = self.inputs['VersD'].sv_get()
        faces_recpt_s = self.inputs['PolsD'].sv_get(default=[[]])
        zcoefs_s = self.inputs['Z_Coef'].sv_get()
        zoffsets_s = self.inputs['Z_Offset'].sv_get()
        zrotations_s = self.inputs['Z_Rotation'].sv_get()
        wcoefs_s = self.inputs['W_Coef'].sv_get()
        if 'PolyRotation' in self.inputs:
            facerots_s = self.inputs['PolyRotation'].sv_get(default = [[0]])
        else:
            facerots_s = [[0]]
        mask_s = self.inputs['PolyMask'].sv_get(default = [[1]])

        verts_out = []
        faces_out = []

        objects = match_long_repeat([verts_recpt_s, faces_recpt_s, verts_donor_s, faces_donor_s, zcoefs_s, zoffsets_s, zrotations_s, wcoefs_s, facerots_s, mask_s])
        for verts_donor, faces_donor, verts_recpt, faces_recpt, zcoefs, zoffsets, zrotations, wcoefs, facerots, mask in zip(*objects):
            fullList(zcoefs, len(faces_recpt))
            fullList(zoffsets, len(faces_recpt))
            fullList(zrotations, len(faces_recpt))
            fullList(wcoefs, len(faces_recpt))
            fullList(facerots, len(faces_recpt))
            mask = cycle_for_length(mask, len(faces_recpt))

            new_verts, new_faces = self._process(verts_recpt, faces_recpt,
                                                 verts_donor, faces_donor,
                                                 zcoefs, zoffsets, zrotations,
                                                 wcoefs, facerots, mask)
            verts_out.extend(new_verts)
            faces_out.extend(new_faces)

            verts_out = Vector_degenerate(verts_out)
            if self.join:
                verts_out, _, faces_out = mesh_join(verts_out, [], faces_out)
                verts_out = [verts_out]
                faces_out = [faces_out]

            self.outputs['Vertices'].sv_set(verts_out)
            self.outputs['Polygons'].sv_set(faces_out)

def register():
    bpy.utils.register_class(SvAdaptivePolygonsNodeMk2)

def unregister():
    bpy.utils.unregister_class(SvAdaptivePolygonsNodeMk2)

