# ##### BEGIN GPL LICENSE BLOCK #####
#
#  This program is free software; you can redistribute it and/or
#  modify it under the terms of the GNU General Public License
#  as published by the Free Software Foundation; either version 2
#  of the License, or (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program; if not, write to the Free Software Foundation,
#  Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
#
# ##### END GPL LICENSE BLOCK #####

import bpy
from bpy.props import IntProperty, EnumProperty, BoolProperty, FloatProperty
import bmesh.ops

from sverchok.node_tree import SverchCustomTreeNode
from sverchok.data_structure import updateNode, match_long_repeat, repeat_last_for_length
from sverchok.utils.sv_bmesh_utils import bmesh_from_pydata, pydata_from_bmesh


class SvRecalcNormalsNode(bpy.types.Node, SverchCustomTreeNode):
    ''' Recalc face normals '''
    bl_idname = 'SvRecalcNormalsNode'
    bl_label = 'Recalc normals'
    bl_icon = 'OUTLINER_OB_EMPTY'
    sv_icon = 'SV_RECALC_NORMALS'

    invert: BoolProperty(
        name="Inside", description="Calculate inside normals", default=False, update=updateNode)

    def sv_init(self, context):
        self.inputs.new('SvVerticesSocket', "Vertices")
        self.inputs.new('SvStringsSocket', 'Edges')
        self.inputs.new('SvStringsSocket', 'Polygons')
        self.inputs.new('SvStringsSocket', 'Mask')

        self.outputs.new('SvVerticesSocket', 'Vertices')
        self.outputs.new('SvStringsSocket', 'Edges')
        self.outputs.new('SvStringsSocket', 'Polygons')

    def draw_buttons(self, context, layout):
        layout.prop(self, "invert")

    def process(self):
        if not (self.inputs['Vertices'].is_linked and self.inputs['Polygons'].is_linked):
            return
        if not (any(self.outputs[name].is_linked for name in ['Vertices', 'Edges', 'Polygons'])):
            return

        vertices_s = self.inputs['Vertices'].sv_get(default=[[]], deepcopy=False)
        edges_s = self.inputs['Edges'].sv_get(default=[[]], deepcopy=False)
        faces_s = self.inputs['Polygons'].sv_get(default=[[]], deepcopy=False)
        mask_s = self.inputs['Mask'].sv_get(default=[[True]], deepcopy=False)

        result_vertices = []
        result_edges = []
        result_faces = []

        meshes = match_long_repeat([vertices_s, edges_s, faces_s, mask_s])

        for vertices, edges, faces, mask in zip(*meshes):

            bm = bmesh_from_pydata(vertices, edges, faces, normal_update=True)
            mask_matched = repeat_last_for_length(mask, len(faces))

            b_faces = []
            for m, face in zip(mask_matched, bm.faces):
                if m:
                    b_faces.append(face)

            bmesh.ops.recalc_face_normals(bm, faces=b_faces)
            new_vertices, new_edges, new_faces = pydata_from_bmesh(bm)
            bm.free()

            if self.invert:
                new_faces = [list(reversed(face)) for face in new_faces]

            result_vertices.append(new_vertices)
            result_edges.append(new_edges)
            result_faces.append(new_faces)

        if self.outputs['Vertices'].is_linked:
            self.outputs['Vertices'].sv_set(result_vertices)
        if self.outputs['Edges'].is_linked:
            self.outputs['Edges'].sv_set(result_edges)
        if self.outputs['Polygons'].is_linked:
            self.outputs['Polygons'].sv_set(result_faces)

def register():
    bpy.utils.register_class(SvRecalcNormalsNode)


def unregister():
    bpy.utils.unregister_class(SvRecalcNormalsNode)
