from sverchok.utils.testing import SverchokTestCase
from sverchok.utils.geom_2d.make_monotone import monotone_sv_face_with_holes
from sverchok.utils.geom_2d.intersections import intersect_sv_edges
from sverchok.utils.geom_2d.merge_mesh import edges_to_faces, merge_mesh_light, crop_mesh, crop_edges, merge_mesh
from sverchok.utils.geom_2d.dissolve_mesh import dissolve_faces


class MakeMonotoneTest(SverchokTestCase):
    def test_make_monotone_one_face(self):
        # https://user-images.githubusercontent.com/28003269/67469986-13241000-f65e-11e9-85c4-7656f1ae16a5.png
        main_face = [[2.309999942779541,0.0,0.0],[1.7819089889526367,-0.5280910134315491,0.0],[1.7819087505340576,-1.7819093465805054,0.0],[0.7080908417701721,-1.7819091081619263,0.0],[2.969295209709344e-08,-2.490000009536743,0.0],[-0.7505173683166504,-1.7394826412200928,0.0],[-1.7394828796386719,-1.7394825220108032,0.0],[-1.7394826412200928,-0.6905173659324646,0.0],[-2.430000066757202,-2.12437356594819e-07,0.0],[-1.6970562934875488,0.7329436540603638,0.0],[-1.6970562934875488,1.6970562934875488,0.0],[-0.6729437708854675,1.6970562934875488,0.0],[-1.0359598690001803e-07,2.369999885559082,0.0],[0.7153699994087219,1.6546299457550049,0.0],[1.6546298265457153,1.6546298265457153,0.0],[1.6546299457550049,0.6553700566291809,0.0]]
        vert_hols = [[0.7099999785423279,9.797174820681343e-17,0.0],[0.48083260655403137,0.48083260655403137,0.0],[-2.841240132056555e-08,0.6499999761581421,0.0],[-0.43840619921684265,0.43840619921684265,0.0],[-0.5899999737739563,-5.157943760991657e-08,0.0],[-0.3959798216819763,-0.39597976207733154,0.0],[6.320186596298072e-09,-0.5299999713897705,0.0],[0.35355329513549805,-0.3535534143447876,0.0],[0.48083263635635376,1.1308326721191406,0.0],[-0.43840622901916504,1.0884062051773071,0.0],[1.1908326148986816,0.48083263635635376,0.0],[-1.0284063816070557,0.43840619921684265,0.0],[1.0635534524917603,-0.35355353355407715,0.0],[-0.9859795570373535,-0.39597970247268677,0.0],[0.3535532057285309,-0.883553147315979,0.0],[-0.3959798514842987,-0.9259797930717468,0.0],[1.8600000143051147,0.0,0.0],[1.5099999904632568,0.3500000536441803,0.0],[1.159999966621399,4.2862643149203325e-17,0.0],[1.5099999904632568,-0.3500000536441803,0.0],[1.3364317417144775,1.3364317417144775,0.0],[0.7990306615829468,1.336431860923767,0.0],[0.7990306615829468,0.7990306615829468,0.0],[1.336431860923767,0.7990306615829468,0.0],[-8.392586892114196e-08,1.9200000762939453,0.0],[-0.41000011563301086,1.5099999904632568,0.0],[-4.808252285215531e-08,1.0999999046325684,0.0],[0.4099999964237213,1.5099999904632568,0.0],[-1.378858208656311,1.378858208656311,0.0],[-1.3788583278656006,0.7566042542457581,0.0],[-0.7566041946411133,0.7566041946411133,0.0],[-0.7566042542457581,1.3788583278656006,0.0],[-1.9800000190734863,-1.730970922153574e-07,0.0],[-1.5099999904632568,-0.47000017762184143,0.0],[-1.0399999618530273,-9.091968422580976e-08,0.0],[-1.5099999904632568,0.46999993920326233,0.0],[-1.4212846755981445,-1.4212844371795654,0.0],[-0.7141778469085693,-1.421284556388855,0.0],[-0.7141779065132141,-0.7141777873039246,0.0],[-1.4212846755981445,-0.7141777276992798,0.0],[2.4326755720949222e-08,-2.0399999618530273,0.0],[0.5300000309944153,-1.5099999904632568,0.0],[1.1686382528353079e-08,-0.9799999594688416,0.0],[-0.5300000309944153,-1.5099999904632568,0.0],[1.4637106657028198,-1.4637112617492676,0.0],[1.463710904121399,-0.6717516183853149,0.0],[0.6717512607574463,-0.6717514991760254,0.0],[0.6717511415481567,-1.463711142539978,0.0]]
        face_hols = [[0,12,7,14,6,15,5,13,4,11,3,9,2,8,1,10],[16,19,18,17],[20,23,22,21],[24,27,26,25],[28,31,30,29],[32,35,34,33],[36,39,38,37],[40,43,42,41],[44,47,46,45]]

        vert_expect_result = [[1.6546299457550049,0.6553700566291809,0.0],[1.6546298265457153,1.6546298265457153,0.0],[0.7153699994087219,1.6546299457550049,0.0],[-1.0359598690001803e-07,2.369999885559082,0.0],[-0.6729437708854675,1.6970562934875488,0.0],[-1.6970562934875488,1.6970562934875488,0.0],[-1.6970562934875488,0.7329436540603638,0.0],[-2.430000066757202,-2.12437356594819e-07,0.0],[-1.7394826412200928,-0.6905173659324646,0.0],[-1.7394828796386719,-1.7394825220108032,0.0],[-0.7505173683166504,-1.7394826412200928,0.0],[2.969295209709344e-08,-2.490000009536743,0.0],[0.7080908417701721,-1.7819091081619263,0.0],[1.7819087505340576,-1.7819093465805054,0.0],[1.7819089889526367,-0.5280910134315491,0.0],[2.309999942779541,0.0,0.0],[1.1908326148986816,0.48083263635635376,0.0],[0.48083260655403137,0.48083260655403137,0.0],[0.48083263635635376,1.1308326721191406,0.0],[-2.841240132056555e-08,0.6499999761581421,0.0],[-0.43840622901916504,1.0884062051773071,0.0],[-0.43840619921684265,0.43840619921684265,0.0],[-1.0284063816070557,0.43840619921684265,0.0],[-0.5899999737739563,-5.157943760991657e-08,0.0],[-0.9859795570373535,-0.39597970247268677,0.0],[-0.3959798216819763,-0.39597976207733154,0.0],[-0.3959798514842987,-0.9259797930717468,0.0],[6.320186596298072e-09,-0.5299999713897705,0.0],[0.3535532057285309,-0.883553147315979,0.0],[0.35355329513549805,-0.3535534143447876,0.0],[1.0635534524917603,-0.35355353355407715,0.0],[0.7099999785423279,9.797174820681343e-17,0.0],[1.5099999904632568,0.3500000536441803,0.0],[1.159999966621399,4.2862643149203325e-17,0.0],[1.5099999904632568,-0.3500000536441803,0.0],[1.8600000143051147,0.0,0.0],[0.7990306615829468,1.336431860923767,0.0],[0.7990306615829468,0.7990306615829468,0.0],[1.336431860923767,0.7990306615829468,0.0],[1.3364317417144775,1.3364317417144775,0.0],[-0.41000011563301086,1.5099999904632568,0.0],[-4.808252285215531e-08,1.0999999046325684,0.0],[0.4099999964237213,1.5099999904632568,0.0],[-8.392586892114196e-08,1.9200000762939453,0.0],[-1.3788583278656006,0.7566042542457581,0.0],[-0.7566041946411133,0.7566041946411133,0.0],[-0.7566042542457581,1.3788583278656006,0.0],[-1.378858208656311,1.378858208656311,0.0],[-1.5099999904632568,-0.47000017762184143,0.0],[-1.0399999618530273,-9.091968422580976e-08,0.0],[-1.5099999904632568,0.46999993920326233,0.0],[-1.9800000190734863,-1.730970922153574e-07,0.0],[-0.7141778469085693,-1.421284556388855,0.0],[-0.7141779065132141,-0.7141777873039246,0.0],[-1.4212846755981445,-0.7141777276992798,0.0],[-1.4212846755981445,-1.4212844371795654,0.0],[0.5300000309944153,-1.5099999904632568,0.0],[1.1686382528353079e-08,-0.9799999594688416,0.0],[-0.5300000309944153,-1.5099999904632568,0.0],[2.4326755720949222e-08,-2.0399999618530273,0.0],[1.463710904121399,-0.6717516183853149,0.0],[0.6717512607574463,-0.6717514991760254,0.0],[0.6717511415481567,-1.463711142539978,0.0],[1.4637106657028198,-1.4637112617492676,0.0]]
        face_expect_result = [[3,43,42,36,39,38,0,1,2],[43,3,4,40],[40,4,5,6,45,44,47],[50,6,7,8,48,51],[54,8,9,10,11,59,58,52,55],[12,56,59,11],[14,61,60,63,56,12,13],[36,18,17,16,32,35,34,30,14,15,0,38,37],[36,42,41,20,19,18],[40,47,46,45,6,50,22,21,20,41],[22,50,49,48,8,54,53,52,58,57,26,25,24,23],[26,28,27],[14,30,29,28,26,57,56,63,62,61],[32,16,31,30,34,33]]

        vert_result, face_result = monotone_sv_face_with_holes(main_face, vert_hols, face_hols, accuracy=5)

        self.assert_sverchok_data_equal(vert_expect_result, vert_result, precision=5)
        self.assert_sverchok_data_equal(face_expect_result, face_result)


class IntersectionEdgesTest(SverchokTestCase):
    def test_intersect_random_edges(self):
        # https://user-images.githubusercontent.com/28003269/67544464-9731d280-f706-11e9-88e8-a3a1fc97a33a.png
        sv_points = [[-0.7997182304959439,0.6003755090045197,0.0],[-0.19766237037635295,-0.1571047955828267,0.0],[0.404393489743238,-0.9145851001701731,0.0],[-0.999900408118644,0.014112896377748907,0.0],[-0.9880298280393633,0.11558423566978356,0.0],[-0.9761592479600828,0.2170555749618182,0.0],[-0.3664489746746277,-0.9304381489169036,0.0],[-0.20949457886748124,0.03409033091160191,0.0],[-0.05254018306033478,0.9986188107401074,0.0],[0.3007765819551887,0.9536946302393412,0.0],[-0.34650456261641804,0.42119219812941183,0.0],[-0.9937857071880247,-0.11131023398051754,0.0],[0.4156821687110949,0.9095099420103339,0.0],[0.18140931605311897,-0.04454590184046181,0.0],[-0.05286353660485697,-0.9986017456912575,0.0],[0.35672372712476613,0.9342099242173653,0.0],[-0.2668731394326972,0.6946259328080246,0.0],[-0.8904700059901606,0.4550419413986839,0.0],[-0.9954551580793016,-0.09523144571680432,0.0],[-0.1569679281812693,0.3182843459402713,0.0],[0.681519301716763,0.7318001375973469,0.0],[0.766509618764487,0.6422328272063963,0.0],[-0.1003227902558877,0.19402312059281424,0.0],[-0.9671551992762624,-0.2541865860207678,0.0],[-0.27129636568262766,0.9624958607533842,0.0],[0.1993058156405517,0.1100258689350121,0.0],[0.669907996963731,-0.74244412288336,0.0],[0.9986057183994101,0.05278843793860495,0.0],[0.9975437310504423,-0.01551083582623871,0.0],[0.9964817437014744,-0.08381010959108237,0.0],[0.6011051906889029,-0.7991699129264425,0.0],[-0.08176893838578297,-0.07735794821257624,0.0],[-0.7646430674604688,0.64445401650129,0.0],[0.8654909293244962,-0.5009245963785566,0.0],[0.0001250654483755942,0.0002159621047279625,0.0],[-0.865240798427745,0.5013565205880125,0.0],[0.08345987434785694,-0.9965111386100207,0.0],[0.5361686525644019,-0.5726219375302932,0.0],[0.9888774307809469,-0.14873273645056573,0.0],[-0.85265158312796,0.5224799305134922,0.0],[-0.39492375570820565,-0.23777297259568342,0.0],[0.0628040717115487,-0.998025875704859,0.0],[-0.7989583617006947,-0.6013863452627952,0.0],[-0.01944389656626916,0.02422709866468492,0.0],[0.7600705685681564,0.6498405425921651,0.0],[0.986067876718799,-0.16634344743114893,0.0],[-0.0060330263665178685,-0.052640351010438416,0.0],[-0.9981339294518348,0.0610627454102721,0.0],[0.8359205354299549,-0.5488504882447474,0.0],[-0.05658854906820854,-0.43191626129376076,0.0],[-0.949097633566372,-0.31498203434277416,0.0],[-0.28579513859882766,-0.9582907381130619,0.0],[0.05964693008125718,-0.02200674044302886,0.0],[0.405088998761342,0.9142772572270041,0.0],[0.7494211287946771,-0.6620936276057275,0.0],[0.764217098789785,-0.017542970659616353,0.0],[0.779013068784893,0.6270076862864948,0.0],[0.47365213462380934,-0.8807120161359835,0.0],[-0.23084319896830222,-0.26347905259630533,0.0],[-0.9353385325604138,0.3537539109433729,0.0],[-0.7707364355568286,0.6371540998103635,0.0],[-0.7459187148394918,0.6649920309164105,0.0],[-0.721100994122155,0.6928299620224574,0.0],[-0.9671844011026475,-0.2540754499429512,0.0],[-0.25865414139742193,0.3195077593233224,0.0],[0.44987611830780366,0.8930909685895961,0.0],[-0.6168595696937054,-0.7870732312036134,0.0],[0.08090846865805429,-0.7072494211385727,0.0],[0.778676507009814,-0.6274256110735321,0.0],[-0.8361837424330437,0.5484494041301068,0.0],[-0.6650355375289796,-0.1605382513568792,0.0],[-0.49388733262491546,-0.8695259068438652,0.0],[0.4753060951867075,-0.8798205020788982,0.0],[0.0651436106782268,0.029387622570932637,0.0],[-0.3450188738302539,0.9385957472207634,0.0],[-0.4604371463733136,-0.887692308313866,0.0],[0.23510390384326643,-0.2608844303709549,0.0],[0.9306449540598465,0.3659234475719562,0.0],[0.7893406601963862,-0.6139554724576801,0.0],[0.0023180245221486873,0.002957205830235121,0.0],[-0.7847046111520888,0.6198698841181504,0.0],[0.5735995446841988,0.8191358631741622,0.0],[0.010094330008686148,0.8260221285680578,0.0],[-0.5534108846668265,0.8329083939619534,0.0],[-0.3342457746501859,-0.9424859479740253,0.0],[-0.33650858754233426,-0.000808616651246985,0.0],[-0.3387714004344827,0.9408687146715313,0.0],[0.926903716800737,0.3752992136695455,0.0],[0.8192983688709086,0.5388949071391894,0.0],[0.71169302094108,0.7024906006088334,0.0],[0.8151309668020815,-0.5792767101828831,0.0],[0.9044981568096389,-0.2343399093720221,0.0],[0.9938653468171962,0.11059689143883888,0.0],[-0.2984165298526775,-0.9544357362917034,0.0],[0.34278063325052155,-0.566363390022831,0.0],[0.9839777963537206,-0.17829104375395857,0.0],[-0.5076642504326659,0.8615549946652503,0.0],[-0.23165456841863827,-0.06873041649305689,0.0],[0.04435511359538943,-0.9990158276513641,0.0],[-0.9818283956192582,-0.18977091862483417,0.0],[-0.6275047778541891,-0.5758667619044103,0.0],[-0.27318116008912013,-0.9619626051839866,0.0],[-0.44097917393379343,-0.8975173358530013,0.0],[-0.44373600573835115,-0.8961522714857384,0.0],[-0.4464928375429089,-0.8947872071184755,0.0],[-0.6423861104759501,0.7663811617384527,0.0],[0.17091836547489297,0.29472392272073156,0.0],[0.984222841425736,-0.17693331629698963,0.0],[-0.3580872521099733,-0.9336881277366272,0.0],[-0.6575841546807589,-0.32193387629488446,0.0],[-0.9570810572515446,0.2898203751468584,0.0],[0.8927025042343348,-0.4506464677923787,0.0],[0.31450830863540047,0.2569810665760992,0.0],[-0.2636858869635339,0.9646086009445771,0.0],[-0.6759156248450373,0.7369790146878965,0.0],[0.11199787171982106,0.15045308307497662,0.0],[0.8999113682846794,-0.43607284853794326,0.0],[-0.6154092700024861,0.7882077330215729,0.0],[0.18770902770951942,0.32653684201283434,0.0],[0.990827325421525,-0.13513404899590423,0.0],[0.8266282080591291,-0.5627484390391085,0.0],[-0.05114796701651814,-0.09623917540422838,0.0],[-0.9289241420921653,0.3702700882306518,0.0],[-0.4541073591723802,-0.8909469716798452,0.0],[-0.6918959801126523,-0.26129521447962845,0.0],[-0.9296846010529245,0.36835654272058826,0.0],[0.6129615763085575,-0.7901127172557902,0.0],[0.7141289668309753,-0.10553440363730704,0.0],[0.815296357353393,0.5790439099811762,0.0],[-0.4575451584624788,0.8891863853925932,0.0],[-0.4410453416503695,-0.00811010344252272,0.0],[-0.42454552483826025,-0.9054065922776386,0.0],[0.994766048331882,-0.10217880937930222,0.0],[-0.0019239118310356296,-0.02477242033902474,0.0],[-0.9986138719939532,0.05263396870125275,0.0],[0.816659713025102,0.5771194963972003,0.0],[0.11217586291639753,0.6914155686568002,0.0],[-0.5923079871923069,0.8057116409164,0.0],[0.3971244522362094,-0.9177647680294175,0.0],[-0.07890008945928711,-0.8748326976754053,0.0],[-0.5549246311547836,-0.8319006273213931,0.0],[0.4328588821599313,-0.9014616953233536,0.0],[-0.06361338931332744,-0.036513497535787864,0.0],[-0.5600856607865862,0.8284347002517779,0.0],[-0.8329854223141496,0.5532949360080189,0.0],[-0.33840815974170857,0.7705126346540385,0.0],[0.15616910283073243,0.9877303333000582,0.0],[-0.5119371019177293,-0.8590229354796508,0.0],[0.07281845147683313,-0.05281648897139474,0.0],[0.6575740048713956,0.7533899575368613,0.0]]
        sv_edges = [[0,1],[1,2],[3,4],[4,5],[6,7],[7,8],[9,10],[10,11],[12,13],[13,14],[15,16],[16,17],[18,19],[19,20],[21,22],[22,23],[24,25],[25,26],[27,28],[28,29],[30,31],[31,32],[33,34],[34,35],[36,37],[37,38],[39,40],[40,41],[42,43],[43,44],[45,46],[46,47],[48,49],[49,50],[51,52],[52,53],[54,55],[55,56],[57,58],[58,59],[60,61],[61,62],[63,64],[64,65],[66,67],[67,68],[69,70],[70,71],[72,73],[73,74],[75,76],[76,77],[78,79],[79,80],[81,82],[82,83],[84,85],[85,86],[87,88],[88,89],[90,91],[91,92],[93,94],[94,95],[96,97],[97,98],[99,100],[100,101],[102,103],[103,104],[105,106],[106,107],[108,109],[109,110],[111,112],[112,113],[114,115],[115,116],[117,118],[118,119],[120,121],[121,122],[123,124],[124,125],[126,127],[127,128],[129,130],[130,131],[132,133],[133,134],[135,136],[136,137],[138,139],[139,140],[141,142],[142,143],[144,145],[145,146],[147,148],[148,149]]
        expected_points = [[-0.7997182304959439,0.6003755090045197,0.0],[-0.19766237037635295,-0.1571047955828267,0.0],[0.404393489743238,-0.9145851001701731,0.0],[-0.999900408118644,0.014112896377748907,0.0],[-0.9880298280393633,0.11558423566978356,0.0],[-0.9761592479600828,0.2170555749618182,0.0],[-0.3664489746746277,-0.9304381489169036,0.0],[-0.20949457886748124,0.03409033091160191,0.0],[-0.05254018306033478,0.9986188107401074,0.0],[0.3007765819551887,0.9536946302393412,0.0],[-0.34650456261641804,0.42119219812941183,0.0],[-0.9937857071880247,-0.11131023398051754,0.0],[0.4156821687110949,0.9095099420103339,0.0],[0.18140931605311897,-0.04454590184046181,0.0],[-0.05286353660485697,-0.9986017456912575,0.0],[0.35672372712476613,0.9342099242173653,0.0],[-0.2668731394326972,0.6946259328080246,0.0],[-0.8904700059901606,0.4550419413986839,0.0],[-0.9954551580793016,-0.09523144571680432,0.0],[-0.1569679281812693,0.3182843459402713,0.0],[0.681519301716763,0.7318001375973469,0.0],[0.766509618764487,0.6422328272063963,0.0],[-0.1003227902558877,0.19402312059281424,0.0],[-0.9671551992762624,-0.2541865860207678,0.0],[-0.27129636568262766,0.9624958607533842,0.0],[0.1993058156405517,0.1100258689350121,0.0],[0.669907996963731,-0.74244412288336,0.0],[0.9986057183994101,0.05278843793860495,0.0],[0.9975437310504423,-0.01551083582623871,0.0],[0.9964817437014744,-0.08381010959108237,0.0],[0.6011051906889029,-0.7991699129264425,0.0],[-0.08176893838578297,-0.07735794821257624,0.0],[-0.7646430674604688,0.64445401650129,0.0],[0.8654909293244962,-0.5009245963785566,0.0],[0.0001250654483755942,0.0002159621047279625,0.0],[-0.865240798427745,0.5013565205880125,0.0],[0.08345987434785694,-0.9965111386100207,0.0],[0.5361686525644019,-0.5726219375302932,0.0],[0.9888774307809469,-0.14873273645056573,0.0],[-0.85265158312796,0.5224799305134922,0.0],[-0.39492375570820565,-0.23777297259568342,0.0],[0.0628040717115487,-0.998025875704859,0.0],[-0.7989583617006947,-0.6013863452627952,0.0],[-0.01944389656626916,0.02422709866468492,0.0],[0.7600705685681564,0.6498405425921651,0.0],[0.986067876718799,-0.16634344743114893,0.0],[-0.0060330263665178685,-0.052640351010438416,0.0],[-0.9981339294518348,0.0610627454102721,0.0],[0.8359205354299549,-0.5488504882447474,0.0],[-0.05658854906820854,-0.43191626129376076,0.0],[-0.949097633566372,-0.31498203434277416,0.0],[-0.28579513859882766,-0.9582907381130619,0.0],[0.05964693008125718,-0.02200674044302886,0.0],[0.405088998761342,0.9142772572270041,0.0],[0.7494211287946771,-0.6620936276057275,0.0],[0.764217098789785,-0.017542970659616353,0.0],[0.779013068784893,0.6270076862864948,0.0],[0.47365213462380934,-0.8807120161359835,0.0],[-0.23084319896830222,-0.26347905259630533,0.0],[-0.9353385325604138,0.3537539109433729,0.0],[-0.7707364355568286,0.6371540998103635,0.0],[-0.7459187148394918,0.6649920309164105,0.0],[-0.721100994122155,0.6928299620224574,0.0],[-0.9671844011026475,-0.2540754499429512,0.0],[-0.25865414139742193,0.3195077593233224,0.0],[0.44987611830780366,0.8930909685895961,0.0],[-0.6168595696937054,-0.7870732312036134,0.0],[0.08090846865805429,-0.7072494211385727,0.0],[0.778676507009814,-0.6274256110735321,0.0],[-0.8361837424330437,0.5484494041301068,0.0],[-0.6650355375289796,-0.1605382513568792,0.0],[-0.49388733262491546,-0.8695259068438652,0.0],[0.4753060951867075,-0.8798205020788982,0.0],[0.0651436106782268,0.029387622570932637,0.0],[-0.3450188738302539,0.9385957472207634,0.0],[-0.4604371463733136,-0.887692308313866,0.0],[0.23510390384326643,-0.2608844303709549,0.0],[0.9306449540598465,0.3659234475719562,0.0],[0.7893406601963862,-0.6139554724576801,0.0],[0.0023180245221486873,0.002957205830235121,0.0],[-0.7847046111520888,0.6198698841181504,0.0],[0.5735995446841988,0.8191358631741622,0.0],[0.010094330008686148,0.8260221285680578,0.0],[-0.5534108846668265,0.8329083939619534,0.0],[-0.3342457746501859,-0.9424859479740253,0.0],[-0.33650858754233426,-0.000808616651246985,0.0],[-0.3387714004344827,0.9408687146715313,0.0],[0.926903716800737,0.3752992136695455,0.0],[0.8192983688709086,0.5388949071391894,0.0],[0.71169302094108,0.7024906006088334,0.0],[0.8151309668020815,-0.5792767101828831,0.0],[0.9044981568096389,-0.2343399093720221,0.0],[0.9938653468171962,0.11059689143883888,0.0],[-0.2984165298526775,-0.9544357362917034,0.0],[0.34278063325052155,-0.566363390022831,0.0],[0.9839777963537206,-0.17829104375395857,0.0],[-0.5076642504326659,0.8615549946652503,0.0],[-0.23165456841863827,-0.06873041649305689,0.0],[0.04435511359538943,-0.9990158276513641,0.0],[-0.9818283956192582,-0.18977091862483417,0.0],[-0.6275047778541891,-0.5758667619044103,0.0],[-0.27318116008912013,-0.9619626051839866,0.0],[-0.44097917393379343,-0.8975173358530013,0.0],[-0.44373600573835115,-0.8961522714857384,0.0],[-0.4464928375429089,-0.8947872071184755,0.0],[-0.6423861104759501,0.7663811617384527,0.0],[0.17091836547489297,0.29472392272073156,0.0],[0.984222841425736,-0.17693331629698963,0.0],[-0.3580872521099733,-0.9336881277366272,0.0],[-0.6575841546807589,-0.32193387629488446,0.0],[-0.9570810572515446,0.2898203751468584,0.0],[0.8927025042343348,-0.4506464677923787,0.0],[0.31450830863540047,0.2569810665760992,0.0],[-0.2636858869635339,0.9646086009445771,0.0],[-0.6759156248450373,0.7369790146878965,0.0],[0.11199787171982106,0.15045308307497662,0.0],[0.8999113682846794,-0.43607284853794326,0.0],[-0.6154092700024861,0.7882077330215729,0.0],[0.18770902770951942,0.32653684201283434,0.0],[0.990827325421525,-0.13513404899590423,0.0],[0.8266282080591291,-0.5627484390391085,0.0],[-0.05114796701651814,-0.09623917540422838,0.0],[-0.9289241420921653,0.3702700882306518,0.0],[-0.4541073591723802,-0.8909469716798452,0.0],[-0.6918959801126523,-0.26129521447962845,0.0],[-0.9296846010529245,0.36835654272058826,0.0],[0.6129615763085575,-0.7901127172557902,0.0],[0.7141289668309753,-0.10553440363730704,0.0],[0.815296357353393,0.5790439099811762,0.0],[-0.4575451584624788,0.8891863853925932,0.0],[-0.4410453416503695,-0.00811010344252272,0.0],[-0.42454552483826025,-0.9054065922776386,0.0],[0.994766048331882,-0.10217880937930222,0.0],[-0.0019239118310356296,-0.02477242033902474,0.0],[-0.9986138719939532,0.05263396870125275,0.0],[0.816659713025102,0.5771194963972003,0.0],[0.11217586291639753,0.6914155686568002,0.0],[-0.5923079871923069,0.8057116409164,0.0],[0.3971244522362094,-0.9177647680294175,0.0],[-0.07890008945928711,-0.8748326976754053,0.0],[-0.5549246311547836,-0.8319006273213931,0.0],[0.4328588821599313,-0.9014616953233536,0.0],[-0.06361338931332744,-0.036513497535787864,0.0],[-0.5600856607865862,0.8284347002517779,0.0],[-0.8329854223141496,0.5532949360080189,0.0],[-0.33840815974170857,0.7705126346540385,0.0],[0.15616910283073243,0.9877303333000582,0.0],[-0.5119371019177293,-0.8590229354796508,0.0],[0.07281845147683313,-0.05281648897139474,0.0],[0.6575740048713956,0.7533899575368613,0.0],[-0.33873245304154165,0.9246606237710551,0],[-0.07051282470250968,0.8881719218047408,0],[0.20731839637494223,0.8768088572805142,0],[0.40216637570664915,0.8544680489618259,0],[-0.16670900810521372,0.845922681577679,0],[0.3748693062867308,0.832369993914812,0],[-0.4989675927697246,0.8322430745520228,0],[-0.4564885183683995,0.8317239627986682,0],[-0.19909133048559216,0.8317004071696456,0],[-0.3385056653842487,0.8302821638296104,0],[-0.295921628065168,0.8297617693862482,0],[-0.20600717352821027,0.8286629777858096,0],[-0.19735616981590873,0.8285572589638093,0],[-0.15206771177379952,0.8280038153873591,0],[-0.08044619881798953,0.8271285710537108,0],[0.07312424142630256,0.8252518771374205,0],[0.14360161678910568,0.8243906146355413,0],[0.3617200082940244,0.8217251182176287,0],[0.3709004861796701,0.8216129290035508,0],[0.39402924380617355,0.8213302860765659,0],[-0.5423946689852238,0.797613660780116,0],[-0.28063901615191045,0.7958847683385898,0],[-0.48597758652626316,0.7884605042777914,0],[-0.45560235424664736,0.7835324001871192,0],[-0.3383620902214405,0.7705328683281834,0],[-0.3483741190984526,0.7661355981016261,0],[-0.33834761484131304,0.764508889334763,0],[-0.09101330878881163,0.7621907369527712,0],[-0.09655344022410538,0.7600622355548607,0],[-0.09222224759743902,0.7547614712396673,0],[-0.2608096966410072,0.7519290700811158,0],[-0.14852164998988357,0.7400962097619209,0],[-0.16020614579841663,0.7356070625362973,0],[-0.14465017667059496,0.7330832485819714,0],[-0.09700767487920951,0.7253536876750238,0],[-0.49838161161680344,0.7209346258423381,0],[-0.06378998383642272,0.7199644206036246,0],[-0.45442596517197603,0.7195577634849801,0],[-0.46425393078415955,0.7152413336079838,0],[-0.47798597605505283,0.7092102369580252,0],[0.0034135035682404976,0.7090612683861319,0],[-0.23967332161908816,0.7050760188601984,0],[-0.4889004304265242,0.7044166226130861,0],[0.6198580652897012,0.701390735895676,0],[-0.45927297165919184,0.6984531051164276,0],[-0.4539819562209275,0.6954115758439859,0],[-0.5154388178319941,0.692760996941964,0],[-0.03144757505803416,0.6803819080955541,0],[-0.45369198882095635,0.6796425080516787,0],[0.18596443664898415,0.6794440463089548,0],[-0.338113918712432,0.6672554452923084,0],[-0.5790087598261635,0.6648411600468905,0],[-0.10688262054869985,0.6646693959990785,0],[-0.46563031619058287,0.6638757011413572,0],[0.31096488854468674,0.6591638643287017,0],[-0.45327025642234414,0.6567077685444465,0],[0.35255174540659073,0.6524167765193721,0],[-0.4455609277536649,0.6522369205630698,0],[-0.37820627854885974,0.6518520843875985,0],[-0.7644296336182219,0.6442284125510899,0],[0.7510732737523264,0.6426196007783507,0],[-0.4529973534281613,0.6418667008367703,0],[-0.4183213959182177,0.6364399801089775,0],[0.7551806649913161,0.6363750085622476,0],[-0.08928811069603773,0.6327979082214922,0],[0.4789660936022257,0.6319072027742978,0],[-0.33802139620707006,0.6287518879549329,0],[-0.4385663773407151,0.6286619207322747,0],[-0.44417998760562366,0.6265051889948177,0],[-0.45265500639603906,0.6232491129335538,0],[0.7217621410047409,0.6190954206331237,0],[0.5599326232492694,0.6187711226189904,0],[-0.11559002619584204,0.6111599629634618,0],[-0.499142820646137,0.6053886381825823,0],[-0.4312355714791766,0.6039535782739239,0],[-0.7237779290717582,0.6012587268091347,0],[0.7784171488053302,0.6010478732805636,0],[0.6826345323682982,0.598863858216981,0],[0.694212998661171,0.5969853577802978,0],[-0.7476089155558762,0.5907921881461565,0],[-0.3379278386353556,0.589817584211651,0],[0.7780117985069069,0.5833897676458212,0],[0.79146773802409,0.5812066643211102,0],[0.8150505477983735,0.5773805687624535,0],[-0.7805886630523375,0.5763075253942799,0],[0.055570077649116374,0.5738846663376902,0],[-0.4516768495183966,0.5700547810307669,0],[-0.17518763930043246,0.5621304582539665,0],[0.3294035087169387,0.5581475932228925,0],[0.811185806575665,0.5512286831088701,0],[-0.1944955199742439,0.5462463350967911,0],[-0.6667607141591553,0.5409903608449039,0],[-0.4125326303443351,0.5409156690363365,0],[-0.6783796600526747,0.5365263976943098,0],[-0.15875304021873035,0.525699844595205,0],[0.2613038781985905,0.5245629760441983,0],[-0.22273518381933854,0.5230142514719355,0],[-0.38201114241038747,0.5181953523737605,0],[-0.7324568019924808,0.5157501266692944,0],[-0.022068460452169404,0.5110332055328907,0],[-0.13199928399411862,0.5103205032089673,0],[-0.617077965209689,0.4884746758822979,0],[0.463195923796957,0.4853995791430064,0],[-0.33767642854761104,0.4851924155481737,0],[-0.8201565448866908,0.48205615613323227,0],[-0.8267811480644874,0.4795110040390572,0],[-0.8272241918360479,0.4793407879894712,0],[-0.8263866401512991,0.47885575485311244,0],[-0.1372523182362439,0.47803914462714225,0],[-0.8182459405045158,0.47414140808056904,0],[-0.1379361213406692,0.47383698408603986,0],[-0.06823054542195053,0.4736631758401501,0],[-0.13444269832554331,0.47181105369618326,0],[0.14338652738915755,0.466409810520142,0],[-0.8146971259923501,0.4594403150591773,0],[-0.3014705292397454,0.4582406005151621,0],[-0.09719874215610795,0.4502122772767668,0],[-0.3375696831686933,0.4407699606915918,0],[-0.11734253489984167,0.4339050820222517,0],[-0.3328024693954374,0.4324645764867302,0],[-0.3375403595338502,0.4285668296684688,0],[0.02961062536962018,0.4174193813389313,0],[0.03627747070966201,0.41358695909537996,0],[0.03270021132184295,0.41182276615961316,0],[-0.14854444051328142,0.4086459079618158,0],[-0.37102615551293383,0.401018880914608,0],[0.28948404503959,0.39557904474337074,0],[-0.0017257913351980602,0.3948449336599506,0],[0.2098411684319723,0.38507877304793553,0],[-0.189887061482788,0.3751774245902879,0],[0.22746420293775882,0.3635106830353707,0],[0.06269766208640801,0.35748403246326516,0],[-0.44773585494863855,0.355734790073837,0],[-0.08188744490344294,0.3553117023192132,0],[-0.15786912192712063,0.35134314340575673,0],[0.19585312302967656,0.34716566327849285,0],[-0.43664170260821694,0.34703856848702036,0],[-0.5980186810190479,0.3466059731635411,0],[-0.9186471808957024,0.33913003486890375,0],[-0.44741300108220317,0.3381772835840346,0],[0.17267324304375428,0.335180131704322,0],[-0.13079739132201235,0.33119083978009084,0],[0.18814883803228688,0.32628401795918577,0],[-0.46286334691775877,0.3254666603820904,0],[0.13486634443727455,0.3156314637541584,0],[-0.1637966852752898,0.31491661558603407,0],[-0.2650120039460281,0.31436081866023685,0],[0.26909765722046636,0.31255725108194665,0],[0.33506504313999586,0.3087446651661217,0],[-0.4468683812415078,0.30855965349935816,0],[0.09669730610599,0.2958955456064724,0],[0.17552012621510515,0.2920552373396372,0],[-0.5036187455709984,0.29193818657057036,0],[-0.7731919270469486,0.2875035208210136,0],[-0.3372002020898341,0.2870091448850189,0],[0.26203833910452806,0.28380879121687913,0],[0.7710370183031833,0.2795503327286601,0],[0.2971055417982434,0.2782795802621639,0],[-0.24293032203798992,0.2758903701217939,0],[0.2864848595955347,0.2697557595161947,0],[-0.33715706357438413,0.269056904005045,0],[0.30550876694939877,0.2679952267703486,0],[-0.5344247315361775,0.266594852244249,0],[-0.04077511119074469,0.2641778974328536,0],[-0.33027286308907045,0.2636606798473657,0],[-0.32893540358629964,0.2626123037441179,0],[-0.32976303249546884,0.26194230547756486,0],[0.30081203592503786,0.26151981186000894,0],[-0.44594861874761293,0.2585409371867994,0],[-0.33712564505599946,0.2559819800246898,0],[0.31964617844316656,0.250693044019578,0],[0.25302614274714835,0.2471074043377014,0],[0.25606741472421013,0.24534368855078667,0],[-0.30611912572511485,0.24472761981524183,0],[0.25174028864402426,0.2418708751901435,0],[-0.00956970559102594,0.24094846974754244,0],[-0.6819982646176528,0.23903707855924647,0],[-0.3222894971947216,0.23675289614478517,0],[-0.026375688756149224,0.2322586645378994,0],[0.2793048312248617,0.2318676829062971,0],[-0.33706193204234647,0.22946759128515,0],[-0.3711040067260417,0.22847515520619674,0],[0.769690076878305,0.2208740863885881,0],[-0.38066536179070976,0.2207348609169898,0],[-0.36091424840797437,0.21770437160440773,0],[-0.36875507981734146,0.21383751771339699,0],[0.7612405216439314,0.21325951384197867,0],[0.1447930464754734,0.20877275479408589,0],[-0.6126943675724452,0.20220432905473754,0],[-0.3433220122836584,0.19910901708744946,0],[-0.3369801564430632,0.19543639570215368,0],[-0.3369728548853248,0.1923978274540413,0],[-0.7499823137821982,0.19135685482145554,0],[-0.42103296579670285,0.18805569225906899,0],[-0.1862236503525207,0.1770966684487295,0],[-0.3044402211475803,0.17659224893405023,0],[-0.44444023972409674,0.17651195365043515,0],[-0.644211485333465,0.1762759639206881,0],[0.1649667493638031,0.17222920055936675,0],[0.38521284000575995,0.17044876724981647,0],[-0.4576483226284749,0.1699981384559683,0],[0.23391851907565211,0.16929326659589872,0],[-0.4443060643509417,0.16921520044455646,0],[-0.4520459397750693,0.16294946581391528,0],[-0.17405593043816991,0.1558982088049094,0],[-0.1900264160598305,0.15372761396667267,0],[-0.4440048127355397,0.15283247208858491,0],[-0.18534727269696724,0.15005983780213564,0],[-0.29640806227077604,0.14952000940726107,0],[-0.19110796779556719,0.14708117729848444,0],[-0.5066371303119225,0.1458383839967137,0],[0.12084590441571996,0.14386657244591738,0],[0.1254014841698824,0.14047538081003821,0],[-0.6898838087565897,0.13870245586025862,0],[-0.48644137606524074,0.13510500262946645,0],[0.11715198463216747,0.13385459563246235,0],[-0.22405945370090158,0.13004307895741932,0],[-0.15872808871356076,0.12919422182023482,0],[0.19568635507868584,0.11658232285302277,0],[-0.19647954714328744,0.11407132602358958,0],[-0.44325679868117723,0.11215381533609207,0],[-0.260269043643618,0.11132032526715947,0],[-0.7294200546204357,0.10617694935422892,0],[-0.5985844869456022,0.10049281090578509,0],[-0.2817751988605523,0.10020022013511694,0],[0.18229690813870364,0.09812220156724814,0],[-0.6290969851382714,0.08544499694311919,0],[0.2131895341145788,0.08487627583137397,0],[-0.3885737910669567,0.08309158929281622,0],[0.6146199743217561,0.08112796804576171,0],[0.2111652372937276,0.07663250285593144,0],[0.04436815743254032,0.07544061765871299,0],[-0.12692883322338563,0.07379392998602151,0],[-0.33668308526996904,0.07180921005953715,0],[0.22232757734763675,0.06832321265130518,0],[-0.35222463140252774,0.06377320245309816,0],[-0.5758470551164587,0.06272757680412056,0],[-0.9944572948016397,0.06064137219829863,0],[-0.3671041946732739,0.05607948378416154,0],[-0.33664387697096626,0.05549254220834132,0],[0.5851024666598655,0.05452737224490702,0],[-0.2062233535003274,0.05419292194979224,0],[-0.9954230275638848,0.052386156688296874,0],[-0.5900471303268986,0.05123205547745879,0],[-0.7142749659058173,0.04343786923393962,0],[-0.8066714260808237,0.042624127629891866,0],[-0.8359903459566503,0.042479728853369515,0],[-0.8347304285946162,0.03990621374829243,0],[-0.8104105201997271,0.039548065969665556,0],[-0.8052880370947786,0.038960986380700245,0],[-0.8121105690004176,0.03814947722864415,0],[-0.8047661606752267,0.0375790850798184,0],[-0.7689807502590156,0.034799866354478334,0],[-0.7368501475390127,0.03230449263381139,0],[-0.7388032696833761,0.031341273611587755,0],[-0.711104720807792,0.030305013761525876,0],[-0.7105738705693831,0.028105947366671484,0],[-0.8276227356444054,0.025387995553343867,0],[-0.6242318607286808,0.023558167025730426,0],[-0.022098717714281492,0.02209642389762581,0],[-0.6300185312262279,0.018873628871593783,0],[-0.5520635651793938,0.017953327622363003,0],[-0.5487339962218039,0.017694741782865318,0],[-0.3365529217382745,0.017641210757905083,0],[-0.4415181552169554,0.01760254282514597,0],[-0.02862013210715101,0.016862544732987393,0],[0.7317626154376587,0.013788763282070756,0],[0.0725508531706288,0.012967971067665734,0],[-0.2557561001894693,0.01250334034244803,0],[-0.5453463946669375,0.012068179965153445,0],[-0.45536403366406913,0.01044330760042531,0],[-0.44136651866569865,0.009356212179715065,0],[-0.5434718214030558,0.008954648996345047,0],[-0.5415399755278998,0.008733243232949682,0],[-0.7924238427824571,0.004897276680508789,0],[-0.3365134453097829,0.0012129599020156952,0],[-0.4737066294171791,0.0009589720178135951,0],[-0.3226390833925641,0.00013542897547329496,0],[0.5245317413720973,-5.777089116955605e-05,0],[-0.441143481562165,-0.0027730382331660162,0],[0.7645042777526696,-0.00503271352544531,0],[-0.8126964449494455,-0.005100539428211981,0],[-0.2504225903652517,-0.005473153610713176,0],[-0.21705594692980953,-0.008064522527793352,0],[-0.2163630496752376,-0.008118335324747351,0],[-0.21641014438157952,-0.008407745412832745,0],[0.021908955607779265,-0.012399269597992459,0],[-0.14179461364550494,-0.013909577937945767,0],[-0.33647504149135454,-0.014768920604168432,0],[-0.30821898937942493,-0.018007301506514074,0],[-0.5107844732148774,-0.018212726360667265,0],[-0.07355208913349627,-0.019198410688546556,0],[-0.07356471955769635,-0.019208547464110694,0],[-0.07354540993141852,-0.01921004711646361,0],[-0.5232381125485275,-0.024652081850749647,0],[-0.24453842272463489,-0.025305629315668793,0],[0.7257308622205724,-0.027026833761533104,0],[0.09095026143546181,-0.027818039635342705,0],[-0.21962451886839804,-0.02816097195629372,0],[0.04205089433829178,-0.028187655858962508,0],[0.05027517322845097,-0.028826381798615137,0],[0.05693985341665165,-0.029343983909699473,0],[0.08808882547362741,-0.03176312079841949,0],[0.05592390341678457,-0.03209760836239848,0],[0.09289847765524174,-0.03213665501949021,0],[-0.16814051482612438,-0.03406147124778512,0],[-0.6955407716899292,-0.0341692107936056,0],[0.05393077369703225,-0.037499774546196095,0],[0.18274307192282987,-0.03911429690915671,0],[-0.11720732911826295,-0.03989884213167888,0],[-0.10157688982572263,-0.04169022175894341,0],[0.07840381426317322,-0.04511591142528235,0],[-0.05775950769141812,-0.04671206178572493,0],[0.28606788908673975,-0.04713885952599877,0],[-0.10949816620367579,-0.048047585498509265,0],[0.7634966351786522,-0.04892822583484394,0],[0.07305497107335478,-0.05249039779990545,0],[0.38807688741062396,-0.05506123110138507,0],[-0.12237666460687575,-0.05838345750474722,0],[0.04613859123724912,-0.058619656591531524,0],[0.45751285619334703,-0.06045386855918133,0],[0.06688473847775327,-0.06099733931786451,0],[0.10595008651274326,-0.061068186451996714,0],[0.08682281631471106,-0.06328241050335488,0],[0.10804626778236819,-0.06571479630312849,0],[0.11543635296525723,-0.0665617621364365,0],[-0.942490528065803,-0.06911093948975129,0],[0.5817654999964943,-0.0701037585650336,0],[0.17434536481639634,-0.07331322968749754,0],[-0.4397145859842862,-0.08047953640481509,0],[-0.2281489564543917,-0.08054601529279327,0],[0.7178035986967214,-0.08066894775442625,0],[-0.07680810058205736,-0.0826016555508321,0],[0.7626879916722912,-0.0841548249312524,0],[-0.7584404432062164,-0.08508897738920061,0],[0.4293795210495627,-0.08580707538348199,0],[0.117959232167113,-0.08768888729971262,0],[0.3090392869623904,-0.08875028449656122,0],[0.8337573588489499,-0.08967431775690861,0],[-0.7692516759691211,-0.0938410970048843,0],[0.9239283445765657,-0.09667730833608155,0],[0.16846272388761413,-0.09726977391233876,0],[0.9398439264240935,-0.09791336746131905,0],[0.41290443025361345,-0.10065410230951331,0],[0.9380595607260248,-0.1048006086884006,0],[-0.6784631892279595,-0.10491371645689604,0],[0.45595197026548684,-0.10558771196294946,0],[-0.23330897945235807,-0.11225579387219661,0],[-0.019754639287517594,-0.11292370002195612,0],[0.022874461541025244,-0.12167460655035332,0],[0.16228378438870383,-0.12243296832796462,0],[0.6267537921592827,-0.12516303536324797,0],[-0.21311752334894707,-0.13120917618178884,0],[0.015594986907566177,-0.1317108675873273,0],[0.018585238888172463,-0.13330008849484984,0],[-0.2162477931905888,-0.13372143090398195,0],[0.7098213295931431,-0.13468327286016007,0],[-0.7391108985171351,-0.13627259471804362,0],[-0.7465861688236987,-0.14013780404350995,0],[0.7613924089926248,-0.14059375171895727,0],[-0.2100637208284046,-0.14150195979970104,0],[0.9278423379001173,-0.14423673904499182,0],[0.0018814409039556396,-0.15061782644973992,0],[-0.2398896095109512,-0.15269559853066253,0],[-0.004342691376112364,-0.1591990798281497,0],[0.34793335593794994,-0.15920475321184532,0],[0.92395907089455,-0.15922525661482373,0],[0.008160074381004188,-0.16155638842506734,0],[0.9610100900217737,-0.16347161460845824,0],[0.15218534422255622,-0.16355798876879415,0],[0.971814067227917,-0.16470984114184462,0],[-0.4381421830367519,-0.16599028236783087,0],[0.9684984950464508,-0.1678143429998839,0],[0.005277892923445942,-0.169368234814837,0],[-0.3360989978444979,-0.17126077154846314,0],[0.3195618919938716,-0.1847725570149194,0],[0.024007828906388515,-0.18916616142414056,0],[0.1433349456211802,-0.19960046834016434,0],[-0.30116166092580454,-0.20187059247453712,0],[0.6982900188247877,-0.2127132115473914,0],[0.17636367796880945,-0.21715415690428855,0],[0.283394219598046,-0.21736615012851443,0],[-0.7075018720097378,-0.21997163382710963,0],[0.3820132893342885,-0.2209386798627214,0],[0.9085627474071009,-0.22393456289720598,0],[0.9069394642665092,-0.22491702403907912,0],[0.9067556106267559,-0.22562665680596483,0],[-0.3359583369251686,-0.22979729464989285,0],[-0.9417278058120848,-0.23346733134286712,0],[-0.9370153895728417,-0.23860231030151297,0],[-0.25453873656958387,-0.24271856462865474,0],[0.24383722374087374,-0.253014134705859,0],[-0.3789512975540997,-0.26430207341505124,0],[0.6885349889519373,-0.2787234334693855,0],[-0.09578925829569793,-0.2852770817105412,0],[0.7580577429923446,-0.2858604129711159,0],[0.20736734921955782,-0.28588006641532765,0],[0.1207794528797379,-0.2914557533924361,0],[0.7740025299957878,-0.30537456651876493,0],[-0.43549985095059385,-0.3096861385246025,0],[0.7573788270252813,-0.31543574549251374,0],[0.7466090149362857,-0.32195396872672777,0],[-0.8556900973230431,-0.3272200476377888,0],[0.7570507678325555,-0.32972685099344545,0],[0.1577805703258243,-0.33056669367151503,0],[0.7946275273766785,-0.330616632513353,0],[-0.15386353436539965,-0.3309236249724936,0],[-0.055970451426876074,-0.3353753598824514,0],[-0.3357028124313862,-0.33613468747079417,0],[0.10937629784613526,-0.33789411180534307,0],[-0.13859822553646148,-0.34429809511950127,0],[0.7785141200336447,-0.3457042554930063,0],[0.45376442056070815,-0.35091191134576927,0],[-0.6571772097738546,-0.3532286916376699,0],[-0.6412409445500065,-0.35531661980199136,0],[-0.14661579939521927,-0.35535197856321293,0],[-0.6172579085070553,-0.35845881502255544,0],[0.12310633819166225,-0.36181442787289164,0],[0.6758143849846039,-0.36480106745003255,0],[0.7562044724228381,-0.36659366421923084,0],[0.46444557984224305,-0.37026024445323497,0],[-0.5132497146725999,-0.3720856989992478,0],[0.6536245711105451,-0.37823102882369136,0],[-0.4341618868659955,-0.38244758121022615,0],[0.09793167701881139,-0.38450133606152104,0],[0.6722368411779591,-0.3890095489913887,0],[0.5079333458511579,-0.39337256656222414,0],[-0.3355604817644832,-0.3953660817410184,0],[-0.07864301946427656,-0.39682694486344106,0],[0.8511136782830733,-0.3997476545749859,0],[-0.2969994058293295,-0.4004182539633447,0],[-0.28055133661225673,-0.40257323736031053,0],[0.8599413343829109,-0.40631899678512623,0],[0.8591087686410559,-0.4095325090756642,0],[0.7093257691031182,-0.41048805516992826,0],[-0.18951710959235185,-0.4145003058322675,0],[-0.28377493483582333,-0.4223831463352396,0],[-0.1266220126785484,-0.4227406584117131,0],[0.5721854930547855,-0.4275204766218029,0],[-0.08997534553477259,-0.42754200975616136,0],[0.560511249754588,-0.43458608958630096,0],[-0.03542863204431651,-0.4346885787777077,0],[0.7545950588097925,-0.4367038767926217,0],[-0.5976222174775622,-0.4398003783458214,0],[-0.5992432523665644,-0.441101369238223,0],[0.029752695426066227,-0.4432284673589807,0],[0.032382445922496665,-0.4435730104529559,0],[0.03100975861415038,-0.4448100491849701,0],[-0.5954258934055165,-0.44889873059206314,0],[0.08183535665509085,-0.4500522026398271,0],[0.6625967009904411,-0.45424233673592035,0],[-0.6182356011094148,-0.45634402250090156,0],[0.5151216987150324,-0.4620572491322393,0],[0.1812286581105828,-0.46307445537839664,0],[0.648065679943208,-0.46784830860783777,0],[0.003953486468775889,-0.46919262807547163,0],[0.29109796473247435,-0.4714859147841848,0],[0.6596742583326549,-0.4740178874937559,0],[0.29396672615804975,-0.47784510484746134,0],[0.29755942509005173,-0.4783158109495931,0],[-0.10949812008846047,-0.48045641440461717,0],[0.8393523980851669,-0.4857875558619164,0],[0.6275053048452367,-0.48709980403611924,0],[-0.24361515363663808,-0.48908564597791726,0],[0.07082257668506113,-0.4949007925137431,0],[0.4543229090260327,-0.498854556298616,0],[0.6546021411288344,-0.5083398307991647,0],[0.5417595651090905,-0.5103102803509942,0],[-0.6867381213252515,-0.5113219600591881,0],[0.5952343238728535,-0.5173164046426556,0],[0.06417857116482764,-0.5219579279062503,0],[0.75260473798171,-0.5234073911751252,0],[0.6521731380348883,-0.5247763804940916,0],[0.6802666514842336,-0.5284571198104692,0],[0.7522722573114446,-0.5378911076541634,0],[0.7888821360456999,-0.5426876390765485,0],[0.8234376025044279,-0.547215006727438,0],[0.562531448291118,-0.5479374053176652,0],[0.36699347703193846,-0.5517090284055387,0],[-0.08820041961406484,-0.5522399109036373,0],[0.8202863880997405,-0.5593779696149367,0],[0.33580399219174906,-0.5705858688804267,0],[-0.3093847585279148,-0.5797626462044897,0],[0.7512122384213843,-0.5840682675287437,0],[0.15688632153176083,-0.6031824254629784,0],[0.2668823754118603,-0.6122993621263579,0],[-0.3350324141124346,-0.6151232428675388,0],[-0.1597832685590969,-0.6167489627135515,0],[-0.16430118759540965,-0.6208204222528373,0],[-0.16243130006517334,-0.6239261705560668,0],[0.7501420159455139,-0.6306899219473858,0],[-0.5500326778575697,-0.6369417748319615,0],[0.6345672746081308,-0.6439115309832824,0],[0.6166453565105884,-0.6459617764912291,0],[0.4569512983255841,-0.6467962911450583,0],[0.20798102562408308,-0.6479482800035372,0],[0.19750587343687706,-0.6542881659555254,0],[0.47183434691442916,-0.6625279788536012,0],[0.43574017500396955,-0.6666571079691179,0],[-0.5423860879909262,-0.6686180511041031,0],[0.6305437403956103,-0.6711379351810182,0],[0.3819211756204717,-0.672813935666223,0],[-0.5350520220018129,-0.6766097633433142,0],[0.30656533522367235,-0.6814345516309991,0],[0.2531730227762229,-0.6875425668645514,0],[0.22637368258222135,-0.6906083786013109,0],[0.11678040428730528,-0.7031457155817825,0],[0.39595081364365853,-0.7039134666042212,0],[0.016880770855645776,-0.7145741114473378,0],[-0.03816846231404576,-0.7208716763847396,0],[-0.09981689443640877,-0.7279241816100683,0],[-0.33474292378202375,-0.7355956348761717,0],[-0.2052515007472786,-0.7399857716025271,0],[-0.42753814097275117,-0.7426615157420461,0],[0.34620430325249096,-0.75049309916467,0],[-0.3098062398121541,-0.7519467058583135,0],[-0.3346967905089063,-0.754794154411453,0],[-0.3379271685172999,-0.755163705704157,0],[0.3359655826224896,-0.7600800197563751,0],[-0.4419085859173272,-0.7624741456285925,0],[-0.42712061818624547,-0.7653673272432544,0],[-0.43978299653210295,-0.7668158878371717,0],[-0.4455350021229001,-0.7674739102408022,0],[-0.4510827504015036,-0.7681085658577689,0],[-0.44824118164548554,-0.7712049382983415,0],[0.002738465208178604,-0.7721674228165748,0],[-0.49845096676802403,-0.7735274318684455,0],[-0.33464983400798326,-0.7743352630161026,0],[-0.5165610513613464,-0.7755992033933409,0],[-0.34214620207731167,-0.7810908420870311,0],[-0.019038836295378957,-0.7853477189504334,0],[0.36763187467294467,-0.7878239640447721,0],[0.3047911856828669,-0.7892698451998704,0],[-0.42659877979790256,-0.7937460516213206,0],[-0.42657922801170906,-0.7948093209473736,0],[-0.4255053547248251,-0.7959794885328424,0],[-0.0529036258904733,-0.8058437365723248,0],[-0.4818361608104303,-0.8175225638700198,0],[-0.00889414386164844,-0.819540212861962,0],[-0.39525045491228483,-0.8289473474922384,0],[-0.5018137012169626,-0.8366906370392454,0],[-0.4961120723860817,-0.8372048599598285,0],[-0.40504429989395635,-0.8377733674488633,0],[-0.47363764552394994,-0.8392318009905881,0],[-0.5002968150569265,-0.8429743930555438,0],[-0.42568284175196675,-0.8435567860050552,0],[-0.4127556394121366,-0.8447226745501234,0],[-0.40112875243447593,-0.8457712892453764,0],[-0.3778873504675394,-0.8478674028214044,0],[-0.3533723071351513,-0.8500783845521549,0],[-0.33446372845471023,-0.8517837261030583,0],[-0.42545102477234603,-0.8561635054756632,0],[-0.24933213372082533,-0.8594616403581647,0],[-0.019254612333095656,-0.8617323208167273,0],[-0.15547988485130934,-0.8679260597282187,0],[-0.35675896939057977,-0.8708903674222281,0],[-0.023694078261815285,-0.8798116602058926,0],[-0.007489622524223914,-0.8812731202207414,0],[0.009886571261279816,-0.882840257854921,0],[-0.4566316253045555,-0.8842628477739981,0],[-0.33435916355870415,-0.8952987655315837,0],[0.18772916387461627,-0.8988796634511558,0],[-0.3647599269591323,-0.9200584802430385,0],[0.025851834329794054,-0.9366508749664926,0],[-0.2803352956650457,-0.943492414364526,0],[-0.28663332925068846,-0.9473041789996528,0],[-0.28314433889991264,-0.9511060273781887,0]]
        expected_edges = [[0,234],[1,510],[1,544],[2,682],[3,392],[4,387],[4,5],[6,712],[7,434],[7,391],[8,151],[9,152],[10,270],[10,275],[11,476],[12,153],[13,458],[13,478],[14,706],[15,152],[16,191],[16,200],[17,256],[18,476],[19,295],[19,291],[20,193],[21,213],[22,328],[22,354],[23,539],[24,158],[25,368],[25,377],[26,650],[27,28],[28,29],[30,647],[31,482],[31,464],[32,209],[33,611],[34,436],[34,415],[35,256],[36,711],[37,644],[37,627],[38,520],[39,255],[40,521],[40,542],[41,713],[42,618],[43,409],[43,381],[44,210],[45,520],[46,469],[46,462],[47,387],[48,626],[49,591],[49,589],[50,552],[51,716],[52,451],[52,417],[53,155],[54,640],[55,465],[55,430],[56,226],[57,681],[58,556],[58,540],[59,288],[60,209],[61,209],[61,62],[63,538],[64,296],[64,279],[65,153],[66,678],[67,658],[67,656],[68,640],[69,254],[70,495],[70,566],[71,694],[72,657],[73,417],[73,381],[74,150],[75,709],[76,546],[76,541],[77,332],[78,633],[79,436],[79,409],[80,229],[81,169],[82,165],[82,164],[83,156],[84,710],[85,438],[85,425],[86,150],[87,88],[88,239],[89,210],[90,630],[91,582],[91,536],[92,492],[93,715],[94,631],[94,628],[95,534],[96,156],[97,445],[97,480],[98,713],[99,538],[100,618],[100,649],[101,716],[102,103],[103,104],[105,196],[106,294],[106,301],[107,522],[108,712],[109,564],[109,532],[110,396],[111,583],[112,320],[112,311],[113,154],[114,201],[115,325],[115,361],[116,582],[117,185],[118,290],[118,292],[119,494],[120,630],[121,498],[121,482],[122,303],[123,709],[124,563],[124,532],[125,288],[126,650],[127,506],[127,481],[128,233],[129,157],[130,429],[130,479],[131,701],[132,492],[133,449],[133,443],[134,392],[135,233],[136,199],[136,190],[137,170],[138,711],[139,706],[139,704],[140,690],[141,681],[142,462],[142,443],[143,170],[144,234],[145,175],[145,174],[146,151],[147,694],[148,471],[148,466],[149,193],[150,160],[150,159],[151,164],[151,154],[152,166],[152,165],[153,169],[153,155],[154,163],[154,158],[155,168],[155,167],[156,157],[156,172],[157,159],[157,173],[158,162],[158,161],[159,160],[159,174],[160,161],[160,171],[161,162],[161,171],[162,163],[162,181],[163,164],[163,178],[164,177],[165,166],[165,177],[166,167],[166,190],[167,168],[167,199],[168,169],[168,204],[169,206],[170,172],[170,185],[171,180],[171,174],[172,173],[172,188],[173,175],[173,187],[174,176],[175,176],[175,187],[176,180],[176,200],[177,179],[177,178],[178,179],[178,181],[179,186],[179,184],[180,182],[180,191],[181,183],[181,182],[182,183],[182,191],[183,184],[183,202],[184,186],[184,202],[185,189],[185,192],[186,190],[186,197],[187,195],[187,188],[188,194],[188,189],[189,194],[189,192],[190,197],[191,237],[192,203],[192,196],[193,221],[193,215],[194,195],[194,198],[195,208],[195,198],[196,203],[196,201],[197,235],[197,214],[198,207],[198,205],[199,204],[199,235],[200,216],[200,208],[201,223],[201,225],[202,214],[202,222],[203,205],[203,211],[204,206],[204,245],[205,207],[205,211],[206,215],[206,238],[207,212],[207,217],[208,216],[208,212],[209,225],[210,213],[210,220],[211,218],[211,219],[212,230],[212,217],[213,226],[213,220],[214,249],[214,222],[215,221],[215,238],[216,240],[216,230],[217,224],[217,218],[218,224],[218,219],[219,236],[219,223],[220,228],[220,227],[221,227],[221,252],[222,250],[222,237],[223,236],[223,241],[224,247],[224,242],[225,241],[225,229],[226,232],[226,231],[227,228],[227,252],[228,231],[228,298],[229,243],[229,234],[230,246],[230,253],[231,232],[231,306],[232,233],[232,239],[233,239],[234,248],[235,263],[235,249],[236,242],[236,282],[237,244],[237,240],[238,276],[238,245],[239,306],[240,244],[240,246],[241,251],[241,243],[242,247],[242,275],[243,251],[243,248],[244,250],[244,258],[245,278],[245,263],[246,260],[246,265],[247,253],[247,267],[248,287],[248,254],[249,271],[249,261],[250,261],[250,258],[251,282],[251,293],[252,298],[252,276],[253,265],[253,267],[254,259],[254,255],[255,257],[255,256],[256,257],[257,259],[257,264],[258,262],[258,260],[259,287],[259,264],[260,262],[260,274],[261,271],[261,266],[262,266],[262,268],[263,278],[263,272],[264,326],[264,303],[265,279],[265,269],[266,277],[266,268],[267,269],[267,270],[268,283],[268,274],[269,296],[269,270],[270,304],[271,272],[271,273],[272,290],[272,273],[273,281],[273,277],[274,284],[274,279],[275,304],[275,286],[276,297],[276,280],[277,281],[277,283],[278,280],[278,285],[279,284],[280,297],[280,285],[281,294],[281,300],[282,286],[282,289],[283,313],[283,291],[284,291],[284,295],[285,292],[285,290],[286,310],[286,289],[287,302],[287,312],[288,342],[288,395],[289,299],[289,293],[290,294],[291,313],[292,305],[292,301],[293,299],[293,302],[294,300],[295,344],[295,308],[296,308],[296,315],[297,307],[297,305],[298,311],[298,307],[299,331],[299,318],[300,337],[300,325],[301,321],[301,337],[302,318],[302,312],[303,326],[303,342],[304,314],[304,310],[305,309],[305,321],[306,332],[306,336],[307,311],[307,309],[308,344],[308,323],[309,317],[309,322],[310,314],[310,319],[311,317],[312,350],[312,338],[313,325],[313,328],[314,315],[314,316],[315,323],[315,316],[316,327],[316,319],[317,320],[317,329],[318,333],[318,346],[319,330],[319,331],[320,379],[320,349],[321,322],[321,324],[322,329],[322,324],[323,355],[323,327],[324,351],[324,348],[325,328],[326,338],[326,347],[327,345],[327,330],[328,381],[329,349],[329,351],[330,340],[330,334],[331,334],[331,333],[332,430],[332,336],[333,335],[333,343],[334,339],[334,335],[335,339],[335,343],[336,416],[336,379],[337,348],[337,361],[338,360],[338,347],[339,340],[339,341],[340,345],[340,341],[341,358],[341,383],[342,363],[342,372],[343,352],[343,346],[344,354],[344,355],[345,366],[345,358],[346,352],[346,350],[347,373],[347,363],[348,368],[348,362],[349,390],[349,428],[350,353],[350,360],[351,377],[351,368],[352,356],[352,353],[353,356],[353,364],[354,367],[354,357],[355,357],[355,359],[356,378],[356,370],[357,367],[357,359],[358,371],[358,374],[359,369],[359,366],[360,364],[360,373],[361,362],[361,365],[362,375],[362,365],[363,376],[363,372],[364,370],[364,386],[365,417],[365,381],[366,369],[366,371],[367,409],[367,382],[368,375],[369,382],[369,391],[370,378],[370,414],[371,391],[371,374],[372,394],[372,395],[373,386],[373,376],[374,418],[374,383],[375,380],[375,447],[376,393],[376,394],[377,384],[377,380],[378,385],[378,388],[379,416],[379,390],[380,384],[380,458],[382,415],[382,441],[383,389],[383,385],[384,467],[384,463],[385,389],[385,388],[386,412],[386,393],[387,396],[387,392],[388,413],[388,414],[389,418],[389,413],[390,446],[390,428],[391,437],[392,397],[393,411],[393,408],[394,405],[394,403],[395,399],[395,398],[396,398],[396,397],[397,400],[397,407],[398,399],[398,400],[399,402],[399,401],[400,401],[400,407],[401,402],[401,424],[402,403],[402,404],[403,405],[403,404],[404,406],[404,424],[405,408],[405,406],[406,410],[406,456],[407,431],[407,476],[408,411],[408,410],[409,415],[410,422],[410,456],[411,412],[411,419],[412,420],[412,419],[413,427],[413,425],[414,421],[414,420],[415,441],[416,430],[416,446],[417,447],[418,433],[418,432],[419,423],[419,422],[420,421],[420,426],[421,425],[421,429],[422,423],[422,444],[423,426],[423,440],[424,484],[424,431],[425,427],[426,429],[426,440],[427,432],[427,439],[428,477],[428,470],[429,438],[430,490],[431,489],[431,476],[432,433],[432,445],[433,434],[433,435],[434,437],[434,435],[435,455],[435,448],[436,450],[436,449],[437,442],[437,459],[438,439],[438,524],[439,445],[439,497],[440,479],[440,444],[441,443],[441,442],[442,443],[442,460],[444,521],[444,495],[445,448],[446,465],[446,481],[447,454],[447,452],[448,455],[448,480],[449,450],[449,457],[450,451],[450,453],[451,452],[451,453],[452,454],[452,461],[453,461],[453,457],[454,458],[454,472],[455,459],[455,468],[456,495],[456,484],[457,466],[457,469],[458,463],[459,460],[459,464],[460,462],[460,464],[461,472],[461,466],[462,498],[463,467],[463,487],[464,468],[465,488],[465,483],[466,473],[467,470],[467,485],[468,482],[468,502],[469,471],[469,499],[470,477],[470,485],[471,473],[471,499],[472,475],[472,474],[473,474],[473,486],[474,475],[474,486],[475,478],[475,491],[477,481],[477,501],[478,487],[478,491],[479,524],[479,521],[480,502],[480,497],[481,483],[482,514],[483,488],[483,509],[484,507],[484,489],[485,496],[485,493],[486,500],[486,519],[487,493],[487,515],[488,490],[488,511],[489,508],[489,538],[490,492],[490,494],[491,525],[491,500],[492,494],[493,496],[493,515],[494,511],[495,507],[496,501],[496,543],[497,505],[497,513],[498,503],[498,512],[499,504],[499,503],[500,531],[500,519],[501,506],[501,529],[502,510],[502,505],[503,504],[503,512],[504,527],[504,517],[505,510],[505,513],[506,509],[506,529],[507,532],[507,508],[508,532],[508,539],[509,516],[509,545],[510,556],[511,518],[511,516],[512,517],[512,514],[513,540],[513,528],[514,523],[514,544],[515,533],[515,525],[516,518],[516,535],[517,526],[517,523],[518,520],[518,522],[519,530],[519,527],[520,522],[521,549],[522,534],[523,526],[523,557],[524,528],[524,537],[525,533],[525,531],[526,547],[526,559],[527,530],[527,547],[528,540],[528,537],[529,545],[529,543],[530,541],[530,546],[531,562],[531,541],[533,572],[533,562],[534,536],[534,535],[535,536],[535,548],[536,555],[537,558],[537,542],[538,539],[539,552],[540,581],[541,570],[542,558],[542,549],[543,551],[543,568],[544,557],[544,560],[545,548],[545,550],[546,606],[546,554],[547,554],[547,559],[548,555],[548,550],[549,573],[549,571],[550,553],[550,551],[551,553],[551,568],[552,563],[552,618],[553,561],[553,569],[554,606],[554,567],[555,579],[555,561],[556,560],[556,565],[557,595],[557,578],[558,580],[558,577],[559,567],[559,574],[560,578],[560,565],[561,579],[561,569],[562,576],[562,570],[563,564],[563,601],[564,566],[564,594],[565,587],[565,585],[566,571],[566,593],[567,603],[567,574],[568,575],[568,572],[569,592],[569,584],[570,576],[570,602],[571,573],[571,593],[572,575],[572,588],[573,577],[573,663],[574,599],[574,596],[575,584],[575,600],[576,588],[576,590],[577,580],[577,636],[578,591],[578,589],[579,582],[579,583],[580,581],[580,586],[581,585],[581,586],[582,583],[583,611],[584,592],[584,600],[585,587],[585,613],[586,613],[586,632],[587,589],[587,610],[588,604],[588,590],[589,610],[590,612],[590,602],[591,595],[591,605],[592,611],[592,621],[593,598],[593,594],[594,598],[594,601],[595,596],[595,597],[596,599],[596,597],[597,614],[597,605],[598,669],[598,641],[599,603],[599,614],[600,607],[600,604],[601,641],[601,618],[602,617],[602,615],[603,608],[603,635],[604,607],[604,612],[605,620],[605,629],[606,609],[606,608],[607,621],[607,616],[608,609],[608,631],[609,615],[609,628],[610,629],[610,637],[611,626],[612,616],[612,619],[613,638],[613,632],[614,634],[614,620],[615,617],[615,628],[616,623],[616,622],[617,619],[617,627],[619,622],[619,627],[620,634],[620,658],[621,625],[621,624],[622,623],[622,642],[623,624],[623,633],[624,625],[624,633],[625,626],[625,630],[626,630],[627,643],[628,644],[629,659],[629,637],[631,651],[631,635],[632,661],[632,636],[633,640],[634,645],[634,646],[635,653],[635,645],[636,661],[636,663],[637,639],[637,638],[638,639],[638,665],[639,660],[639,662],[640,642],[641,652],[641,649],[642,650],[642,643],[643,650],[643,647],[644,647],[644,648],[645,654],[645,646],[646,655],[646,656],[647,648],[648,657],[648,651],[649,652],[649,678],[651,657],[651,653],[652,673],[652,676],[653,664],[653,654],[654,668],[654,655],[655,682],[655,656],[656,675],[657,664],[658,675],[658,659],[659,680],[659,660],[660,686],[660,662],[661,666],[661,667],[662,702],[662,665],[663,670],[663,669],[664,681],[664,668],[665,677],[665,666],[666,677],[666,667],[667,679],[667,670],[668,681],[668,682],[669,671],[669,672],[670,683],[670,671],[671,683],[671,672],[672,674],[672,673],[673,674],[673,676],[674,684],[674,687],[675,688],[675,680],[676,687],[676,678],[677,700],[677,679],[678,690],[679,699],[679,689],[680,688],[680,686],[682,711],[683,685],[683,684],[684,685],[684,695],[685,689],[685,692],[686,703],[686,704],[687,693],[687,691],[688,708],[688,703],[689,698],[689,692],[690,691],[690,694],[691,693],[691,694],[692,697],[692,696],[693,695],[693,709],[695,696],[695,701],[696,697],[696,701],[697,698],[697,712],[698,699],[698,705],[699,700],[699,705],[700,702],[700,710],[701,709],[702,704],[702,714],[703,707],[703,706],[704,714],[705,710],[705,712],[706,707],[707,708],[707,713],[708,711],[708,713],[710,715],[714,716],[714,715],[715,716]]

        result_points, result_edges = intersect_sv_edges(sv_points, sv_edges, accuracy=6)
        self.assert_sverchok_data_equal(expected_points, result_points, precision=5)
        self.assert_sverchok_data_equal(expected_edges, result_edges)

    def test_random_edges_above_xy(self):
        sv_points = [[-0.7997182011604309,0.6003755331039429,1.0],[-0.1976623684167862,-0.15710479021072388,1.0],[0.4043934941291809,-0.9145851135253906,1.0],[-0.9999004006385803,0.014112896285951138,1.0],[-0.9880298376083374,0.11558423936367035,1.0],[-0.9761592745780945,0.2170555740594864,1.0],[-0.3664489686489105,-0.9304381608963013,1.0],[-0.20949457585811615,0.03409033268690109,1.0],[-0.05254018306732178,0.9986187815666199,1.0],[0.30077657103538513,0.9536946415901184,1.0],[-0.3465045690536499,0.4211921989917755,1.0],[-0.9937856793403625,-0.11131023615598679,1.0],[0.4156821668148041,0.9095099568367004,1.0],[0.1814093142747879,-0.04454590007662773,1.0],[-0.05286353826522827,-0.9986017346382141,1.0],[0.35672372579574585,0.934209942817688,1.0],[-0.26687315106391907,0.6946259140968323,1.0],[-0.890470027923584,0.45504194498062134,1.0],[-0.9954551458358765,-0.09523144364356995,1.0],[-0.1569679230451584,0.3182843327522278,1.0],[0.6815193295478821,0.7318001389503479,1.0],[0.7665095925331116,0.6422328352928162,1.0],[-0.10032279044389725,0.19402311742305756,1.0],[-0.9671552181243896,-0.25418660044670105,1.0],[-0.27129635214805603,0.9624958634376526,1.0],[0.19930581748485565,0.11002586781978607,1.0],[0.6699079871177673,-0.7424440979957581,1.0],[0.9986057281494141,0.05278843641281128,1.0],[0.9975437521934509,-0.01551083568483591,1.0],[0.996481716632843,-0.08381010591983795,1.0]]
        sv_edges = [[0,1],[1,2],[3,4],[4,5],[6,7],[7,8],[9,10],[10,11],[12,13],[13,14],[15,16],[16,17],[18,19],[19,20],[21,22],[22,23],[24,25],[25,26],[27,28],[28,29]]

        expected_points = [[-0.7997182011604309,0.6003755331039429,1.0],[-0.1976623684167862,-0.15710479021072388,1.0],[0.4043934941291809,-0.9145851135253906,1.0],[-0.9999004006385803,0.014112896285951138,1.0],[-0.9880298376083374,0.11558423936367035,1.0],[-0.9761592745780945,0.2170555740594864,1.0],[-0.3664489686489105,-0.9304381608963013,1.0],[-0.20949457585811615,0.03409033268690109,1.0],[-0.05254018306732178,0.9986187815666199,1.0],[0.30077657103538513,0.9536946415901184,1.0],[-0.3465045690536499,0.4211921989917755,1.0],[-0.9937856793403625,-0.11131023615598679,1.0],[0.4156821668148041,0.9095099568367004,1.0],[0.1814093142747879,-0.04454590007662773,1.0],[-0.05286353826522827,-0.9986017346382141,1.0],[0.35672372579574585,0.934209942817688,1.0],[-0.26687315106391907,0.6946259140968323,1.0],[-0.890470027923584,0.45504194498062134,1.0],[-0.9954551458358765,-0.09523144364356995,1.0],[-0.1569679230451584,0.3182843327522278,1.0],[0.6815193295478821,0.7318001389503479,1.0],[0.7665095925331116,0.6422328352928162,1.0],[-0.10032279044389725,0.19402311742305756,1.0],[-0.9671552181243896,-0.25418660044670105,1.0],[-0.27129635214805603,0.9624958634376526,1.0],[0.19930581748485565,0.11002586781978607,1.0],[0.6699079871177673,-0.7424440979957581,1.0],[0.9986057281494141,0.05278843641281128,1.0],[0.9975437521934509,-0.01551083568483591,1.0],[0.996481716632843,-0.08381010591983795,1.0],[0.20731838002863684,0.8768088621202107,1.0],[-0.091013305100021,0.7621907335334465,1.0],[-0.148521636965027,0.7400962068536171,1.0],[-0.10688261510567819,0.6646694046149882,1.0],[-0.08928810851297521,0.6327979218088642,1.0],[-0.1155900195525236,0.6111599796441393,1.0],[0.32940349996093676,0.5581475751260591,1.0],[-0.7324567717243287,0.5157501436551215,1.0],[0.032700222617127,0.4118227567665387,1.0],[0.2894840430120289,0.39557905173985225,1.0],[-0.16379668444254658,0.3149166003714477,1.0],[0.09669730830685244,0.29589554929099304,1.0],[-0.5344247120770506,0.2665948653403245,1.0],[-0.45764829847336047,0.1699981388632524,1.0],[-0.19110796514263892,0.14708117542930688,1.0],[0.21318953267418536,0.0848762808085898,1.0],[-0.367104179819294,0.05607948789732546,1.0],[-0.9424904737370718,-0.06911091740395585,1.0],[-0.23330897504856202,-0.11225578842718731,1.0],[0.07082257437451021,-0.4949007887591867,1.0]]
        expected_edges = [[0,37],[1,48],[1,49],[2,49],[3,4],[4,5],[6,48],[7,48],[7,44],[8,31],[9,30],[10,35],[10,42],[11,47],[12,36],[13,45],[13,49],[14,49],[15,30],[16,32],[16,37],[17,37],[18,47],[19,40],[19,38],[20,36],[21,39],[22,41],[22,44],[23,46],[24,32],[25,41],[25,45],[26,45],[27,28],[28,29],[30,34],[30,31],[31,33],[31,32],[32,33],[33,34],[33,35],[34,38],[34,35],[35,40],[36,39],[36,38],[37,42],[38,41],[39,45],[39,41],[40,44],[40,43],[42,43],[42,47],[43,46],[43,47],[44,46],[46,48]]

        result_points, result_edges = intersect_sv_edges(sv_points, sv_edges, 6)
        self.assert_sverchok_data_equal(expected_points, result_points, precision=5)
        self.assert_sverchok_data_equal(expected_edges, result_edges)

    def test_random_edges_turned(self):
        sv_points = [[-0.7997182011604309,0.37439823150634766,1.4693365097045898],[-0.1976623684167862,-0.0979716032743454,0.8771851658821106],[0.4043934941291809,-0.5703414678573608,0.2850338816642761],[-0.9999004006385803,0.008800896815955639,1.0110325813293457],[-0.9880298376083374,0.07207910716533661,1.0903565883636475],[-0.9761592745780945,0.1353573203086853,1.1696805953979492],[-0.3664489686489105,-0.5802274942398071,0.2726409435272217],[-0.20949457585811615,0.021258961409330368,1.0266497135162354],[-0.05254018306732178,0.6227453947067261,1.780658483505249],[0.30077657103538513,0.5947304368019104,1.7455395460128784],[-0.3465045690536499,0.2626582980155945,1.3292620182037354],[-0.9937856793403625,-0.06941381096839905,0.9129845499992371],[0.4156821668148041,0.5671765208244324,1.710998773574829],[0.1814093142747879,-0.027779122814536095,0.9651767611503601],[-0.05286353826522827,-0.622734785079956,0.21935486793518066],[0.35672372579574585,0.5825796127319336,1.7303075790405273],[-0.26687315106391907,0.43317338824272156,1.5430155992507935],[-0.890470027923584,0.2837672233581543,1.3557236194610596],[-0.9954551458358765,-0.0593869686126709,0.9255539178848267],[-0.1569679230451584,0.19848425686359406,1.2488150596618652],[0.6815193295478821,0.4563555121421814,1.5720760822296143],[0.7665095925331116,0.40050071477890015,1.5020580291748047],[-0.10032279044389725,0.12099412083625793,1.1516752243041992],[-0.9671552181243896,-0.15851247310638428,0.8012925982475281],[-0.27129635214805603,0.6002188920974731,1.7524198293685913],[0.19930581748485565,0.06861287355422974,1.0860114097595215],[0.6699079871177673,-0.46299314498901367,0.41960304975509644],[0.9986057281494141,0.032919224351644516,1.041266679763794],[0.9975437521934509,-0.009672661311924458,0.9878745675086975],[0.996481716632843,-0.05226454883813858,0.9344824552536011]]
        sv_edges = [[0,1],[1,2],[3,4],[4,5],[6,7],[7,8],[9,10],[10,11],[12,13],[13,14],[15,16],[16,17],[18,19],[19,20],[21,22],[22,23],[24,25],[25,26],[27,28],[28,29]]

        expected_points = [[-0.7997182011604309,0.37439823150634766,1.4693365097045898],[-0.1976623684167862,-0.0979716032743454,0.8771851658821106],[0.4043934941291809,-0.5703414678573608,0.2850338816642761],[-0.9999004006385803,0.008800896815955639,1.0110325813293457],[-0.9880298376083374,0.07207910716533661,1.0903565883636475],[-0.9761592745780945,0.1353573203086853,1.1696805953979492],[-0.3664489686489105,-0.5802274942398071,0.2726409435272217],[-0.20949457585811615,0.021258961409330368,1.0266497135162354],[-0.05254018306732178,0.6227453947067261,1.780658483505249],[0.30077657103538513,0.5947304368019104,1.7455395460128784],[-0.3465045690536499,0.2626582980155945,1.3292620182037354],[-0.9937856793403625,-0.06941381096839905,0.9129845499992371],[0.4156821668148041,0.5671765208244324,1.710998773574829],[0.1814093142747879,-0.027779122814536095,0.9651767611503601],[-0.05286353826522827,-0.622734785079956,0.21935486793518066],[0.35672372579574585,0.5825796127319336,1.7303075790405273],[-0.26687315106391907,0.43317338824272156,1.5430155992507935],[-0.890470027923584,0.2837672233581543,1.3557236194610596],[-0.9954551458358765,-0.0593869686126709,0.9255539178848267],[-0.1569679230451584,0.19848425686359406,1.2488150596618652],[0.6815193295478821,0.4563555121421814,1.5720760822296143],[0.7665095925331116,0.40050071477890015,1.5020580291748047],[-0.10032279044389725,0.12099412083625793,1.1516752243041992],[-0.9671552181243896,-0.15851247310638428,0.8012925982475281],[-0.27129635214805603,0.6002188920974731,1.7524198293685913],[0.19930581748485565,0.06861287355422974,1.0860114097595215],[0.6699079871177673,-0.46299314498901367,0.41960304975509644],[0.9986057281494141,0.032919224351644516,1.041266679763794],[0.9975437521934509,-0.009672661311924458,0.9878745675086975],[0.996481716632843,-0.05226454883813858,0.9344824552536011],[0.2073182593137172,0.5467838807559494,1.685434927096305],[-0.09101330643544041,0.47530726735628437,1.595833597044254],[-0.14852163660505255,0.46152897114256813,1.5785614627351334],[-0.10688261602575662,0.41449231178448165,1.5195974899884934],[-0.08928812291532559,0.3946170561348557,1.494682351248927],[-0.11559001408000374,0.381123498680087,1.4777671022022523],[0.32940350878262004,0.34806460083329094,1.4363252763218874],[-0.7324567654930312,0.3216252634104626,1.4031815995588406],[0.032700216147091066,0.25681544912513893,1.3219375984174622],[0.28948404391736077,0.24668575708281096,1.3092392331142026],[-0.16379668331505182,0.1963841165973892,1.2461823745334977],[0.09669730950910475,0.18452245475356974,1.2313128140750673],[-0.5344247127320285,0.16625035803382535,1.208407384271331],[-0.4576482975277487,0.10601198711758193,1.1328940289535274],[-0.19110796553155404,0.09172081003368791,1.114978918261121],[0.21318953338722893,0.052929420715065394,1.0663510151793498],[-0.3671041767039269,0.03497154689960066,1.0438394307691625],[-0.9424904791873705,-0.04309803405963825,0.9459733329328598],[-0.2333089749423008,-0.07000346357837224,0.9122453604923121],[0.07082257295143106,-0.30862347677387564,0.6131171404466779]]
        expected_edges = [[0,37],[1,48],[1,49],[2,49],[3,4],[4,5],[6,48],[7,48],[7,44],[8,31],[9,30],[10,35],[10,42],[11,47],[12,36],[13,45],[13,49],[14,49],[15,30],[16,32],[16,37],[17,37],[18,47],[19,40],[19,38],[20,36],[21,39],[22,41],[22,44],[23,46],[24,32],[25,41],[25,45],[26,45],[27,28],[28,29],[30,34],[30,31],[31,33],[31,32],[32,33],[33,34],[33,35],[34,38],[34,35],[35,40],[36,39],[36,38],[37,42],[38,41],[39,45],[39,41],[40,44],[40,43],[42,43],[42,47],[43,46],[43,47],[44,46],[46,48]]

        result_points, result_edges = intersect_sv_edges(sv_points, sv_edges)
        self.assert_sverchok_data_equal(expected_points, result_points, precision=5)
        self.assert_sverchok_data_equal(expected_edges, result_edges)

    def test_intersect_edges_in_one_point(self):
        # https://user-images.githubusercontent.com/28003269/67545684-0ad5de80-f70b-11e9-9425-230f7cef6152.png
        sv_points = [[0.2637213169404952,0.964598915089153,0.0],[0.8114818083715089,-0.7605266562502776,0.0],[-0.9967237585943025,-0.08088107970129142,0.0],[0.8108691038779797,0.012434708894221082,0.0],[0.9325567754326458,0.36102335186891416,0.0],[-0.3500158989622022,-0.916124692704458,0.0],[0.9946071148610472,0.10371444965762261,0.0],[-0.7636998926209322,-0.32576926400531464,0.0],[0.055501526814968984,-0.99845860230718,0.0],[0.8845824885847082,0.6104927803308741,0.0],[-0.5573705547300624,-0.8302638524709495,0.0],[0.900046033284509,0.24306586473709757,0.0],[-0.9627204180816907,0.27049842256215584,0.0],[0.8197341147759787,-0.04407371726762857,0.0],[-0.41025164931159364,0.9119723593602589,0.0],[0.9149318958321713,-0.32090572242323384,0.0],[0.7726191716376339,-0.6348697627214391,0.0],[0.2223035919417532,1.0894424351145353,0.0],[0.5731572792338614,-0.8194453814997297,0.0],[0.5661834822118534,0.9905411264657407,0.0],[0.05921293539707373,0.9982453747860102,0.0],[0.8836968612654509,-0.6130665282441294,0.0],[0.47676753222498464,-0.8790294194258222,0.0],[0.6676707150741168,0.9208749579342674,0.0],[-0.5976373253453103,-0.8017665666227943,0.0],[0.8944653450833764,0.22279646890089821,0.0],[-0.5966958843929318,-0.8024674582489542,0.0],[0.8946026824259724,0.22326563445393044,0.0],[-0.43236607747849026,-0.901698161829037,0.0],[0.9132841526547624,0.30880803435678433,0.0],[-0.9575128354236951,0.2883906551864601,0.0],[0.43776737639541874,-0.5542793432938754,0.0],[-0.6519394470698725,-0.7582710315937362,0.0],[-0.2914842056045297,0.8313740906687771,0.0],[0.2437985105165767,0.9698259051344724,0.0],[-0.7311203567082857,-0.33647911214205606,0.0],[-0.12854952783656848,0.9917030900894659,0.0],[-0.673026263377309,-0.5446707837731882,0.0],[0.7194168224474339,-0.6945786028806511,0.0],[-0.6911364167969631,0.12226847894677328,0.0],[-0.9485927266719594,-0.3164993505602457,0.0],[0.4021641361778955,0.5958358363622241,0.0],[-0.9758424260078077,0.21847553547982518,0.0],[0.5157827035527763,-0.4387565914709277,0.0],[-0.559741726950337,-0.8286671220162258,0.0],[-0.40360977552876076,0.7938378578153096,0.0],[0.6037760732744754,-0.7971539709123046,0.0],[-0.709284720644688,0.1686467058865972,0.0],[-0.3692147218646271,-0.9293441177295018,0.0],[-0.5621576432475662,0.6891961582355597,0.0],[0.9801740250700544,0.19813853885089736,0.0],[-0.6350232255264014,-0.021035906508249796,0.0],[0.6229625564561608,-0.7822516559609219,0.0],[-0.7065960846421828,0.16070826668349558,0.0],[0.3840775805426537,0.9233008242845347,0.0],[-0.7297052531301415,-0.2668199238955139,0.0],[-0.06555772829225004,0.9978487782530775,0.0],[-0.6907732105100768,-0.5074768332436412,0.0],[0.5015492649142971,0.8651290856652135,0.0],[-0.7212694627072884,-0.2126528752205526,0.0]]
        sv_edges = [[0,1],[2,3],[4,5],[6,7],[8,9],[10,11],[12,13],[14,15],[16,17],[18,19],[20,21],[22,23],[24,25],[26,27],[28,29],[30,31],[32,33],[34,35],[36,37],[38,39],[40,41],[42,43],[44,45],[46,47],[48,49],[50,51],[52,53],[54,55],[56,57],[58,59]]

        expected_points = [[0.2637213169404952,0.964598915089153,0.0],[0.8114818083715089,-0.7605266562502776,0.0],[-0.9967237585943025,-0.08088107970129142,0.0],[0.8108691038779797,0.012434708894221082,0.0],[0.9325567754326458,0.36102335186891416,0.0],[-0.3500158989622022,-0.916124692704458,0.0],[0.9946071148610472,0.10371444965762261,0.0],[-0.7636998926209322,-0.32576926400531464,0.0],[0.055501526814968984,-0.99845860230718,0.0],[0.8845824885847082,0.6104927803308741,0.0],[-0.5573705547300624,-0.8302638524709495,0.0],[0.900046033284509,0.24306586473709757,0.0],[-0.9627204180816907,0.27049842256215584,0.0],[0.8197341147759787,-0.04407371726762857,0.0],[-0.41025164931159364,0.9119723593602589,0.0],[0.9149318958321713,-0.32090572242323384,0.0],[0.7726191716376339,-0.6348697627214391,0.0],[0.2223035919417532,1.0894424351145353,0.0],[0.5731572792338614,-0.8194453814997297,0.0],[0.5661834822118534,0.9905411264657407,0.0],[0.05921293539707373,0.9982453747860102,0.0],[0.8836968612654509,-0.6130665282441294,0.0],[0.47676753222498464,-0.8790294194258222,0.0],[0.6676707150741168,0.9208749579342674,0.0],[-0.5976373253453103,-0.8017665666227943,0.0],[0.8944653450833764,0.22279646890089821,0.0],[-0.5966958843929318,-0.8024674582489542,0.0],[0.8946026824259724,0.22326563445393044,0.0],[-0.43236607747849026,-0.901698161829037,0.0],[0.9132841526547624,0.30880803435678433,0.0],[-0.9575128354236951,0.2883906551864601,0.0],[0.43776737639541874,-0.5542793432938754,0.0],[-0.6519394470698725,-0.7582710315937362,0.0],[-0.2914842056045297,0.8313740906687771,0.0],[0.2437985105165767,0.9698259051344724,0.0],[-0.7311203567082857,-0.33647911214205606,0.0],[-0.12854952783656848,0.9917030900894659,0.0],[-0.673026263377309,-0.5446707837731882,0.0],[0.7194168224474339,-0.6945786028806511,0.0],[-0.6911364167969631,0.12226847894677328,0.0],[-0.9485927266719594,-0.3164993505602457,0.0],[0.4021641361778955,0.5958358363622241,0.0],[-0.9758424260078077,0.21847553547982518,0.0],[0.5157827035527763,-0.4387565914709277,0.0],[-0.559741726950337,-0.8286671220162258,0.0],[-0.40360977552876076,0.7938378578153096,0.0],[0.6037760732744754,-0.7971539709123046,0.0],[-0.709284720644688,0.1686467058865972,0.0],[-0.3692147218646271,-0.9293441177295018,0.0],[-0.5621576432475662,0.6891961582355597,0.0],[0.9801740250700544,0.19813853885089736,0.0],[-0.6350232255264014,-0.021035906508249796,0.0],[0.6229625564561608,-0.7822516559609219,0.0],[-0.7065960846421828,0.16070826668349558,0.0],[0.3840775805426537,0.9233008242845347,0.0],[-0.7297052531301415,-0.2668199238955139,0.0],[-0.06555772829225004,0.9978487782530775,0.0],[-0.6907732105100768,-0.5074768332436412,0.0],[0.5015492649142971,0.8651290856652135,0.0],[-0.7212694627072884,-0.2126528752205526,0.0],[0.30400232608931677,0.8377372665966724,0],[0.3029830806645228,0.8366481627867898,0],[0.14291899810423322,0.8346562847333734,0],[-0.29708232014351454,0.8066858230364704,0],[-0.2196483620808588,0.7346455158420179,0],[0.19884037265257773,0.7253675847157831,0],[0.3404030646099808,0.723096203905961,0],[0.3394822881418598,0.7222846394075344,0],[-0.1873541178774979,0.7046007224305754,0],[0.243641536873483,0.6378114215040858,0],[0.3845814629164034,0.5839600562438276,0],[0.38379764764901064,0.5834306476817415,0],[-0.04971276915752133,0.5765467614484626,0],[0.3003182428234281,0.5270465562879447,0],[0.00870380424421626,0.5221991858059614,0],[0.05918568534187384,0.47523361265152675,0],[0.1283450502509359,0.4108915330513767,0],[-0.9135057893306843,0.2618128988689864,0],[-0.5025647152638305,0.18928896592697148,0],[-0.46246586887749713,0.1822122188459496,0],[-0.4395980886312018,0.17817645443646335,0],[0.8286020362497568,0.1775709539510321,0],[0.828061132369144,0.1774975559137446,0],[0.8071639969173342,0.17466191582671867,0],[-0.41819452673114427,0.174399099029767,0],[-0.4082916609850482,0.17265141592718475,0],[0.75652264534223,0.16779012964728313,0],[0.7356597908190727,0.1649591413162709,0],[-0.3577977674806966,0.16374012431486204,0],[-0.3311618680002146,0.15903935257487659,0],[-0.3048310076567387,0.15439241492282496,0],[0.6489384472657269,0.15319147494172397,0],[-0.2624802998731194,0.14691825357665003,0],[0.5853325024842844,0.14456045624486252,0],[0.5694513109998893,0.14240545546464867,0],[0.5266285789755544,0.13659461809680948,0],[0.5264149652094128,0.13656563174316266,0],[0.5018285382644329,0.1332293726658028,0],[0.43634659516609836,0.12434379008180768,0],[0.11359359119648428,0.08054777036538206,0],[-0.4799999594688416,-1.5753435389402893e-15,0],[0.5699999928474425,4.15085022446408e-16,0],[-0.3698795956313927,-0.04852066347715669,0],[-0.39405779254298257,-0.04976884693619997,0],[-0.39731503209061403,-0.04993699977821551,0],[-0.40875673992785067,-0.050527670394302636,0],[-0.4111376209394219,-0.050650581805251116,0],[-0.4735777471592734,-0.05387401211368882,0],[-0.48524214499790086,-0.05447617895413242,0],[-0.492436762694486,-0.05484759634459602,0],[-0.49956790757030817,-0.05521573699257691,0],[-0.5030067607538147,-0.055393265518687715,0],[-0.5220756360735075,-0.05637768362247071,0],[-0.5333038794854819,-0.05695733427848055,0],[-0.5453261871519487,-0.057577977977582025,0],[-0.5668955706587857,-0.058691483166303925,0],[-0.10551859898564406,-0.1650020344765105,0],[-0.16850342308006605,-0.18038669485798628,0],[-0.17762803201518643,-0.18261546998912198,0],[-0.21101609227830304,-0.19077083082928648,0],[-0.21823793103333836,-0.19253483570146707,0],[-0.4502914025988663,-0.24921616968481083,0],[-0.5052741314935064,-0.26264623991288205,0],[-0.5415655773559225,-0.2715107811317156,0],[-0.5795050472479312,-0.28077786836636054,0],[0.15958837188945138,-0.2818120927711424,0],[0.1600083874736779,-0.2819971578590811,0],[-0.5985490449624703,-0.2854295513924902,0],[0.17695331530537825,-0.28946334406039304,0],[0.2247896713256929,-0.31054075110439117,0],[-0.7140823521401216,-0.31364969361297623,0],[0.24790986470962073,-0.3207278507798358,0],[0.0896130447318635,-0.3298611546924267,0],[0.09005093465148264,-0.3301147354102975,0],[0.07864702701158476,-0.33739104966933703,0],[0.07908629799492546,-0.3376563443934567,0],[0.10779913546452155,-0.3403926644089817,0],[0.09690346038030527,-0.3484168984907854,0],[0.03651109670238633,-0.36632398941791616,0],[0.03695209383577794,-0.36663675683556524,0],[0.15878373630416673,-0.3699176913667088,0],[0.026957259036564193,-0.3728842007986673,0],[0.027397857236311395,-0.37320827569411985,0],[0.3757139124332492,-0.3770402073188142,0],[0.05488953891673652,-0.3793584911789308,0],[0.1482275389496836,-0.3794137268294284,0],[0.18390288481805553,-0.38446411371412126,0],[0.045330502627540716,-0.3863983436146994,0],[0.17359135493996186,-0.3947320310674836,0],[0.10711442992775694,-0.41639783466150393,0],[0.0976687410967237,-0.4248948902864836,0],[0.1332317334798427,-0.4349209518643757,0],[0.12391406304244393,-0.44419921329678747,0],[0.3286848455198216,-0.4683069058967417,0],[0.32078856248881404,-0.4836307838848718,0],[0.28897167720664785,-0.5453760463218539,0],[0.28141290164161653,-0.5600449423129614,0],[0.5092401380978715,-0.5728659255406098,0],[0.5723480281345101,-0.609411512323515,0],[-0.40055514965575983,-0.6664386725214545,0],[-0.40043166289663107,-0.6674745658922279,0],[0.4965417915657924,-0.6925905376042387,0],[-0.3952580557463043,-0.7108744039624992,0],[0.4940143171145825,-0.7164204825035518,0],[0.5728771088326886,-0.7467296651613096,0],[-0.5542888048333838,-0.7720009930859545,0],[-0.5544226008180857,-0.7733913853117778,0],[0.5729841276974834,-0.774505452263408,0],[-0.3783073737748158,-0.8530685890576702,0]]
        expected_edges = [[0,60],[1,101],[2,115],[3,101],[4,87],[5,152],[6,101],[7,130],[8,156],[9,91],[10,162],[11,83],[12,77],[13,101],[14,63],[15,101],[16,101],[17,61],[18,167],[19,94],[20,62],[21,101],[22,163],[23,93],[24,165],[25,81],[26,166],[27,82],[28,168],[29,86],[30,77],[31,154],[32,123],[33,63],[34,62],[35,130],[36,64],[37,124],[38,158],[39,100],[40,115],[41,70],[42,100],[43,143],[44,166],[45,79],[46,167],[47,100],[48,168],[49,78],[50,81],[51,100],[52,164],[53,100],[54,60],[55,113],[56,68],[57,127],[58,66],[59,114],[60,66],[60,61],[61,67],[61,65],[62,65],[62,72],[63,64],[63,80],[64,68],[64,84],[65,69],[65,74],[66,70],[66,67],[67,71],[67,69],[68,72],[68,85],[69,73],[69,75],[70,95],[70,71],[71,96],[71,73],[72,74],[72,88],[73,97],[73,76],[74,75],[74,89],[75,76],[75,90],[76,98],[76,92],[77,78],[77,100],[78,79],[78,100],[79,80],[79,100],[80,84],[80,100],[81,101],[81,82],[82,101],[82,83],[83,101],[83,86],[84,85],[84,100],[85,88],[85,100],[86,101],[86,87],[87,101],[87,91],[88,89],[88,100],[89,90],[89,100],[90,92],[90,100],[91,101],[91,93],[92,99],[92,100],[93,101],[93,94],[94,101],[94,95],[95,101],[95,96],[96,101],[96,97],[97,101],[97,98],[98,101],[98,99],[99,101],[99,100],[100,102],[100,103],[100,104],[100,105],[100,106],[100,107],[100,108],[100,109],[100,110],[100,111],[100,112],[100,113],[100,114],[100,115],[101,158],[101,157],[101,143],[101,131],[101,129],[101,128],[101,126],[101,125],[101,116],[101,102],[102,116],[102,103],[103,117],[103,104],[104,118],[104,105],[105,119],[105,106],[106,120],[106,107],[107,121],[107,108],[108,122],[108,109],[109,123],[109,110],[110,124],[110,111],[111,127],[111,112],[112,130],[112,113],[113,114],[114,115],[116,125],[116,117],[117,132],[117,118],[118,134],[118,119],[119,138],[119,120],[120,141],[120,121],[121,159],[121,122],[122,165],[122,123],[123,124],[124,127],[125,126],[125,132],[126,128],[126,133],[127,130],[128,129],[128,136],[129,131],[129,140],[131,143],[131,146],[132,133],[132,134],[133,136],[133,135],[134,135],[134,138],[135,137],[135,139],[136,140],[136,137],[137,145],[137,144],[138,139],[138,141],[139,144],[139,142],[140,146],[140,145],[141,142],[141,159],[142,147],[142,160],[143,153],[144,149],[144,147],[145,148],[145,149],[146,153],[146,148],[147,150],[147,162],[148,154],[148,151],[149,151],[149,150],[150,152],[150,168],[151,155],[151,152],[152,156],[153,157],[153,154],[154,155],[155,161],[155,156],[156,163],[157,158],[157,161],[158,164],[159,160],[159,165],[160,162],[160,166],[161,164],[161,163],[162,168],[163,167],[164,167],[165,166]]

        result_points, result_edges = intersect_sv_edges(sv_points, sv_edges, accuracy=6)
        self.assert_sverchok_data_equal(expected_points, result_points, precision=5)
        self.assert_sverchok_data_equal(expected_edges, result_edges)

    def test_intersect_overlapping_points_with_edges(self):
        # https://user-images.githubusercontent.com/28003269/67550584-e2a0ac80-f717-11e9-92d0-03fa38aee775.gif
        sv_points = [[-0.5,-0.5,0.0],[0.5,-0.5,0.0],[-0.5,0.5,0.0],[0.5,0.5,0.0],[0.0,-0.4999999701976776,0.0],[0.4999999701976776,0.0,0.0],[-0.4999999701976776,0.0,0.0],[0.0,0.4999999701976776,0.0],[0.25,-0.2499999850988388,0.0],[0.2499999850988388,0.25,0.0],[-0.2499999850988388,-0.25,0.0],[-0.25,0.2499999850988388,0.0],[0.2499999850988388,0.0,0.0],[0.0,0.2499999850988388,0.0],[0.0,-0.2499999850988388,0.0],[-0.2499999850988388,0.0,0.0],[0.1249999925494194,0.1250000149011612,0.0],[-0.1250000149011612,0.1249999925494194,0.0],[0.1250000149011612,-0.1249999925494194,0.0],[-0.1249999925494194,-0.1250000149011612,0.0],[7.450580596923828e-09,0.1249999925494194,0.0],[-0.1249999925494194,7.450580596923828e-09,0.0],[0.1249999925494194,-7.450580596923828e-09,0.0],[-7.450580596923828e-09,-0.1249999925494194,0.0],[-0.0625,0.0625,0.0],[-0.0625,-0.0625,0.0],[0.0625,0.0625,0.0],[0.0625,-0.0625,0.0],[-0.0625,9.313225746154785e-09,0.0],[-9.313225746154785e-09,-0.0625,0.0],[9.313225746154785e-09,0.0625,0.0],[0.0625,-9.313225746154785e-09,0.0],[-0.031249994412064552,-0.0312500037252903,0.0],[0.0312500037252903,-0.031249994412064552,0.0],[-0.0312500037252903,0.031249994412064552,0.0],[0.031249994412064552,0.0312500037252903,0.0],[0.0,-0.03124999813735485,0.0],[0.03124999813735485,0.0,0.0],[-0.03124999813735485,0.0,0.0],[0.0,0.03124999813735485,0.0]]
        sv_edges = [[0,1],[2,3],[0,2],[1,3],[4,5],[6,7],[4,6],[5,7],[8,9],[10,11],[8,10],[9,11],[12,13],[14,15],[12,14],[13,15],[16,17],[18,19],[16,18],[17,19],[20,21],[22,23],[20,22],[21,23],[24,25],[26,27],[24,26],[25,27],[28,29],[30,31],[28,30],[29,31],[32,33],[34,35],[32,34],[33,35],[36,37],[38,39],[36,38],[37,39]]

        expected_points = [[-0.5,-0.5,0.0],[0.5,-0.5,0.0],[-0.5,0.5,0.0],[0.5,0.5,0.0],[0.0,-0.4999999701976776,0.0],[0.4999999701976776,0.0,0.0],[-0.4999999701976776,0.0,0.0],[0.0,0.4999999701976776,0.0],[0.25,-0.2499999850988388,0.0],[0.2499999850988388,0.25,0.0],[-0.2499999850988388,-0.25,0.0],[-0.25,0.2499999850988388,0.0],[0.2499999850988388,0.0,0.0],[0.0,0.2499999850988388,0.0],[0.0,-0.2499999850988388,0.0],[-0.2499999850988388,0.0,0.0],[0.1249999925494194,0.1250000149011612,0.0],[-0.1250000149011612,0.1249999925494194,0.0],[0.1250000149011612,-0.1249999925494194,0.0],[-0.1249999925494194,-0.1250000149011612,0.0],[7.450580596923828e-09,0.1249999925494194,0.0],[-0.1249999925494194,7.450580596923828e-09,0.0],[0.1249999925494194,-7.450580596923828e-09,0.0],[-7.450580596923828e-09,-0.1249999925494194,0.0],[-0.0625,0.0625,0.0],[-0.0625,-0.0625,0.0],[0.0625,0.0625,0.0],[0.0625,-0.0625,0.0],[-0.0625,9.313225746154785e-09,0.0],[-9.313225746154785e-09,-0.0625,0.0],[9.313225746154785e-09,0.0625,0.0],[0.0625,-9.313225746154785e-09,0.0],[-0.031249994412064552,-0.0312500037252903,0.0],[0.0312500037252903,-0.031249994412064552,0.0],[-0.0312500037252903,0.031249994412064552,0.0],[0.031249994412064552,0.0312500037252903,0.0],[0.0,-0.03124999813735485,0.0],[0.03124999813735485,0.0,0.0],[-0.03124999813735485,0.0,0.0],[0.0,0.03124999813735485,0.0]]
        expected_edges = [[0,4],[1,4],[2,7],[3,7],[0,6],[2,6],[1,5],[3,5],[4,8],[5,8],[6,11],[7,11],[4,10],[6,10],[5,9],[7,9],[8,12],[9,12],[10,15],[11,15],[8,14],[10,14],[9,13],[11,13],[12,16],[13,16],[14,19],[15,19],[12,18],[14,18],[13,17],[15,17],[16,20],[17,20],[18,23],[19,23],[16,22],[18,22],[17,21],[19,21],[20,24],[21,24],[22,27],[23,27],[20,26],[22,26],[21,25],[23,25],[24,28],[25,28],[26,31],[27,31],[24,30],[26,30],[25,29],[27,29],[28,32],[29,32],[30,35],[31,35],[28,34],[30,34],[29,33],[31,33],[32,36],[33,36],[34,39],[35,39],[32,38],[34,38],[33,37],[35,37],[36,37],[38,39],[36,38],[37,39]]

        result_points, result_edges = intersect_sv_edges(sv_points, sv_edges, accuracy=6)
        self.assert_sverchok_data_equal(expected_points, result_points, precision=5)
        self.assert_sverchok_data_equal(expected_edges, result_edges)

    def test_intersect_overlapping_edges(self):
        # https://user-images.githubusercontent.com/28003269/67559195-8050a780-f729-11e9-9fa2-3bae40836c6b.png
        sv_points = [[-0.25,-0.5,0.0],[0.25,-0.5,0.0],[-0.25,0.5,0.0],[0.25,0.5,0.0],[-0.25,-0.3999999985098839,0.0],[0.25,-0.3999999985098839,0.0],[-0.25,0.3999999985098839,0.0],[0.25,0.3999999985098839,0.0],[-0.25,-0.29999999701976776,0.0],[0.25,-0.29999999701976776,0.0],[-0.25,0.29999999701976776,0.0],[0.25,0.29999999701976776,0.0],[-0.25,-0.19999999552965164,0.0],[0.25,-0.19999999552965164,0.0],[-0.25,0.19999999552965164,0.0],[0.25,0.19999999552965164,0.0],[-0.25,-0.09999999403953552,0.0],[0.25,-0.09999999403953552,0.0],[-0.25,0.09999999403953552,0.0],[0.25,0.09999999403953552,0.0],[-0.25,7.450580596923828e-09,0.0],[0.25,7.450580596923828e-09,0.0],[-0.25,-7.450580596923828e-09,0.0],[0.25,-7.450580596923828e-09,0.0],[-0.5,-0.5,0.0],[0.5,-0.5,0.0],[-0.5,0.5,0.0],[0.5,0.5,0.0],[-0.3999999985098839,-0.5,0.0],[0.3999999985098839,-0.5,0.0],[-0.3999999985098839,0.5,0.0],[0.3999999985098839,0.5,0.0],[-0.29999999701976776,-0.5,0.0],[0.29999999701976776,-0.5,0.0],[-0.29999999701976776,0.5,0.0],[0.29999999701976776,0.5,0.0],[-0.19999999552965164,-0.5,0.0],[0.19999999552965164,-0.5,0.0],[-0.19999999552965164,0.5,0.0],[0.19999999552965164,0.5,0.0],[-0.09999999403953552,-0.5,0.0],[0.09999999403953552,-0.5,0.0],[-0.09999999403953552,0.5,0.0],[0.09999999403953552,0.5,0.0],[7.450580596923828e-09,-0.5,0.0],[-7.450580596923828e-09,-0.5,0.0],[7.450580596923828e-09,0.5,0.0],[-7.450580596923828e-09,0.5,0.0],[-0.19999998807907104,-0.4000000059604645,0.0],[-0.19999998807907104,0.3999999761581421,0.0],[-0.19999998807907104,-0.30000001192092896,0.0],[-0.19999998807907104,0.30000001192092896,0.0],[-0.19999998807907104,-0.19999998807907104,0.0],[-0.19999998807907104,0.19999998807907104,0.0],[-0.19999998807907104,-0.09999999403953552,0.0],[-0.19999998807907104,0.10000002384185791,0.0],[-0.19999998807907104,0.0,0.0],[0.19999998807907104,-0.4000000059604645,0.0],[0.19999998807907104,0.3999999761581421,0.0],[0.19999998807907104,-0.30000001192092896,0.0],[0.19999998807907104,0.30000001192092896,0.0],[0.19999998807907104,-0.19999998807907104,0.0],[0.19999998807907104,0.19999998807907104,0.0],[0.19999998807907104,-0.09999999403953552,0.0],[0.19999998807907104,0.10000002384185791,0.0],[0.19999998807907104,0.0,0.0],[-0.09999999403953552,-0.4000000059604645,0.0],[-0.09999999403953552,0.3999999761581421,0.0],[-0.09999999403953552,-0.30000001192092896,0.0],[-0.09999999403953552,0.30000001192092896,0.0],[-0.09999999403953552,-0.19999998807907104,0.0],[-0.09999999403953552,0.19999998807907104,0.0],[-0.09999999403953552,-0.09999999403953552,0.0],[-0.09999999403953552,0.10000002384185791,0.0],[-0.09999999403953552,0.0,0.0],[0.09999999403953552,-0.4000000059604645,0.0],[0.09999999403953552,0.3999999761581421,0.0],[0.09999999403953552,-0.30000001192092896,0.0],[0.09999999403953552,0.30000001192092896,0.0],[0.09999999403953552,-0.19999998807907104,0.0],[0.09999999403953552,0.19999998807907104,0.0],[0.09999999403953552,-0.09999999403953552,0.0],[0.09999999403953552,0.10000002384185791,0.0],[0.09999999403953552,0.0,0.0],[7.450580596923828e-09,-0.4000000059604645,0.0],[7.450580596923828e-09,0.3999999761581421,0.0],[7.450580596923828e-09,-0.30000001192092896,0.0],[7.450580596923828e-09,0.30000001192092896,0.0],[7.450580596923828e-09,-0.19999998807907104,0.0],[7.450580596923828e-09,0.19999998807907104,0.0],[7.450580596923828e-09,-0.09999999403953552,0.0],[7.450580596923828e-09,0.10000002384185791,0.0],[7.450580596923828e-09,0.0,0.0],[-7.450580596923828e-09,-0.4000000059604645,0.0],[-7.450580596923828e-09,0.3999999761581421,0.0],[-7.450580596923828e-09,-0.30000001192092896,0.0],[-7.450580596923828e-09,0.30000001192092896,0.0],[-7.450580596923828e-09,-0.19999998807907104,0.0],[-7.450580596923828e-09,0.19999998807907104,0.0],[-7.450580596923828e-09,-0.09999999403953552,0.0],[-7.450580596923828e-09,0.10000002384185791,0.0],[-7.450580596923828e-09,0.0,0.0]]
        sv_edges = [[0,1],[2,3],[0,2],[1,3],[4,5],[6,7],[4,6],[5,7],[8,9],[10,11],[8,10],[9,11],[12,13],[14,15],[12,14],[13,15],[16,17],[18,19],[16,18],[17,19],[20,21],[22,23],[20,22],[21,23],[24,25],[26,27],[24,26],[25,27],[28,29],[30,31],[28,30],[29,31],[32,33],[34,35],[32,34],[33,35],[36,37],[38,39],[36,38],[37,39],[40,41],[42,43],[40,42],[41,43],[44,45],[46,47],[44,46],[45,47]]

        expected_points = [[-0.25,-0.5,0.0],[0.25,-0.5,0.0],[-0.25,0.5,0.0],[0.25,0.5,0.0],[-0.25,-0.3999999985098839,0.0],[0.25,-0.3999999985098839,0.0],[-0.25,0.3999999985098839,0.0],[0.25,0.3999999985098839,0.0],[-0.25,-0.29999999701976776,0.0],[0.25,-0.29999999701976776,0.0],[-0.25,0.29999999701976776,0.0],[0.25,0.29999999701976776,0.0],[-0.25,-0.19999999552965164,0.0],[0.25,-0.19999999552965164,0.0],[-0.25,0.19999999552965164,0.0],[0.25,0.19999999552965164,0.0],[-0.25,-0.09999999403953552,0.0],[0.25,-0.09999999403953552,0.0],[-0.25,0.09999999403953552,0.0],[0.25,0.09999999403953552,0.0],[-0.25,7.450580596923828e-09,0.0],[0.25,7.450580596923828e-09,0.0],[-0.5,-0.5,0.0],[0.5,-0.5,0.0],[-0.5,0.5,0.0],[0.5,0.5,0.0],[-0.3999999985098839,-0.5,0.0],[0.3999999985098839,-0.5,0.0],[-0.3999999985098839,0.5,0.0],[0.3999999985098839,0.5,0.0],[-0.29999999701976776,-0.5,0.0],[0.29999999701976776,-0.5,0.0],[-0.29999999701976776,0.5,0.0],[0.29999999701976776,0.5,0.0],[-0.19999999552965164,-0.5,0.0],[0.19999999552965164,-0.5,0.0],[-0.19999999552965164,0.5,0.0],[0.19999999552965164,0.5,0.0],[-0.09999999403953552,-0.5,0.0],[0.09999999403953552,-0.5,0.0],[-0.09999999403953552,0.5,0.0],[0.09999999403953552,0.5,0.0],[7.450580596923828e-09,-0.5,0.0],[7.450580596923828e-09,0.5,0.0],[-0.19999999552965164,0.3999999985098839,0],[-0.09999999403953552,0.3999999985098839,0],[7.450580596923828e-09,0.3999999985098839,0],[0.09999999403953552,0.3999999985098839,0],[0.19999999552965164,0.3999999985098839,0],[-0.19999999552965164,0.29999999701976776,0],[-0.09999999403953552,0.29999999701976776,0],[7.450580596923828e-09,0.29999999701976776,0],[0.09999999403953554,0.29999999701976776,0],[0.19999999552965164,0.29999999701976776,0],[-0.19999999552965164,0.19999999552965164,0],[-0.09999999403953552,0.19999999552965164,0],[7.450580596923828e-09,0.19999999552965164,0],[0.09999999403953552,0.19999999552965164,0],[0.19999999552965164,0.19999999552965164,0],[-0.19999999552965164,0.09999999403953552,0],[-0.09999999403953552,0.09999999403953552,0],[7.450580596923828e-09,0.09999999403953552,0],[0.09999999403953554,0.09999999403953554,0],[0.19999999552965164,0.09999999403953552,0],[-0.19999999552965164,7.450580596923828e-09,0],[-0.09999999403953552,7.450580596923828e-09,0],[7.450580596923828e-09,7.450580596923828e-09,0],[0.09999999403953554,7.450580596923828e-09,0],[0.19999999552965167,7.450580596923828e-09,0],[-0.19999999552965164,-0.09999999403953552,0],[-0.09999999403953552,-0.09999999403953552,0],[7.450580596923828e-09,-0.09999999403953552,0],[0.09999999403953554,-0.09999999403953552,0],[0.19999999552965167,-0.09999999403953551,0],[-0.19999999552965164,-0.19999999552965164,0],[-0.09999999403953554,-0.19999999552965164,0],[7.450580596923828e-09,-0.19999999552965164,0],[0.09999999403953554,-0.19999999552965161,0],[0.19999999552965167,-0.1999999955296516,0],[-0.19999999552965164,-0.29999999701976776,0],[-0.09999999403953554,-0.29999999701976776,0],[7.450580596923828e-09,-0.29999999701976776,0],[0.09999999403953555,-0.29999999701976776,0],[0.19999999552965164,-0.29999999701976776,0],[-0.19999999552965164,-0.3999999985098839,0],[-0.09999999403953554,-0.3999999985098838,0],[7.450580596923828e-09,-0.3999999985098839,0],[0.09999999403953554,-0.3999999985098839,0],[0.19999999552965164,-0.3999999985098838,0]]
        expected_edges = [[0,34],[1,35],[2,36],[3,37],[0,4],[2,6],[1,5],[3,7],[4,84],[5,88],[6,44],[7,48],[4,8],[6,10],[5,9],[7,11],[8,79],[9,83],[10,49],[11,53],[8,12],[10,14],[9,13],[11,15],[12,74],[13,78],[14,54],[15,58],[12,16],[14,18],[13,17],[15,19],[16,69],[17,73],[18,59],[19,63],[16,20],[18,20],[17,21],[19,21],[20,64],[21,68],[22,26],[23,27],[24,28],[25,29],[22,24],[23,25],[26,30],[27,31],[28,32],[29,33],[26,28],[27,29],[30,0],[31,1],[32,2],[33,3],[30,32],[31,33],[34,38],[35,39],[36,40],[37,41],[34,84],[36,44],[35,88],[37,48],[38,42],[39,42],[40,43],[41,43],[38,85],[40,45],[39,87],[41,47],[42,86],[43,46],[44,45],[44,49],[45,46],[45,50],[46,47],[46,51],[47,48],[47,52],[48,53],[49,50],[49,54],[50,51],[50,55],[51,52],[51,56],[52,53],[52,57],[53,58],[54,55],[54,59],[55,56],[55,60],[56,57],[56,61],[57,58],[57,62],[58,63],[59,60],[59,64],[60,61],[60,65],[61,62],[61,66],[62,63],[62,67],[63,68],[64,65],[64,69],[65,66],[65,70],[66,67],[66,71],[67,68],[67,72],[68,73],[69,70],[69,74],[70,71],[70,75],[71,72],[71,76],[72,73],[72,77],[73,78],[74,75],[74,79],[75,76],[75,80],[76,77],[76,81],[77,78],[77,82],[78,83],[79,80],[79,84],[80,81],[80,85],[81,82],[81,86],[82,83],[82,87],[83,88],[84,85],[85,86],[86,87],[87,88]]

        result_points, result_edges = intersect_sv_edges(sv_points, sv_edges, accuracy=6)
        self.assert_sverchok_data_equal(expected_points, result_points, precision=5)
        self.assert_sverchok_data_equal(expected_edges, result_edges)


class EdgesToFacesTest(SverchokTestCase):

    def test_random_edges_intersection(self):
        # https://user-images.githubusercontent.com/28003269/67630013-d3496c80-f899-11e9-8bad-77f9bc196eab.gif
        sv_points = [[-0.25728213871198163,-0.9663363291834726,0.0],[0.7540366763931518,-0.6568323154763088,0.0],[-0.9842388272607548,-0.17684436918481167,0.0],[0.8825502870986872,0.4702180246885743,0.0],[0.5590865085460759,-0.8291093269055406,0.0],[0.9918336231764538,-0.1275384802193761,0.0],[-0.5116071504859823,-0.859219485097733,0.0],[0.014630095343144033,0.9998929744278887,0.0],[-0.3773023955002547,0.9260901156743707,0.0],[0.01812435429027781,-0.9998357404001722,0.0],[-0.5172492100333327,-0.8558348291112563,0.0],[-0.9859866214274391,-0.16682440578675503,0.0],[-0.4262620665127514,-0.9045997184679413,0.0],[0.8359928285965739,-0.5487403671456106,0.0],[0.5820521032563607,0.8131514920940912,0.0],[0.7462353462520661,-0.6656822124738342,0.0],[-0.24072300967552557,-0.9705938556434184,0.0],[0.8467801940557685,0.5319429508461175,0.0],[-0.5883337371805069,0.8086182125671041,0.0],[-0.9940676195343837,-0.10876381656251169,0.0],[1.0,0.0,0],[0.9135454576426009,0.40673664307580015,0],[0.6691306063588582,0.7431448254773941,0],[0.30901699437494745,0.9510565162951535,0],[-0.10452846326765333,0.9945218953682734,0],[-0.4999999999999998,0.8660254037844387,0],[-0.8090169943749473,0.5877852522924732,0],[-0.9781476007338056,0.20791169081775973,0],[-0.9781476007338057,-0.20791169081775907,0],[-0.8090169943749475,-0.587785252292473,0],[-0.5000000000000004,-0.8660254037844385,0],[-0.10452846326765423,-0.9945218953682733,0],[0.30901699437494723,-0.9510565162951536,0],[0.6691306063588578,-0.7431448254773946,0],[0.9135454576426005,-0.40673664307580093,0]]
        sv_edges = [[0,1],[2,3],[4,5],[6,7],[8,9],[10,11],[12,13],[14,15],[16,17],[18,19],[20,21],[21,22],[22,23],[23,24],[24,25],[25,26],[26,27],[27,28],[28,29],[29,30],[30,31],[31,32],[32,33],[33,34],[34,20]]

        expected_points = [[1.0,0.0,0],[0.9135454576426009,0.40673664307580015,0],[0.6691306063588582,0.7431448254773941,0],[0.30901699437494745,0.9510565162951535,0],[-0.10452846326765333,0.9945218953682734,0],[-0.4999999999999998,0.8660254037844387,0],[-0.8090169943749473,0.5877852522924732,0],[-0.9781476007338056,0.20791169081775973,0],[-0.9781476007338057,-0.20791169081775907,0],[-0.8090169943749475,-0.587785252292473,0],[-0.5000000000000004,-0.8660254037844385,0],[-0.10452846326765423,-0.9945218953682733,0],[0.30901699437494723,-0.9510565162951536,0],[0.6691306063588578,-0.7431448254773946,0],[0.9135454576426005,-0.40673664307580093,0],[0.00971106890785301,0.9825148367018581,0],[-0.3734147702544803,0.9071554381809255,0],[0.5843928134527131,0.7920682130222916,0],[-0.6045973109149059,0.7718455620139787,0],[0.83470203824244,0.515255300086308,0],[0.8704666003566213,0.4660296033987811,0],[0.7751647695755288,0.4329962895169987,0],[0.6298497839480579,0.3826275243776903,0],[0.6443272314196953,0.25222604803622706,0],[-0.2213129695317568,0.1663436672965157,0],[-0.20621951412631276,0.09283099765984774,0],[-0.24602697384780564,0.07903302231101146,0],[-0.9781476007338058,-0.07276795872010959,0],[0.9629311423208354,-0.17439526396136865,0],[-0.9781476007338058,-0.17473304145752036,0],[-0.9801371803495432,-0.17542266522865824,0],[-0.9781476007338059,-0.17834720476818436,0],[0.7330650778067074,-0.5470547781520907,0],[0.8037744384240053,-0.5578234893199692,0],[0.7363703729831974,-0.5768262811468737,0],[0.7101398049395521,-0.5842212953908226,0],[0.7434749388256303,-0.6408186303806307,0],[0.7455414558599267,-0.6594321927979696,0],[0.7254937694512065,-0.665567586816518,0],[0.6447142796140658,-0.6902893421684272,0],[-0.04375455021604554,-0.6984545561544799,0],[-0.024674732702977438,-0.7913828011380759,0],[0.5805747079207677,-0.7942725972789548,0],[-0.1331479428260042,-0.8219639510625434,0],[-0.5266266562213346,-0.8420506548366204,0],[-0.5107849070484407,-0.8563146298606465,0],[-0.004629226794354452,-0.8890144311377223,0],[-0.40217774425717084,-0.8978097814052163,0],[-0.22881351525123847,-0.9541392340314763,0],[-0.223290884360314,-0.9559336455829619,0],[-0.23205315905575996,-0.9586152521613611,0],[0.014465466654666094,-0.9820151293625302,0]]
        expected_faces = [[0,1,20,21,23,32,28],[2,17,22,21,19],[3,15,24,25,22,17],[4,16,24,15],[5,18,27,29,26,24,16],[6,7,27,18],[8,9,44,31],[10,47,43,40,25,26,45],[11,51,46,49],[12,42,39,46,51],[13,38,39,42],[14,28,32,34,33],[19,21,20],[21,22,23],[22,25,40,23],[23,40,41,35,32],[24,26,25],[26,29,31,44,45],[29,30,31],[32,35,34],[33,34,36],[34,35,39,38,36],[35,41,46,39],[36,38,37],[40,43,41],[41,43,48,49,46],[43,47,48],[48,50,49]]

        result_points, result_faces = edges_to_faces(sv_points, sv_edges, True, True, 7)
        self.assert_sverchok_data_equal(expected_points, result_points, precision=5)
        self.assert_sverchok_data_equal(expected_faces, result_faces)

    def test_random_boxes_intersection(self):
        # https://user-images.githubusercontent.com/28003269/68003096-62cd9180-fc85-11e9-828a-648ee8f11f58.png
        sv_points = [[-3.5888919830322266,-3.670072555541992,0.0],[6.411108016967773,-3.670072555541992,0.0],[-3.5888919830322266,6.329927444458008,0.0],[6.411108016967773,6.329927444458008,0.0],[-0.3651900291442871,-2.0452067852020264,0.0],[-0.16519004106521606,-2.0452067852020264,0.0],[-0.3651900291442871,-1.8452068567276,0.0],[-0.16519004106521606,-1.8452068567276,0.0],[-2.3631088733673096,-3.2004289627075195,0.0],[0.5003190636634827,-3.2004289627075195,0.0],[-2.3631088733673096,-0.33700108528137207,0.0],[0.5003190636634827,-0.33700108528137207,0.0],[-2.475944757461548,-2.4749131202697754,0.0],[0.677252471446991,-2.4749131202697754,0.0],[-2.475944757461548,0.6782842874526978,0.0],[0.677252471446991,0.6782842874526978,0.0],[-0.232285737991333,0.08767950534820557,0.0],[1.8574066162109375,0.08767950534820557,0.0],[-0.232285737991333,2.1773719787597656,0.0],[1.8574066162109375,2.1773719787597656,0.0]]
        sv_edges = [[0,1],[2,3],[0,2],[1,3],[4,5],[6,7],[4,6],[5,7],[8,9],[10,11],[8,10],[9,11],[12,13],[14,15],[12,14],[13,15],[16,17],[18,19],[16,18],[17,19]]

        expected_points = [[-3.5888919830322266,-3.670072555541992,0.0],[6.411108016967773,-3.670072555541992,0.0],[-3.5888919830322266,6.329927444458008,0.0],[6.411108016967773,6.329927444458008,0.0],[-0.3651900291442871,-2.0452067852020264,0.0],[-0.16519004106521606,-2.0452067852020264,0.0],[-0.3651900291442871,-1.8452068567276,0.0],[-0.16519004106521606,-1.8452068567276,0.0],[-2.3631088733673096,-3.2004289627075195,0.0],[0.5003190636634827,-3.2004289627075195,0.0],[-2.475944757461548,-2.4749131202697754,0.0],[0.677252471446991,-2.4749131202697754,0.0],[-2.475944757461548,0.6782842874526978,0.0],[1.8574066162109375,0.08767950534820557,0.0],[-0.232285737991333,2.1773719787597656,0.0],[1.8574066162109375,2.1773719787597656,0.0],[-0.232285737991333,0.6782842874526978,0],[0.677252471446991,0.08767950534820557,0],[-2.3631088733673096,-2.4749131202697754,0],[0.5003190636634827,-2.4749131202697754,0]]
        expected_faces = [[3,14,15,13,11,9,0,1],[14,3,2,0,9,8,18,10,12],[4,5,7,6],[9,11,19],[14,12,16],[11,13,17]]

        result_points, result_faces = edges_to_faces(sv_points, sv_edges, True, False, 7)
        self.assert_sverchok_data_equal(expected_points, result_points, precision=5)
        self.assert_sverchok_data_equal(expected_faces, result_faces)

    def test_holes_in_holes(self):
        # https://user-images.githubusercontent.com/28003269/68003706-04ee7900-fc88-11e9-870a-b9011150effa.png
        sv_points = [[-1.0,-1.0,0.0],[1.0,-1.0,0.0],[-1.0,1.0,0.0],[1.0,1.0,0.0],[-1.0,0.5,0.0],[-1.0,0.0,0.0],[-1.0,-0.5,0.0],[-0.5,-1.0,0.0],[0.0,-1.0,0.0],[0.5,-1.0,0.0],[1.0,-0.5,0.0],[1.0,0.0,0.0],[1.0,0.5,0.0],[0.5,1.0,0.0],[0.0,1.0,0.0],[-0.5,1.0,0.0],[-0.5,0.5,0.0],[-0.5,0.0,0.0],[-0.5,-0.5,0.0],[0.0,0.5,0.0],[-0.386924147605896,0.386924147605896,0.0],[0.0,-0.5,0.0],[0.5,0.5,0.0],[0.5,0.0,0.0],[0.5,-0.5,0.0],[-0.386924147605896,0.0,0.0],[-0.386924147605896,-0.386924147605896,0.0],[0.0,0.386924147605896,0.0],[0.0,-0.386924147605896,0.0],[0.386924147605896,0.386924147605896,0.0],[0.386924147605896,0.0,0.0],[0.386924147605896,-0.386924147605896,0.0],[-0.2558380365371704,0.2558380365371704,0.0],[-0.2558380365371704,-0.2558380365371704,0.0],[0.2558380365371704,0.2558380365371704,0.0],[0.2558380365371704,-0.2558380365371704,0.0],[0.2558380365371704,0.08527934551239014,0.0],[0.2558380365371704,-0.08527934551239014,0.0],[-0.2558380365371704,-0.08527934551239014,0.0],[-0.2558380365371704,0.08527934551239014,0.0],[-0.08527934551239014,0.2558380365371704,0.0],[0.08527934551239014,0.2558380365371704,0.0],[0.08527934551239014,-0.2558380365371704,0.0],[-0.08527934551239014,-0.2558380365371704,0.0],[0.08527934551239014,0.08527934551239014,0.0],[0.08527934551239014,-0.08527934551239014,0.0],[-0.08527934551239014,0.08527934551239014,0.0],[-0.08527934551239014,-0.08527934551239014,0.0],[0.05401650816202164,0.05401650816202164,0.0],[0.05401650816202164,-0.05401650816202164,0.0],[-0.05401650816202164,0.05401650816202164,0.0],[-0.05401650816202164,-0.05401650816202164,0.0]]
        sv_edges = [[0,6],[1,9],[3,12],[2,15],[2,4],[4,5],[5,6],[0,7],[7,8],[8,9],[1,10],[10,11],[11,12],[3,13],[13,14],[14,15],[7,18],[15,16],[16,17],[17,18],[8,21],[14,19],[30,31],[29,30],[9,24],[13,22],[22,23],[23,24],[4,16],[5,17],[6,18],[16,19],[25,26],[18,21],[19,22],[20,25],[21,24],[12,22],[11,23],[10,24],[20,27],[26,28],[27,29],[28,31],[35,37],[32,39],[34,41],[33,43],[34,36],[36,37],[33,38],[38,39],[32,40],[40,41],[35,42],[42,43],[42,45],[41,44],[44,45],[43,47],[40,46],[46,47],[36,44],[37,45],[44,46],[45,47],[39,46],[38,47],[48,49],[50,51],[48,50],[49,51]]

        expected_points_with_holes = [[-1.0,-1.0,0.0],[-1.0,-0.5,0.0],[1.0,-1.0,0.0],[0.5,-1.0,0.0],[1.0,1.0,0.0],[1.0,0.5,0.0],[-1.0,1.0,0.0],[-0.5,1.0,0.0],[-1.0,0.5,0.0],[-1.0,0.0,0.0],[-0.5,-1.0,0.0],[0.0,-1.0,0.0],[1.0,-0.5,0.0],[1.0,0.0,0.0],[0.5,1.0,0.0],[0.0,1.0,0.0],[-0.5,-0.5,0.0],[-0.5,0.5,0.0],[-0.5,0.0,0.0],[0.0,-0.5,0.0],[0.0,0.5,0.0],[0.386924147605896,0.0,0.0],[0.386924147605896,-0.386924147605896,0.0],[0.386924147605896,0.386924147605896,0.0],[0.5,-0.5,0.0],[0.5,0.5,0.0],[0.5,0.0,0.0],[-0.386924147605896,0.0,0.0],[-0.386924147605896,-0.386924147605896,0.0],[-0.386924147605896,0.386924147605896,0.0],[0.0,0.386924147605896,0.0],[0.0,-0.386924147605896,0.0],[0.2558380365371704,-0.2558380365371704,0.0],[0.2558380365371704,-0.08527934551239014,0.0],[-0.2558380365371704,0.2558380365371704,0.0],[-0.2558380365371704,0.08527934551239014,0.0],[0.2558380365371704,0.2558380365371704,0.0],[0.08527934551239014,0.2558380365371704,0.0],[-0.2558380365371704,-0.2558380365371704,0.0],[-0.08527934551239014,-0.2558380365371704,0.0],[0.2558380365371704,0.08527934551239014,0.0],[-0.2558380365371704,-0.08527934551239014,0.0],[-0.08527934551239014,0.2558380365371704,0.0],[0.08527934551239014,-0.2558380365371704,0.0],[0.08527934551239014,-0.08527934551239014,0.0],[0.08527934551239014,0.08527934551239014,0.0],[-0.08527934551239014,-0.08527934551239014,0.0],[-0.08527934551239014,0.08527934551239014,0.0],[0.05401650816202164,0.05401650816202164,0.0],[0.05401650816202164,-0.05401650816202164,0.0],[-0.05401650816202164,0.05401650816202164,0.0],[-0.05401650816202164,-0.05401650816202164,0.0]]
        expected_faces_with_holes = [[1,0,10,16],[3,2,12,24],[5,4,14,25],[7,6,8,17],[8,9,18,17],[9,1,16,18],[10,11,19,16],[11,3,24,19],[12,13,26,24],[13,5,25,26],[14,15,20,25],[15,7,17,20],[29,25,20,17,18,16,22,31,28,27],[25,29,30,23,21,22,16,19,24,26],[32,33,44,43],[34,35,47,42],[36,37,45,40],[38,39,46,41],[33,40,45,44],[35,41,46,47],[37,42,47,45],[39,43,44,46],[45,50,48,49,46,44],[50,45,47,46,49,51]]
        expected_points_without_holes = [[-1.0,-1.0,0.0],[-1.0,-0.5,0.0],[1.0,-1.0,0.0],[0.5,-1.0,0.0],[1.0,1.0,0.0],[1.0,0.5,0.0],[-1.0,1.0,0.0],[-0.5,1.0,0.0],[-1.0,0.5,0.0],[-1.0,0.0,0.0],[-0.5,-1.0,0.0],[0.0,-1.0,0.0],[1.0,-0.5,0.0],[1.0,0.0,0.0],[0.5,1.0,0.0],[0.0,1.0,0.0],[-0.5,-0.5,0.0],[-0.5,0.5,0.0],[-0.5,0.0,0.0],[0.0,-0.5,0.0],[0.0,0.5,0.0],[0.386924147605896,0.0,0.0],[0.386924147605896,-0.386924147605896,0.0],[0.386924147605896,0.386924147605896,0.0],[0.5,-0.5,0.0],[0.5,0.5,0.0],[0.5,0.0,0.0],[-0.386924147605896,0.0,0.0],[-0.386924147605896,-0.386924147605896,0.0],[-0.386924147605896,0.386924147605896,0.0],[0.0,0.386924147605896,0.0],[0.0,-0.386924147605896,0.0],[0.2558380365371704,-0.2558380365371704,0.0],[0.2558380365371704,-0.08527934551239014,0.0],[-0.2558380365371704,0.2558380365371704,0.0],[-0.2558380365371704,0.08527934551239014,0.0],[0.2558380365371704,0.2558380365371704,0.0],[0.08527934551239014,0.2558380365371704,0.0],[-0.2558380365371704,-0.2558380365371704,0.0],[-0.08527934551239014,-0.2558380365371704,0.0],[0.2558380365371704,0.08527934551239014,0.0],[-0.2558380365371704,-0.08527934551239014,0.0],[-0.08527934551239014,0.2558380365371704,0.0],[0.08527934551239014,-0.2558380365371704,0.0],[0.08527934551239014,-0.08527934551239014,0.0],[0.08527934551239014,0.08527934551239014,0.0],[-0.08527934551239014,-0.08527934551239014,0.0],[-0.08527934551239014,0.08527934551239014,0.0],[0.05401650816202164,0.05401650816202164,0.0],[0.05401650816202164,-0.05401650816202164,0.0],[-0.05401650816202164,0.05401650816202164,0.0],[-0.05401650816202164,-0.05401650816202164,0.0]]
        expected_faces_without_holes = [[1,0,10,16],[3,2,12,24],[5,4,14,25],[7,6,8,17],[8,9,18,17],[9,1,16,18],[10,11,19,16],[11,3,24,19],[12,13,26,24],[13,5,25,26],[14,15,20,25],[15,7,17,20],[17,18,16,19,24,26,25,20],[22,21,23,30,29,27,28,31],[32,33,44,43],[34,35,47,42],[36,37,45,40],[38,39,46,41],[33,40,45,44],[35,41,46,47],[37,42,47,45],[39,43,44,46],[44,45,47,46],[49,48,50,51]]

        result_points_with_holes, result_faces_with_holes = edges_to_faces(sv_points, sv_edges, True, False, 5)
        result_points_without_holes, result_faces_without_holes = edges_to_faces(sv_points, sv_edges, False, False, 5)
        self.assert_sverchok_data_equal(expected_points_with_holes, result_points_with_holes, precision=5)
        self.assert_sverchok_data_equal(expected_faces_with_holes, result_faces_with_holes)
        self.assert_sverchok_data_equal(expected_points_without_holes, result_points_without_holes, precision=5)
        self.assert_sverchok_data_equal(expected_faces_without_holes, result_faces_without_holes)


class MergeMesh2DLightTest(SverchokTestCase):

    def test_merge_rings_and_square(self):
        # https://user-images.githubusercontent.com/28003269/68269307-c0415400-0072-11ea-9858-365e8baca476.png
        sv_points = [[-0.9050000905990601,-1.7000000476837158,0.0],[-0.5850001573562622,-1.7000000476837158,0.0],[-1.7450001239776611,-0.8600000143051147,0.0],[-1.7450001239776611,-0.5400000810623169,0.0],[-2.5850000381469727,-1.7000000476837158,0.0],[-2.9050002098083496,-1.7000000476837158,0.0],[-1.7450001239776611,-2.5399999618530273,0.0],[-1.7450001239776611,-2.8600001335144043,0.0],[2.5850000381469727,-1.7000000476837158,0.0],[2.9050002098083496,-1.7000000476837158,0.0],[1.7450001239776611,-0.8600000143051147,0.0],[1.7450001239776611,-0.5400000810623169,0.0],[0.9050000905990601,-1.7000000476837158,0.0],[0.5850001573562622,-1.7000000476837158,0.0],[1.7450001239776611,-2.5399999618530273,0.0],[1.7450001239776611,-2.8600001335144043,0.0],[-0.9050000905990601,1.7000000476837158,0.0],[-0.5850001573562622,1.7000000476837158,0.0],[-1.7450001239776611,2.5399999618530273,0.0],[-1.7450001239776611,2.8600001335144043,0.0],[-2.5850000381469727,1.7000000476837158,0.0],[-2.9050002098083496,1.7000000476837158,0.0],[-1.7450001239776611,0.8600000143051147,0.0],[-1.7450001239776611,0.5400000810623169,0.0],[2.5850000381469727,1.7000000476837158,0.0],[2.9050002098083496,1.7000000476837158,0.0],[1.7450001239776611,2.5399999618530273,0.0],[1.7450001239776611,2.8600001335144043,0.0],[0.9050000905990601,1.7000000476837158,0.0],[0.5850001573562622,1.7000000476837158,0.0],[1.7450001239776611,0.8600000143051147,0.0],[1.7450001239776611,0.5400000810623169,0.0],[0.8400000333786011,0.0,0.0],[1.159999966621399,0.0,0.0],[5.143516582334358e-17,0.8400000333786011,0.0],[7.102951612645076e-17,1.159999966621399,0.0],[-0.8400000333786011,1.0287033164668716e-16,0.0],[-1.159999966621399,1.420590322529015e-16,0.0],[-1.543055007787532e-16,-0.8400000333786011,0.0],[-2.1308854176190737e-16,-1.159999966621399,0.0],[-1.745000123977661,-1.7000000476837158,0.0],[1.745000123977661,-1.7000000476837158,0.0],[-1.745000123977661,1.7000000476837158,0.0],[1.745000123977661,1.7000000476837158,0.0]]
        sv_faces = [[0,2,3,1],[2,4,5,3],[4,6,7,5],[6,0,1,7],[8,10,11,9],[10,12,13,11],[12,14,15,13],[14,8,9,15],[16,18,19,17],[18,20,21,19],[20,22,23,21],[22,16,17,23],[24,26,27,25],[26,28,29,27],[28,30,31,29],[30,24,25,31],[32,34,35,33],[34,36,37,35],[36,38,39,37],[38,32,33,39],[40,41,43,42]]

        expected_points = [[-0.5850001573562622,-1.7000000476837158,0.0],[-1.7450001239776611,-0.5400000810623169,0.0],[-1.7450001239776611,-0.8600000143051147,0.0],[-0.9050000905990601,-1.7000000476837158,0.0],[-2.9050002098083496,-1.7000000476837158,0.0],[-2.5850000381469727,-1.7000000476837158,0.0],[-1.7450001239776611,-2.8600001335144043,0.0],[-1.7450001239776611,-2.5399999618530273,0.0],[2.9050002098083496,-1.7000000476837158,0.0],[1.7450001239776611,-0.5400000810623169,0.0],[1.7450001239776611,-0.8600000143051147,0.0],[2.5850000381469727,-1.7000000476837158,0.0],[0.5850001573562622,-1.7000000476837158,0.0],[0.9050000905990601,-1.7000000476837158,0.0],[1.7450001239776611,-2.8600001335144043,0.0],[1.7450001239776611,-2.5399999618530273,0.0],[-0.5850001573562622,1.7000000476837158,0.0],[-1.7450001239776611,2.8600001335144043,0.0],[-1.7450001239776611,2.5399999618530273,0.0],[-0.9050000905990601,1.7000000476837158,0.0],[-2.9050002098083496,1.7000000476837158,0.0],[-2.5850000381469727,1.7000000476837158,0.0],[-1.7450001239776611,0.5400000810623169,0.0],[-1.7450001239776611,0.8600000143051147,0.0],[2.9050002098083496,1.7000000476837158,0.0],[1.7450001239776611,2.8600001335144043,0.0],[1.7450001239776611,2.5399999618530273,0.0],[2.5850000381469727,1.7000000476837158,0.0],[0.5850001573562622,1.7000000476837158,0.0],[0.9050000905990601,1.7000000476837158,0.0],[1.7450001239776611,0.5400000810623169,0.0],[1.7450001239776611,0.8600000143051147,0.0],[1.159999966621399,0.0,0.0],[7.102951612645076e-17,1.159999966621399,0.0],[5.143516582334358e-17,0.8400000333786011,0.0],[0.8400000333786011,0.0,0.0],[-1.159999966621399,1.420590322529015e-16,0.0],[-0.8400000333786011,1.0287033164668716e-16,0.0],[-2.1308854176190737e-16,-1.159999966621399,0.0],[-1.543055007787532e-16,-0.8400000333786011,0.0],[-1.745000123977661,-1.7000000476837158,0.0],[1.745000123977661,-1.7000000476837158,0.0],[1.745000123977661,1.7000000476837158,0.0],[-1.745000123977661,1.7000000476837158,0.0]]
        expected_faces = [[0,1,2,3],[1,4,5,2],[4,6,7,5],[6,0,3,7],[8,9,10,11],[9,12,13,10],[12,14,15,13],[14,8,11,15],[16,17,18,19],[17,20,21,18],[20,22,23,21],[22,16,19,23],[24,25,26,27],[25,28,29,26],[28,30,31,29],[30,24,27,31],[32,33,34,35],[33,36,37,34],[36,38,39,37],[38,32,35,39],[40,3,2],[41,10,13],[42,29,31],[43,23,19],[33,28,16,22,1,0,38,36],[28,33,32,38,0,12,9,30],[35,34,37,39]]
        expected_face_index_mask = [0,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19,20,20,20,20,20,20,20]

        result_points, result_faces, result_face_index_mask, _ = merge_mesh_light(sv_points, sv_faces, True, False, 5)
        self.assert_sverchok_data_equal(expected_points, result_points, precision=5)
        self.assert_sverchok_data_equal(expected_faces, result_faces)
        self.assert_sverchok_data_equal(expected_face_index_mask, result_face_index_mask)

    def test_merge_rings_and_square_turned(self):
        sv_points = [[-0.2840900421142578,-1.8835808038711548,-0.283621609210968],[-0.046072930097579956,-1.7416378259658813,-0.44362157583236694],[-1.1011228561401367,-1.523825764656067,0.500109076499939],[-1.17435622215271,-1.24483323097229,0.6386731863021851],[-1.5336802005767822,-2.628781795501709,0.5563783645629883],[-1.7716975212097168,-2.7707247734069824,0.7163784503936768],[-0.7166475653648376,-2.988536834716797,-0.22735226154327393],[-0.6434141397476196,-3.2675294876098633,-0.36591637134552],[2.3117849826812744,-0.3355146646499634,-2.0286216735839844],[2.549802303314209,-0.1935715675354004,-2.188621759414673],[1.4947521686553955,0.024240493774414062,-1.2448910474777222],[1.4215188026428223,0.3032330572605133,-1.106326937675476],[1.0621947050094604,-1.080715537071228,-1.1886217594146729],[0.824177622795105,-1.2226585149765015,-1.0286216735839844],[1.8792273998260498,-1.4404706954956055,-1.972352385520935],[1.9524608850479126,-1.7194633483886719,-2.1109166145324707],[-1.0621947050094604,1.080715537071228,1.1886217594146729],[-0.824177622795105,1.2226585149765015,1.0286216735839844],[-1.8792273998260498,1.4404706954956055,1.972352385520935],[-1.9524608850479126,1.7194633483886719,2.1109166145324707],[-2.3117849826812744,0.3355146646499634,2.0286216735839844],[-2.549802303314209,0.1935715675354004,2.188621759414673],[-1.4947521686553955,-0.024240493774414062,1.2448910474777222],[-1.4215188026428223,-0.3032330572605133,1.106326937675476],[1.5336802005767822,2.628781795501709,-0.5563783645629883],[1.7716975212097168,2.7707247734069824,-0.7163784503936768],[0.7166475653648376,2.988536834716797,0.22735226154327393],[0.6434141397476196,3.2675294876098633,0.36591637134552],[0.2840900421142578,1.8835808038711548,0.283621609210968],[0.046072930097579956,1.7416378259658813,0.44362157583236694],[1.1011228561401367,1.523825764656067,-0.500109076499939],[1.17435622215271,1.24483323097229,-0.6386731863021851],[0.624795138835907,0.3726004660129547,-0.42000001668930054],[0.8628122210502625,0.5145434737205505,-0.5799999833106995],[-0.19223763048648834,0.7323555946350098,0.36373069882392883],[-0.2654709815979004,1.0113481283187866,0.5022947192192078],[-0.624795138835907,-0.3726004660129547,0.42000001668930054],[-0.8628122210502625,-0.5145434737205505,0.5799999833106995],[0.19223763048648834,-0.7323555946350098,-0.36373069882392883],[0.2654709815979004,-1.0113481283187866,-0.5022947192192078],[-0.9088851809501648,-2.256181240081787,0.13637840747833252],[1.6869897842407227,-0.7081150412559509,-1.6086217164993286],[-1.6869897842407227,0.7081150412559509,1.6086217164993286],[0.9088851809501648,2.256181240081787,-0.13637840747833252]]
        sv_faces = [[0,2,3,1],[2,4,5,3],[4,6,7,5],[6,0,1,7],[8,10,11,9],[10,12,13,11],[12,14,15,13],[14,8,9,15],[16,18,19,17],[18,20,21,19],[20,22,23,21],[22,16,17,23],[24,26,27,25],[26,28,29,27],[28,30,31,29],[30,24,25,31],[32,34,35,33],[34,36,37,35],[36,38,39,37],[38,32,33,39],[40,41,43,42]]

        expected_points = [[-0.046072930097579956,-1.7416378259658813,-0.44362157583236694],[-1.17435622215271,-1.24483323097229,0.6386731863021851],[-1.1011228561401367,-1.523825764656067,0.500109076499939],[-0.2840900421142578,-1.8835808038711548,-0.283621609210968],[-1.7716975212097168,-2.7707247734069824,0.7163784503936768],[-1.5336802005767822,-2.628781795501709,0.5563783645629883],[-0.6434141397476196,-3.2675294876098633,-0.36591637134552],[-0.7166475653648376,-2.988536834716797,-0.22735226154327393],[2.549802303314209,-0.1935715675354004,-2.188621759414673],[1.4215188026428223,0.3032330572605133,-1.106326937675476],[1.4947521686553955,0.024240493774414062,-1.2448910474777222],[2.3117849826812744,-0.3355146646499634,-2.0286216735839844],[0.824177622795105,-1.2226585149765015,-1.0286216735839844],[1.0621947050094604,-1.080715537071228,-1.1886217594146729],[1.9524608850479126,-1.7194633483886719,-2.1109166145324707],[1.8792273998260498,-1.4404706954956055,-1.972352385520935],[-0.824177622795105,1.2226585149765015,1.0286216735839844],[-1.9524608850479126,1.7194633483886719,2.1109166145324707],[-1.8792273998260498,1.4404706954956055,1.972352385520935],[-1.0621947050094604,1.080715537071228,1.1886217594146729],[-2.549802303314209,0.1935715675354004,2.188621759414673],[-2.3117849826812744,0.3355146646499634,2.0286216735839844],[-1.4215188026428223,-0.3032330572605133,1.106326937675476],[-1.4947521686553955,-0.024240493774414062,1.2448910474777222],[1.7716975212097168,2.7707247734069824,-0.7163784503936768],[0.6434141397476196,3.2675294876098633,0.36591637134552],[0.7166475653648376,2.988536834716797,0.22735226154327393],[1.5336802005767822,2.628781795501709,-0.5563783645629883],[0.046072930097579956,1.7416378259658813,0.44362157583236694],[0.2840900421142578,1.8835808038711548,0.283621609210968],[1.17435622215271,1.24483323097229,-0.6386731863021851],[1.1011228561401367,1.523825764656067,-0.500109076499939],[0.8628122210502625,0.5145434737205505,-0.5799999833106995],[-0.2654709815979004,1.0113481283187866,0.5022947192192078],[-0.19223763048648834,0.7323555946350098,0.36373069882392883],[0.624795138835907,0.3726004660129547,-0.42000001668930054],[-0.8628122210502625,-0.5145434737205505,0.5799999833106995],[-0.624795138835907,-0.3726004660129547,0.42000001668930054],[0.2654709815979004,-1.0113481283187866,-0.5022947192192078],[0.19223763048648834,-0.7323555946350098,-0.36373069882392883],[-0.9088851809501648,-2.256181240081787,0.13637840747833252],[1.6869897842407227,-0.7081150412559509,-1.6086217164993286],[0.9088851809501648,2.256181240081787,-0.13637840747833252],[-1.6869897842407227,0.7081150412559509,1.6086217164993286]]
        expected_faces = [[0,1,2,3],[1,4,5,2],[4,6,7,5],[6,0,3,7],[8,9,10,11],[9,12,13,10],[12,14,15,13],[14,8,11,15],[16,17,18,19],[17,20,21,18],[20,22,23,21],[22,16,19,23],[24,25,26,27],[25,28,29,26],[28,30,31,29],[30,24,27,31],[32,33,34,35],[33,36,37,34],[36,38,39,37],[38,32,35,39],[40,3,2],[41,10,13],[42,29,31],[43,23,19],[33,16,22,1,0,12,38,36],[16,33,32,38,12,9,30,28],[35,34,37,39]]

        result_points, result_faces, _, _ = merge_mesh_light(sv_points, sv_faces, False, False, accuracy=5)
        self.assert_sverchok_data_equal(expected_points, result_points, precision=5)
        self.assert_sverchok_data_equal(expected_faces, result_faces)

    def test_merge_random_squares(self):
        # https://user-images.githubusercontent.com/28003269/68272228-647bc880-007c-11ea-8632-ecd9630914d8.png
        sv_points = [[1.8655242919921875,0.39576175808906555,0.0],[2.657252788543701,0.39576175808906555,0.0],[1.8655242919921875,1.3890951871871948,0.0],[2.657252788543701,1.3890951871871948,0.0],[0.3097015619277954,-0.8997174501419067,0.0],[2.8397016525268555,-0.8997174501419067,0.0],[0.3097015619277954,1.9602826833724976,0.0],[2.8397016525268555,1.9602826833724976,0.0],[2.324446439743042,0.7085106372833252,0.0],[2.384446382522583,0.7085106372833252,0.0],[2.324446439743042,0.7685105800628662,0.0],[2.384446382522583,0.7685105800628662,0.0],[0.1693570911884308,0.9782793521881104,0.0],[0.7705341577529907,0.9782793521881104,0.0],[0.1693570911884308,1.9716126918792725,0.0],[0.7705341577529907,1.9716126918792725,0.0],[1.8147730827331543,1.5254225730895996,0.0],[2.3079166412353516,1.5254225730895996,0.0],[1.8147730827331543,1.5854225158691406,0.0],[2.3079166412353516,1.5854225158691406,0.0],[1.734034776687622,1.3174173831939697,0.0],[2.6335136890411377,1.3174173831939697,0.0],[1.734034776687622,1.3774173259735107,0.0],[2.6335136890411377,1.3774173259735107,0.0]]
        sv_faces = [[0,1,3,2],[4,5,7,6],[8,9,11,10],[12,13,15,14],[16,17,19,18],[20,21,23,22]]

        expected_points = [[1.8655242919921875,0.39576175808906555,0.0],[2.657252788543701,0.39576175808906555,0.0],[2.657252788543701,1.3890951871871948,0.0],[1.8655242919921875,1.3890951871871948,0.0],[0.3097015619277954,-0.8997174501419067,0.0],[2.8397016525268555,-0.8997174501419067,0.0],[2.8397016525268555,1.9602826833724976,0.0],[0.3097015619277954,1.9602826833724976,0.0],[2.324446439743042,0.7085106372833252,0.0],[2.384446382522583,0.7085106372833252,0.0],[2.384446382522583,0.7685105800628662,0.0],[2.324446439743042,0.7685105800628662,0.0],[0.1693570911884308,0.9782793521881104,0.0],[0.7705341577529907,0.9782793521881104,0.0],[0.7705341577529907,1.9716126918792725,0.0],[0.1693570911884308,1.9716126918792725,0.0],[1.8147730827331543,1.5254225730895996,0.0],[2.3079166412353516,1.5254225730895996,0.0],[2.3079166412353516,1.5854225158691406,0.0],[1.8147730827331543,1.5854225158691406,0.0],[1.734034776687622,1.3174173831939697,0.0],[2.6335136890411377,1.3174173831939697,0.0],[2.6335136890411377,1.3774173259735107,0.0],[1.734034776687622,1.3774173259735107,0.0],[0.7705341577529907,1.9602826833724976,0],[1.8655242919921875,1.3774173259735107,0],[1.8655242919921875,1.3174173831939697,0],[0.3097015619277954,0.9782793521881102,0]]
        expected_faces = [[21,11,10,9,0,1,2,3,25,22],[6,19,18,17,3,2,1,4,5],[19,6,24,13,0,26,20,23,3,17,16],[7,27,13,24],[8,9,10,11],[12,27,7,24,14,15],[16,17,18,19],[20,26,25,23],[21,22,25,26],[0,13,27,4,1],[11,21,26,0,9,8],[3,23,25]]
        expected_face_index_mask = [0,1,1,1,0,3,1,1,0,1,0,1]

        result_points, result_faces, result_face_index_mask, _ = merge_mesh_light(sv_points, sv_faces, True, False, 5)
        self.assert_sverchok_data_equal(expected_points, result_points, precision=5)
        self.assert_sverchok_data_equal(expected_faces, result_faces)
        self.assert_sverchok_data_equal(expected_face_index_mask, result_face_index_mask)

    def test_number_overlapping_of_flour(self):
        # https://user-images.githubusercontent.com/28003269/69257088-e7039c80-0bd3-11ea-8883-3ef1048e7b1a.png
        sv_points = [[1.0,1.2000000476837158,0.0],[0.80901700258255,1.787785291671753,0.0],[0.30901700258255005,2.1510565280914307,0.0],[-0.30901700258255005,2.1510565280914307,0.0],[-0.80901700258255,1.787785291671753,0.0],[-1.0,1.2000000476837158,0.0],[-0.80901700258255,0.6122148036956787,0.0],[-0.30901700258255005,0.2489435374736786,0.0],[0.30901700258255005,0.2489435374736786,0.0],[0.80901700258255,0.6122148036956787,0.0],[0.10367470979690552,1.558605670928955,0.0],[-0.3963252902030945,1.9218769073486328,0.0],[-1.0143592357635498,1.9218769073486328,0.0],[-1.5143592357635498,1.558605670928955,0.0],[-1.7053422927856445,0.970820426940918,0.0],[-1.5143592357635498,0.38303518295288086,0.0],[-1.0143593549728394,0.019763916730880737,0.0],[-0.39632534980773926,0.019763916730880737,0.0],[0.10367466509342194,0.38303518295288086,0.0],[0.29465770721435547,0.970820426940918,0.0],[-0.8322505950927734,1.321877121925354,0.0],[-1.450284719467163,1.3218772411346436,0.0],[-1.9502848386764526,0.9586060643196106,0.0],[-2.141268014907837,0.3708208203315735,0.0],[-1.9502849578857422,-0.2169644832611084,0.0],[-1.4502849578857422,-0.5802358388900757,0.0],[-0.8322509527206421,-0.5802359580993652,0.0],[-0.3322509229183197,-0.21696478128433228,0.0],[-0.1412678062915802,0.37082046270370483,0.0],[-0.3322506844997406,0.9586057662963867,0.0],[-1.4502848386764526,0.58023601770401,0.0],[-1.9502848386764526,0.21696478128433228,0.0],[-2.141267776489258,-0.37082046270370483,0.0],[-1.9502848386764526,-0.9586057066917419,0.0],[-1.4502848386764526,-1.3218770027160645,0.0],[-0.832250714302063,-1.3218770027160645,0.0],[-0.33225080370903015,-0.9586056470870972,0.0],[-0.14126785099506378,-0.37082040309906006,0.0],[-0.3322508931159973,0.21696482598781586,0.0],[-0.8322508335113525,0.5802360773086548,0.0],[-1.5143592357635498,-0.38303565979003906,0.0],[-1.7053420543670654,-0.970820963382721,0.0],[-1.5143587589263916,-1.5586061477661133,0.0],[-1.0143587589263916,-1.921877145767212,0.0],[-0.39632469415664673,-1.921877145767212,0.0],[0.10367512702941895,-1.558605670928955,0.0],[0.29465797543525696,-0.9708203673362732,0.0],[0.10367478430271149,-0.38303515315055847,0.0],[-0.39632534980773926,-0.019764035940170288,0.0],[-1.0143593549728394,-0.019764244556427002,0.0],[-1.0000001192092896,-1.1999999284744263,0.0],[-0.8090171813964844,-1.7877851724624634,0.0],[-0.3090171813964844,-2.1510565280914307,0.0],[0.3090168237686157,-2.1510565280914307,0.0],[0.8090168237686157,-1.7877854108810425,0.0],[0.9999998807907104,-1.2000001668930054,0.0],[0.8090169429779053,-0.6122148633003235,0.0],[0.30901697278022766,-0.24894356727600098,0.0],[-0.30901703238487244,-0.2489435076713562,0.0],[-0.8090170621871948,-0.6122147440910339,0.0],[-0.10367512702941895,-1.558605670928955,0.0],[0.39632469415664673,-1.921877145767212,0.0],[1.0143587589263916,-1.921877145767212,0.0],[1.5143587589263916,-1.5586061477661133,0.0],[1.7053420543670654,-0.970820963382721,0.0],[1.5143592357635498,-0.38303565979003906,0.0],[1.0143593549728394,-0.019764244556427002,0.0],[0.39632534980773926,-0.019764035940170288,0.0],[-0.10367478430271149,-0.38303515315055847,0.0],[-0.29465797543525696,-0.9708203673362732,0.0],[0.832250714302063,-1.3218770027160645,0.0],[1.4502848386764526,-1.3218770027160645,0.0],[1.9502848386764526,-0.9586057066917419,0.0],[2.141267776489258,-0.37082046270370483,0.0],[1.9502848386764526,0.21696478128433228,0.0],[1.4502848386764526,0.58023601770401,0.0],[0.8322508335113525,0.5802360773086548,0.0],[0.3322508931159973,0.21696482598781586,0.0],[0.14126785099506378,-0.37082040309906006,0.0],[0.33225080370903015,-0.9586056470870972,0.0],[1.4502849578857422,-0.5802358388900757,0.0],[1.9502849578857422,-0.2169644832611084,0.0],[2.141268014907837,0.3708208203315735,0.0],[1.9502848386764526,0.9586060643196106,0.0],[1.450284719467163,1.3218772411346436,0.0],[0.8322505950927734,1.321877121925354,0.0],[0.3322506844997406,0.9586057662963867,0.0],[0.1412678062915802,0.37082046270370483,0.0],[0.3322509229183197,-0.21696478128433228,0.0],[0.8322509527206421,-0.5802359580993652,0.0],[1.514359474182129,0.3830344080924988,0.0],[1.7053427696228027,0.9708195924758911,0.0],[1.5143600702285767,1.5586049556732178,0.0],[1.0143603086471558,1.9218764305114746,0.0],[0.3963262438774109,1.9218767881393433,0.0],[-0.10367387533187866,1.558605670928955,0.0],[-0.2946572005748749,0.9708206057548523,0.0],[-0.10367447137832642,0.38303524255752563,0.0],[0.39632534980773926,0.019763708114624023,0.0],[1.0143593549728394,0.019763410091400146,0.0]]
        sv_faces = [[0,1,2,3,4,5,6,7,8,9],[10,11,12,13,14,15,16,17,18,19],[20,21,22,23,24,25,26,27,28,29],[30,31,32,33,34,35,36,37,38,39],[40,41,42,43,44,45,46,47,48,49],[50,51,52,53,54,55,56,57,58,59],[60,61,62,63,64,65,66,67,68,69],[70,71,72,73,74,75,76,77,78,79],[80,81,82,83,84,85,86,87,88,89],[90,91,92,93,94,95,96,97,98,99]]

        expected_points = [[1.0,1.2000000476837158,0.0],[0.80901700258255,1.787785291671753,0.0],[0.30901700258255005,2.1510565280914307,0.0],[-0.30901700258255005,2.1510565280914307,0.0],[-0.80901700258255,1.787785291671753,0.0],[-1.0,1.2000000476837158,0.0],[-0.80901700258255,0.6122148036956787,0.0],[-0.30901700258255005,0.2489435374736786,0.0],[0.30901700258255005,0.2489435374736786,0.0],[0.80901700258255,0.6122148036956787,0.0],[0.10367470979690552,1.558605670928955,0.0],[-0.3963252902030945,1.9218769073486328,0.0],[-1.0143592357635498,1.9218769073486328,0.0],[-1.5143592357635498,1.558605670928955,0.0],[-1.7053422927856445,0.970820426940918,0.0],[-1.5143592357635498,0.38303518295288086,0.0],[-1.0143593549728394,0.019763916730880737,0.0],[-0.39632534980773926,0.019763916730880737,0.0],[0.10367466509342194,0.38303518295288086,0.0],[0.29465770721435547,0.970820426940918,0.0],[-0.8322505950927734,1.321877121925354,0.0],[-1.450284719467163,1.3218772411346436,0.0],[-1.9502848386764526,0.9586060643196106,0.0],[-2.141268014907837,0.3708208203315735,0.0],[-1.9502849578857422,-0.2169644832611084,0.0],[-1.4502849578857422,-0.5802358388900757,0.0],[-0.8322509527206421,-0.5802359580993652,0.0],[-0.3322509229183197,-0.21696478128433228,0.0],[-0.1412678062915802,0.37082046270370483,0.0],[-0.3322506844997406,0.9586057662963867,0.0],[-1.4502848386764526,0.58023601770401,0.0],[-1.9502848386764526,0.21696478128433228,0.0],[-2.141267776489258,-0.37082046270370483,0.0],[-1.9502848386764526,-0.9586057066917419,0.0],[-1.4502848386764526,-1.3218770027160645,0.0],[-0.832250714302063,-1.3218770027160645,0.0],[-0.33225080370903015,-0.9586056470870972,0.0],[-0.14126785099506378,-0.37082040309906006,0.0],[-0.3322508931159973,0.21696482598781586,0.0],[-0.8322508335113525,0.5802360773086548,0.0],[-1.5143592357635498,-0.38303565979003906,0.0],[-1.7053420543670654,-0.970820963382721,0.0],[-1.5143587589263916,-1.5586061477661133,0.0],[-1.0143587589263916,-1.921877145767212,0.0],[-0.39632469415664673,-1.921877145767212,0.0],[0.10367512702941895,-1.558605670928955,0.0],[0.29465797543525696,-0.9708203673362732,0.0],[0.10367478430271149,-0.38303515315055847,0.0],[-0.39632534980773926,-0.019764035940170288,0.0],[-1.0143593549728394,-0.019764244556427002,0.0],[-1.0000001192092896,-1.1999999284744263,0.0],[-0.8090171813964844,-1.7877851724624634,0.0],[-0.3090171813964844,-2.1510565280914307,0.0],[0.3090168237686157,-2.1510565280914307,0.0],[0.8090168237686157,-1.7877854108810425,0.0],[0.9999998807907104,-1.2000001668930054,0.0],[0.8090169429779053,-0.6122148633003235,0.0],[0.30901697278022766,-0.24894356727600098,0.0],[-0.30901703238487244,-0.2489435076713562,0.0],[-0.8090170621871948,-0.6122147440910339,0.0],[-0.10367512702941895,-1.558605670928955,0.0],[0.39632469415664673,-1.921877145767212,0.0],[1.0143587589263916,-1.921877145767212,0.0],[1.5143587589263916,-1.5586061477661133,0.0],[1.7053420543670654,-0.970820963382721,0.0],[1.5143592357635498,-0.38303565979003906,0.0],[1.0143593549728394,-0.019764244556427002,0.0],[0.39632534980773926,-0.019764035940170288,0.0],[-0.10367478430271149,-0.38303515315055847,0.0],[-0.29465797543525696,-0.9708203673362732,0.0],[0.832250714302063,-1.3218770027160645,0.0],[1.4502848386764526,-1.3218770027160645,0.0],[1.9502848386764526,-0.9586057066917419,0.0],[2.141267776489258,-0.37082046270370483,0.0],[1.9502848386764526,0.21696478128433228,0.0],[1.4502848386764526,0.58023601770401,0.0],[0.8322508335113525,0.5802360773086548,0.0],[0.3322508931159973,0.21696482598781586,0.0],[0.14126785099506378,-0.37082040309906006,0.0],[0.33225080370903015,-0.9586056470870972,0.0],[1.4502849578857422,-0.5802358388900757,0.0],[1.9502849578857422,-0.2169644832611084,0.0],[2.141268014907837,0.3708208203315735,0.0],[1.9502848386764526,0.9586060643196106,0.0],[1.450284719467163,1.3218772411346436,0.0],[0.8322505950927734,1.321877121925354,0.0],[0.3322506844997406,0.9586057662963867,0.0],[0.1412678062915802,0.37082046270370483,0.0],[0.3322509229183197,-0.21696478128433228,0.0],[0.8322509527206421,-0.5802359580993652,0.0],[1.514359474182129,0.3830344080924988,0.0],[1.7053427696228027,0.9708195924758911,0.0],[1.5143600702285767,1.5586049556732178,0.0],[1.0143603086471558,1.9218764305114746,0.0],[0.3963262438774109,1.9218767881393433,0.0],[-0.10367387533187866,1.558605670928955,0.0],[-0.2946572005748749,0.9708206057548523,0.0],[-0.10367447137832642,0.38303524255752563,0.0],[0.39632534980773926,0.019763708114624023,0.0],[1.0143593549728394,0.019763410091400146,0.0],[-0.6244557130644786,1.9218769073486328,0],[0.6244560588357492,1.921876656131119,0],[4.4660207465354304e-07,1.6339294264803255,0],[-0.9603997311706439,1.3218771466433539,0],[0.9603997311706439,1.3218771466433539,0],[-1.6348459826753148,1.1877856985443698,0],[1.6348464434616987,1.1877853637636262,0],[-1.5539594324843593,0.5049120219481978,0],[1.553959872296601,0.504911702405924,0],[7.951958910472636e-08,0.30771118924481605,0],[-0.18086804339074125,0.2489435374736786,0],[-0.0808866558032044,0.2489435374736786,0],[0.08088672378186391,0.24894353747367856,0],[0.18086804339074133,0.2489435374736786,0],[-0.21176401439581255,0.1538555727560657,0],[0.2117640389195207,0.1538554972798979,0],[-0.29265069091540447,0.09508796370795398,0],[0.292650658367919,0.09508786353711694,0],[-2.0207809750844525,-9.114529240243115e-08,0],[-0.2617547408145704,2.864209562698558e-08,0],[0.26175474081457045,2.8642095704967685e-08,0],[2.0207809750844525,-9.114529240243115e-08,0],[-0.29265073358435095,-0.09508800306036488,0],[0.292650733584351,-0.09508800306036484,0],[-0.21176400019048505,-0.15385561531263992,0],[0.2117640001904851,-0.15385561531263986,0],[-0.1808680599503668,-0.24894352003034276,0],[-0.08088662602032483,-0.2489435296727861,0],[0.0808866045462904,-0.24894354527457474,0],[0.18086804861499586,-0.24894355491701903,0],[-4.7992052235983476e-17,-0.30771106391366954,0],[-1.5539592331626186,-0.5049120498427124,0],[1.5539592331626186,-0.5049120498427124,0],[-1.6348458710660827,-1.1877855518525289,0],[1.6348458710660827,-1.1877855518525289,0],[-0.9603998707703014,-1.3218770027160645,0],[0.9603996851006303,-1.3218770027160645,0],[0.0,-1.6339301304671223,0],[-0.6244554602107031,-1.921877145767212,0],[0.6244553096082286,-1.921877145767212,0]]
        expected_faces = [[0,104,85,86,87,113,8,9],[1,101,94,102,10,19,18,109,112,113,87,86,85,104],[2,3,100,11,102,94,101],[4,103,20,29,28,110,111,109,97,96,95,102,11,100],[5,6,7,110,28,29,20,103],[10,102,95,96,97,109,18,19],[12,13,105,21,103,4,100],[14,107,30,39,38,116,114,110,7,6,5,103,21,105],[15,16,17,116,38,39,30,107],[22,23,118,31,107,14,105],[24,131,40,49,48,122,119,116,17,16,15,107,31,118],[25,26,27,122,48,49,40,131],[32,33,133,41,131,24,118],[34,135,50,59,58,126,124,122,27,26,25,131,41,133],[35,36,37,126,58,59,50,135],[42,43,138,51,135,34,133],[44,137,60,69,68,130,127,126,37,36,35,135,51,138],[45,46,47,130,68,69,60,137],[52,53,139,61,137,44,138],[54,136,70,79,78,129,128,130,47,46,45,137,61,139],[55,56,57,129,78,79,70,136],[62,63,134,71,136,54,139],[64,132,80,89,88,123,125,129,57,56,55,136,71,134],[65,66,67,123,88,89,80,132],[72,73,121,81,132,64,134],[74,108,90,99,98,117,120,123,67,66,65,132,81,121],[75,76,77,117,98,99,90,108],[82,83,106,91,108,74,121],[84,104,0,9,8,113,115,117,77,76,75,108,91,106],[92,93,101,1,104,84,106],[109,111,112],[110,114,111],[112,115,113],[114,116,119],[115,120,117],[119,122,124],[120,125,123],[124,126,127],[125,128,129],[127,130,128]]
        expected_overlapping_mask = [2,1,0,1,2,2,0,1,2,0,1,2,0,1,2,0,1,2,0,1,2,0,1,2,0,1,2,0,1,0,0,0,0,0,0,0,0,0,0,0]

        result_points, result_faces, _, result_overlapping_mask = merge_mesh_light(sv_points, sv_faces, False, True, 5)
        self.assert_sverchok_data_equal(expected_points, result_points, precision=5)
        self.assert_sverchok_data_equal(expected_faces, result_faces)
        self.assert_sverchok_data_equal(expected_overlapping_mask, result_overlapping_mask)


class CropMesh2DTest(SverchokTestCase):

    def test_crop_ring_by_squares(self):
        # https://user-images.githubusercontent.com/28003269/68451521-8904ab80-0208-11ea-802c-07618440213f.png
        sv_points = [[0.10000000149011612,0.0,0.0],[1.050000000745058,0.0,0.0],[2.0,0.0,0.0],[0.050000000745058074,0.08660254166892227,0.0],[0.5250000003725291,0.9093266746188997,0.0],[1.0000000000000002,1.7320508075688772,0.0],[-0.05000000074505804,0.08660254166892228,0.0],[-0.5250000003725288,0.9093266746188998,0.0],[-0.9999999999999996,1.7320508075688774,0.0],[-0.10000000149011612,1.2246468173960125e-17,0.0],[-1.050000000745058,1.2858791400171537e-16,0.0],[-2.0,2.4492935982947064e-16,0.0],[-0.0500000007450581,-0.08660254166892226,0.0],[-0.5250000003725295,-0.9093266746188996,0.0],[-1.0000000000000009,-1.732050807568877,0.0],[0.05000000074505799,-0.08660254166892231,0.0],[0.5250000003725284,-0.9093266746189002,0.0],[0.9999999999999987,-1.732050807568878,0.0]]
        sv_faces = [[0,3,4,1],[1,4,5,2],[3,6,7,4],[4,7,8,5],[6,9,10,7],[7,10,11,8],[9,12,13,10],[10,13,14,11],[12,15,16,13],[13,16,17,14],[15,0,1,16],[16,1,2,17]]
        sv_points_crop = [[-1.1607272624969482,-1.1950111389160156,0.0],[-1.0042164325714111,-1.1950111389160156,0.0],[-1.1607272624969482,-1.0385003089904785,0.0],[-1.0042164325714111,-1.0385003089904785,0.0],[-0.8702228665351868,1.188905119895935,0.0],[-0.6739181280136108,1.188905119895935,0.0],[-0.8702228665351868,1.3852097988128662,0.0],[-0.6739181280136108,1.3852097988128662,0.0],[0.8670322895050049,1.410539984703064,0.0],[1.036759614944458,1.410539984703064,0.0],[0.8670322895050049,1.580267310142517,0.0],[1.036759614944458,1.580267310142517,0.0],[1.3265749216079712,-0.16742661595344543,0.0],[1.5081655979156494,-0.16742661595344543,0.0],[1.3265749216079712,0.01416410505771637,0.0],[1.5081655979156494,0.01416410505771637,0.0],[0.5823652148246765,-0.46723949909210205,0.0],[0.8293453454971313,-0.46723949909210205,0.0],[0.5823652148246765,-0.22025935351848602,0.0],[0.8293453454971313,-0.22025935351848602,0.0],[-0.47994688153266907,-1.5945452451705933,0.0],[-0.2972078323364258,-1.5945452451705933,0.0],[-0.47994688153266907,-1.4118062257766724,0.0],[-0.2972078323364258,-1.4118062257766724,0.0],[-0.3061579167842865,-1.2619863748550415,0.0],[-0.2091394066810608,-1.2619863748550415,0.0],[-0.3061579167842865,-1.1649678945541382,0.0],[-0.2091394066810608,-1.1649678945541382,0.0],[0.7330341935157776,-0.7124685645103455,0.0],[0.8915227055549622,-0.7124685645103455,0.0],[0.7330341935157776,-0.5539800524711609,0.0],[0.8915227055549622,-0.5539800524711609,0.0],[0.1817319393157959,-0.11247162520885468,0.0],[0.39064377546310425,-0.11247162520885468,0.0],[0.1817319393157959,0.09644022583961487,0.0],[0.39064377546310425,0.09644022583961487,0.0],[-0.9996795058250427,0.7186518907546997,0.0],[-0.7958779335021973,0.7186518907546997,0.0],[-0.9996795058250427,0.9224535226821899,0.0],[-0.7958779335021973,0.9224535226821899,0.0]]
        sv_faces_crop = [[0,1,3,2],[4,5,7,6],[8,9,11,10],[12,13,15,14],[16,17,19,18],[20,21,23,22],[24,25,27,26],[28,29,31,30],[32,33,35,34],[36,37,39,38]]

        expected_points_inner = [[-1.1607272624969482,-1.1950111389160156,0.0],[-1.0042164325714111,-1.1950111389160156,0.0],[-1.0042164325714111,-1.0385003089904785,0.0],[-1.1607272624969482,-1.0385003089904785,0.0],[-0.8702228665351868,1.188905119895935,0.0],[-0.6739181280136108,1.188905119895935,0.0],[-0.6739181280136108,1.3852097988128662,0.0],[-0.8702228665351868,1.3852097988128662,0.0],[0.8670322895050049,1.410539984703064,0.0],[1.036759614944458,1.410539984703064,0.0],[1.036759614944458,1.580267310142517,0.0],[0.8670322895050049,1.580267310142517,0.0],[1.3265749216079712,-0.16742661595344543,0.0],[1.5081655979156494,-0.16742661595344543,0.0],[1.5081655979156494,0.01416410505771637,0.0],[1.3265749216079712,0.01416410505771637,0.0],[0.5823652148246765,-0.46723949909210205,0.0],[0.8293453454971313,-0.46723949909210205,0.0],[0.8293453454971313,-0.22025935351848602,0.0],[0.5823652148246765,-0.22025935351848602,0.0],[-0.47994688153266907,-1.5945452451705933,0.0],[-0.2972078323364258,-1.5945452451705933,0.0],[-0.2972078323364258,-1.4118062257766724,0.0],[-0.47994688153266907,-1.4118062257766724,0.0],[-0.3061579167842865,-1.2619863748550415,0.0],[-0.2091394066810608,-1.2619863748550415,0.0],[-0.2091394066810608,-1.1649678945541382,0.0],[-0.3061579167842865,-1.1649678945541382,0.0],[0.7330341935157776,-0.7124685645103455,0.0],[0.8915227055549622,-0.7124685645103455,0.0],[0.8915227055549622,-0.5539800524711609,0.0],[0.7330341935157776,-0.5539800524711609,0.0],[0.1817319393157959,-0.11247162520885468,0.0],[0.39064377546310425,-0.11247162520885468,0.0],[0.39064377546310425,0.09644022583961487,0.0],[0.1817319393157959,0.09644022583961487,0.0],[-0.9996795058250427,0.7186518907546997,0.0],[-0.7958779335021973,0.7186518907546997,0.0],[-0.7958779335021973,0.9224535226821899,0.0],[-0.9996795058250427,0.9224535226821899,0.0],[0.9123677569023483,1.580267310142517,0],[0.8670322895050049,1.5017439772254357,0],[-0.7997512502287153,1.3852097988128662,0],[-0.686414691012842,1.188905119895935,0],[0.1817319393157959,-0.0,0],[0.39064377546310425,-0.0,0],[1.3265749216079712,-0.0,0],[1.5081655979156494,-0.0,0],[0.8293453454971313,-0.38218507381600336,0],[0.7802391501682067,-0.46723949909210205,0]]
        expected_faces_inner = [[0,1,2,3],[4,43,42,7],[5,6,42,43],[8,9,10,40,41],[11,41,40],[12,13,47,46],[14,15,46,47],[16,49,48,18,19],[17,48,49],[20,21,22,23],[24,25,26,27],[28,29,30,31],[32,33,45,44],[34,35,44,45],[36,37,38,39]]
        expected_face_index_mask_inner = [7,5,3,1,3,11,1,10,11,9,9,11,10,0,5]

        expected_points_outer = [[1.050000000745058,0.0,0.0],[0.5250000003725291,0.9093266746188997,0.0],[0.050000000745058074,0.08660254166892227,0.0],[0.10000000149011612,0.0,0.0],[2.0,0.0,0.0],[1.0000000000000002,1.7320508075688772,0.0],[-0.5250000003725288,0.9093266746188998,0.0],[-0.05000000074505804,0.08660254166892228,0.0],[-0.9999999999999996,1.7320508075688774,0.0],[-1.050000000745058,1.2858791400171537e-16,0.0],[-0.10000000149011612,1.2246468173960125e-17,0.0],[-2.0,2.4492935982947064e-16,0.0],[-0.5250000003725295,-0.9093266746188996,0.0],[-0.0500000007450581,-0.08660254166892226,0.0],[-1.0000000000000009,-1.732050807568877,0.0],[0.5250000003725284,-0.9093266746189002,0.0],[0.05000000074505799,-0.08660254166892231,0.0],[0.9999999999999987,-1.732050807568878,0.0],[-1.1607272624969482,-1.1950111389160156,0.0],[-1.0042164325714111,-1.1950111389160156,0.0],[-1.0042164325714111,-1.0385003089904785,0.0],[-1.1607272624969482,-1.0385003089904785,0.0],[-0.8702228665351868,1.188905119895935,0.0],[-0.6739181280136108,1.188905119895935,0.0],[-0.6739181280136108,1.3852097988128662,0.0],[-0.8702228665351868,1.3852097988128662,0.0],[0.8670322895050049,1.410539984703064,0.0],[1.036759614944458,1.410539984703064,0.0],[1.036759614944458,1.580267310142517,0.0],[0.8670322895050049,1.580267310142517,0.0],[1.3265749216079712,-0.16742661595344543,0.0],[1.5081655979156494,-0.16742661595344543,0.0],[1.5081655979156494,0.01416410505771637,0.0],[1.3265749216079712,0.01416410505771637,0.0],[0.5823652148246765,-0.46723949909210205,0.0],[0.8293453454971313,-0.46723949909210205,0.0],[0.8293453454971313,-0.22025935351848602,0.0],[0.5823652148246765,-0.22025935351848602,0.0],[-0.47994688153266907,-1.5945452451705933,0.0],[-0.2972078323364258,-1.5945452451705933,0.0],[-0.2972078323364258,-1.4118062257766724,0.0],[-0.47994688153266907,-1.4118062257766724,0.0],[-0.3061579167842865,-1.2619863748550415,0.0],[-0.2091394066810608,-1.2619863748550415,0.0],[-0.2091394066810608,-1.1649678945541382,0.0],[-0.3061579167842865,-1.1649678945541382,0.0],[0.7330341935157776,-0.7124685645103455,0.0],[0.8915227055549622,-0.7124685645103455,0.0],[0.8915227055549622,-0.5539800524711609,0.0],[0.7330341935157776,-0.5539800524711609,0.0],[0.1817319393157959,-0.11247162520885468,0.0],[0.39064377546310425,-0.11247162520885468,0.0],[0.39064377546310425,0.09644022583961487,0.0],[0.1817319393157959,0.09644022583961487,0.0],[-0.9996795058250427,0.7186518907546997,0.0],[-0.7958779335021973,0.7186518907546997,0.0],[-0.7958779335021973,0.9224535226821899,0.0],[-0.9996795058250427,0.9224535226821899,0.0],[0.9123677569023483,1.580267310142517,0],[0.8670322895050049,1.5017439772254357,0],[-0.7997512502287153,1.3852097988128662,0],[-0.686414691012842,1.188905119895935,0],[0.1817319393157959,-0.0,0],[0.39064377546310425,-0.0,0],[1.3265749216079712,-0.0,0],[1.5081655979156494,-0.0,0],[0.8293453454971313,-0.38218507381600336,0],[0.7802391501682067,-0.46723949909210205,0]]
        expected_faces_outer = [[0,1,2,3,62,53,52,63],[4,5,58,28,27,26,59,1,0,64,33,32,65],[1,6,7,2],[5,8,60,24,23,61,6,1,59,29,58],[6,9,10,7],[25,8,11,55,54,57,61,22],[61,57,56,55,11,9,6],[9,12,13,10],[21,12,9,11,14,19,18],[12,21,20,19,14],[12,15,16,13],[15,45,44,43,41,40,39,14,17],[45,15,12,14,39,38,41,43,42],[15,67,34,37,36,66,0,63,51,50,62,3,16],[31,66,35,49,48,47,15,17,4,65],[66,31,30,64,0],[8,25,60],[49,35,67,15,47,46]]
        expected_face_index_mask_outer = [0,1,2,3,4,5,5,6,7,7,8,9,9,10,11,11,5,11]

        resoult_points_inner, result_faces_inner, result_face_index_mask_inner = crop_mesh(
            sv_points, sv_faces, sv_points_crop, sv_faces_crop, 'inner', 5)
        result_points_outer, result_faces_outer, result_face_index_mask_outer = crop_mesh(
            sv_points, sv_faces, sv_points_crop, sv_faces_crop, 'outer', 5)

        self.assert_sverchok_data_equal(expected_points_inner, resoult_points_inner, precision=5)
        self.assert_sverchok_data_equal(expected_faces_inner, result_faces_inner)
        self.assert_sverchok_data_equal(expected_face_index_mask_inner, result_face_index_mask_inner)

        self.assert_sverchok_data_equal(expected_points_outer, result_points_outer, precision=5)
        self.assert_sverchok_data_equal(expected_faces_outer, expected_faces_outer)
        self.assert_sverchok_data_equal(expected_face_index_mask_outer, result_face_index_mask_outer)

    def test_crop_voronoi_diagram(self):
        # https://user-images.githubusercontent.com/28003269/68539337-5dfe9100-039b-11ea-9811-1a1733a447c8.png
        sv_points = [[0.44226984516779116,-0.9838968933451726,0],[-0.7717948131634805,-1.1243142330730151,0],[-0.6412958948828816,-0.9516264430997181,0],[0.8248657980787115,-0.8594954620924617,0],[0.05868619192567045,-0.8973215738133425,0],[-0.8728987114388891,-0.5675260628504114,0],[-0.22243873605918113,-0.7423955410770146,0],[0.0012446428355074245,-0.7517340980124427,0],[0.03744206883701466,-0.7565220057126554,0],[-0.7314112959867994,-0.4857827600180362,0],[-0.7028994639073691,-0.47944910897141346,0],[0.6181079752942285,-0.5804383408898783,0],[0.18964515389924808,-0.5542568813809652,0],[-0.60051260163077,-0.3610753395397653,0],[0.8325039896277248,-0.2808137020907292,0],[-0.5845466067979652,-0.25223074883309776,0],[-0.8356697404006861,-0.10895380128928052,0],[-0.467987814390869,-0.16730622549479474,0],[0.8197388895413075,0.02798159392958952,0],[0.9630913114646215,0.12469426635818207,0],[-0.6224337484282502,0.3984974291350859,0],[-1.574350742285174,-0.010646327699981698,0],[0.830438255194535,0.5924881938998082,0],[-1.7629028477204671,-0.026846861611540766,0],[0.5976034958508046,0.6348222303592513,0],[0.021298872839596802,0.2889651679712403,0],[0.015628740098684094,0.2960974104836859,0],[0.007807561843564917,0.32804585497652267,0],[-0.25198319028726085,0.6941433825719657,0]]
        sv_edges = [[1,2],[0,3],[4,0],[5,1],[2,6],[6,7],[7,8],[8,4],[5,9],[9,10],[10,2],[11,3],[8,12],[12,11],[10,13],[13,6],[11,14],[13,15],[16,15],[15,17],[17,7],[18,14],[18,19],[16,20],[21,16],[21,20],[22,19],[23,9],[23,21],[24,22],[25,12],[25,18],[17,26],[26,25],[27,26],[27,24],[20,28],[28,27]]
        sv_points_crop = [[1.0,0.0,0],[0.30901699437494745,0.9510565162951535,0],[-0.8090169943749473,0.5877852522924732,0],[-0.8090169943749475,-0.587785252292473,0],[0.30901699437494723,-0.9510565162951536,0]]
        sv_faces_crop = [[0,1,2,3,4]]

        expected_points_inner = [[-0.22243873605918113,-0.7423955410770146,0],[0.0012446428355074245,-0.7517340980124427,0],[0.03744206883701466,-0.7565220057126554,0],[-0.7314112959867994,-0.4857827600180362,0],[-0.7028994639073691,-0.47944910897141346,0],[0.18964515389924808,-0.5542568813809652,0],[-0.60051260163077,-0.3610753395397653,0],[-0.5845466067979652,-0.25223074883309776,0],[-0.467987814390869,-0.16730622549479474,0],[0.8197388895413075,0.02798159392958952,0],[-0.6224337484282502,0.3984974291350859,0],[0.021298872839596802,0.2889651679712403,0],[0.015628740098684094,0.2960974104836859,0],[0.007807561843564917,0.32804585497652267,0],[-0.25198319028726085,0.6941433825719657,0],[0.5549090443762749,0.6126151442857328,0],[-0.8090169943749472,0.31830202270599633,0],[0.927063678827927,0.1003882338069202,0],[-0.8090169943749473,-0.045526569142197285,0],[-0.8090169943749472,-0.12416038158096214,0],[0.8305375767565504,-0.23324501555151803,0],[-0.8090169943749475,-0.4512540807948599,0],[-0.8090169943749473,-0.5306188755452752,0],[0.5799795996679871,-0.5781084852460469,0],[-0.6834418773161244,-0.6285870811816373,0],[-0.26608072629389345,-0.7641959395911969,0],[0.0543067773767166,-0.8682961499606708,0]]
        expected_edges_inner = [[0,25],[0,1],[1,2],[2,26],[3,22],[3,4],[4,24],[2,5],[5,23],[4,6],[6,0],[6,7],[7,19],[7,8],[8,1],[9,20],[9,17],[10,18],[10,16],[3,21],[11,5],[11,9],[8,12],[12,11],[13,12],[13,15],[10,14],[14,13]]

        expected_points_outer = [[-0.7717948131634805,-1.1243142330730151,0],[-0.6412958948828816,-0.9516264430997181,0],[0.44226984516779116,-0.9838968933451726,0],[0.8248657980787115,-0.8594954620924617,0],[0.05868619192567045,-0.8973215738133425,0],[-0.8728987114388891,-0.5675260628504114,0],[0.6181079752942285,-0.5804383408898783,0],[0.8325039896277248,-0.2808137020907292,0],[-0.8356697404006861,-0.10895380128928052,0],[0.9630913114646215,0.12469426635818207,0],[-1.574350742285174,-0.010646327699981698,0],[0.830438255194535,0.5924881938998082,0],[-1.7629028477204671,-0.026846861611540766,0],[0.5976034958508046,0.6348222303592513,0],[0.5549090443762749,0.6126151442857328,0],[-0.8090169943749472,0.31830202270599633,0],[0.927063678827927,0.1003882338069202,0],[-0.8090169943749473,-0.045526569142197285,0],[-0.8090169943749472,-0.12416038158096214,0],[0.8305375767565504,-0.23324501555151803,0],[-0.8090169943749475,-0.4512540807948599,0],[-0.8090169943749473,-0.5306188755452752,0],[0.5799795996679871,-0.5781084852460469,0],[-0.6834418773161244,-0.6285870811816373,0],[-0.26608072629389345,-0.7641959395911969,0],[0.0543067773767166,-0.8682961499606708,0]]
        expected_edges_outer = [[0,1],[2,3],[4,2],[5,0],[1,24],[4,25],[5,21],[1,23],[6,3],[6,22],[6,7],[8,18],[7,19],[9,16],[8,17],[10,8],[10,15],[11,9],[12,20],[12,10],[13,11],[13,14]]

        result_points_inner, result_edges_inner = crop_edges(sv_points, sv_edges, sv_points_crop, sv_faces_crop,
                                                             'inner', 5)
        result_points_outer, result_edges_outer = crop_edges(sv_points, sv_edges, sv_points_crop, sv_faces_crop,
                                                             'outer', 5)

        self.assert_sverchok_data_equal(expected_points_inner, result_points_inner, precision=5)
        self.assert_sverchok_data_equal(expected_edges_inner, result_edges_inner)
        self.assert_sverchok_data_equal(expected_points_outer, result_points_outer, precision=5)
        self.assert_sverchok_data_equal(expected_edges_outer, expected_edges_outer)


class MergeMesh2DTest(SverchokTestCase):

    def test_merge_square_ring(self):
        # https://user-images.githubusercontent.com/28003269/68469770-4062e780-0234-11ea-8af6-33b6def2ea8d.png
        sv_points_a = [[-0.5,-0.5,0.0],[0.0,-0.5,0.0],[0.5,-0.5,0.0],[-0.5,0.0,0.0],[0.0,0.0,0.0],[0.5,0.0,0.0],[-0.5,0.5,0.0],[0.0,0.5,0.0],[0.5,0.5,0.0]]
        sv_faces_a = [[0,1,4,3],[3,4,7,6],[1,2,5,4],[4,5,8,7]]
        sv_points_b = [[0.20000000298023224,0.0,0.0],[0.4750000014901161,0.0,0.0],[0.75,0.0,0.0],[0.14142135834465194,0.14142135834465194,0.0],[0.3358757221172813,0.3358757221172813,0.0],[0.5303300858899107,0.5303300858899107,0.0],[1.2246468173960125e-17,0.20000000298023224,0.0],[2.9085361570992937e-17,0.4750000014901161,0.0],[4.592425496802574e-17,0.75,0.0],[-0.14142135834465192,0.14142135834465194,0.0],[-0.3358757221172813,0.3358757221172813,0.0],[-0.5303300858899106,0.5303300858899107,0.0],[-0.20000000298023224,2.449293634792025e-17,0.0],[-0.4750000014901161,5.817072314198587e-17,0.0],[-0.75,9.184850993605148e-17,0.0],[-0.14142135834465197,-0.14142135834465192,0.0],[-0.33587572211728134,-0.3358757221172813,0.0],[-0.5303300858899107,-0.5303300858899106,0.0],[-3.6739404521880375e-17,-0.20000000298023224,0.0],[-8.72560847129788e-17,-0.4750000014901161,0.0],[-1.3777276490407724e-16,-0.75,0.0],[0.1414213583446519,-0.14142135834465197,0.0],[0.33587572211728123,-0.33587572211728134,0.0],[0.5303300858899105,-0.5303300858899107,0.0]]
        sv_faces_b = [[0,3,4,1],[1,4,5,2],[3,6,7,4],[4,7,8,5],[6,9,10,7],[7,10,11,8],[9,12,13,10],[10,13,14,11],[12,15,16,13],[13,16,17,14],[15,18,19,16],[16,19,20,17],[18,21,22,19],[19,22,23,20],[21,0,1,22],[22,1,2,23]]

        expected_points = [[-0.5,-0.5,0.0],[0.0,-0.5,0.0],[0.0,0.0,0.0],[-0.5,0.0,0.0],[-0.20000000298023224,2.449293634792025e-17,0.0],[0.0,0.5,0.0],[-0.5,0.5,0.0],[0.5,-0.5,0.0],[0.5,0.0,0.0],[0.5,0.5,0.0],[1.2246468173960125e-17,0.20000000298023224,0.0],[0.4750000014901161,0.0,0.0],[0.3358757221172813,0.3358757221172813,0.0],[0.14142135834465194,0.14142135834465194,0.0],[0.20000000298023224,0.0,0.0],[0.75,0.0,0.0],[0.5303300858899107,0.5303300858899107,0.0],[2.9085361570992937e-17,0.4750000014901161,0.0],[4.592425496802574e-17,0.75,0.0],[-0.3358757221172813,0.3358757221172813,0.0],[-0.14142135834465192,0.14142135834465194,0.0],[-0.5303300858899106,0.5303300858899107,0.0],[-0.4750000014901161,5.817072314198587e-17,0.0],[-0.75,9.184850993605148e-17,0.0],[-0.33587572211728134,-0.3358757221172813,0.0],[-0.14142135834465197,-0.14142135834465192,0.0],[-0.5303300858899107,-0.5303300858899106,0.0],[-8.72560847129788e-17,-0.4750000014901161,0.0],[-3.6739404521880375e-17,-0.20000000298023224,0.0],[-1.3777276490407724e-16,-0.75,0.0],[0.33587572211728123,-0.33587572211728134,0.0],[0.1414213583446519,-0.14142135834465197,0.0],[0.5303300858899105,-0.5303300858899107,0.0]]
        expected_faces = [[0,1,27,24],[2,4,25,28],[3,0,24,22],[4,2,10,20],[5,6,19,17],[6,3,22,19],[1,7,30,27],[7,8,11,30],[2,28,31,14],[2,14,13,10],[8,9,12,11],[9,5,17,12],[1,0,26,29],[0,3,23,26],[6,5,18,21],[3,6,21,23],[7,1,29,32],[8,7,32,15],[9,8,15,16],[5,9,16,18],[11,12,13,14],[12,17,10,13],[17,19,20,10],[19,22,4,20],[22,24,25,4],[24,27,28,25],[27,30,31,28],[30,11,14,31]]
        expected_mask_a = [1,1,1,1,1,1,1,1,1,1,1,1,0,0,0,0,0,0,0,0,1,1,1,1,1,1,1,1]
        expected_mask_b = [1,0,1,0,1,1,1,1,0,0,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1]
        expected_face_index_a = [0,0,0,1,1,1,2,2,2,3,3,3,-1,-1,-1,-1,-1,-1,-1,-1,3,3,1,1,0,0,2,2]
        expected_face_index_b = [0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0]

        result_points, result_faces, result_mask_a, result_mask_b, result_face_index_a, result_face_index_b = \
            merge_mesh(sv_points_a, sv_faces_a, sv_points_b, sv_faces_b, True, True, 5)

        self.assert_sverchok_data_equal(expected_points, result_points, precision=5)
        self.assert_sverchok_data_equal(expected_faces, result_faces)
        self.assert_sverchok_data_equal(expected_mask_a, result_mask_a)
        self.assert_sverchok_data_equal(expected_mask_b, result_mask_b)
        self.assert_sverchok_data_equal(expected_face_index_a, result_face_index_a)
        self.assert_sverchok_data_equal(expected_face_index_b, result_face_index_b)


class DissolveFaces2DTest(SverchokTestCase):

    def test_dissolve_ngon_faces_inside_grid(self):
        # https://user-images.githubusercontent.com/28003269/68987705-8bd55100-0845-11ea-99ea-512445e9a2c6.png
        sv_points = [[0.50776606798172,0.4593293070793152,0.0],[0.29009705781936646,0.7589249610900879,0.0],[-0.06209878623485565,0.6444896459579468,0.0],[-0.06209878623485565,0.274168998003006,0.0],[0.29009705781936646,0.1597336232662201,0.0],[-0.4833621382713318,-0.4777355492115021,0.0],[-0.7481659054756165,-0.11326441168785095,0.0],[-1.1766273975372314,-0.25248000025749207,0.0],[-1.1766273975372314,-0.7029911279678345,0.0],[-0.7481659054756165,-0.8422067165374756,0.0],[0.14240731298923492,-0.6991196870803833,0.0],[-0.09054585546255112,-0.37848716974258423,0.0],[-0.46747201681137085,-0.5009579062461853,0.0],[-0.46747201681137085,-0.8972814679145813,0.0],[-0.09054585546255112,-1.0197522640228271,0.0],[0.04169577360153198,0.5265645980834961,0.0],[-0.17485976219177246,0.824627697467804,0.0],[-0.5252540111541748,0.710777759552002,0.0],[-0.5252540111541748,0.3423514664173126,0.0],[-0.17485976219177246,0.22850149869918823,0.0],[-0.3059031367301941,0.14020152390003204,0.0],[-0.48811429738998413,0.3909936547279358,0.0],[-0.7829381227493286,0.29519957304000854,0.0],[-0.7829381227493286,-0.014796540141105652,0.0],[-0.48811429738998413,-0.1105906218290329,0.0],[-1.25,-1.25,0.0],[-0.75,-1.25,0.0],[-0.75,-0.75,0.0],[-1.25,-0.75,0.0],[-0.75,-0.25,0.0],[-1.25,-0.25,0.0],[-0.75,0.25,0.0],[-1.25,0.25,0.0],[-0.75,0.75,0.0],[-1.25,0.75,0.0],[-0.75,1.25,0.0],[-1.25,1.25,0.0],[-0.25,-1.25,0.0],[-0.25,-0.75,0.0],[-0.25,-0.25,0.0],[-0.25,0.25,0.0],[-0.25,0.75,0.0],[-0.25,1.25,0.0],[0.25,-1.25,0.0],[0.25,-0.75,0.0],[0.25,-0.25,0.0],[0.25,0.25,0.0],[0.25,0.75,0.0],[0.25,1.25,0.0],[0.75,-1.25,0.0],[0.75,-0.75,0.0],[0.75,-0.25,0.0],[0.75,0.25,0.0],[0.75,0.75,0.0],[0.75,1.25,0.0],[1.25,-1.25,0.0],[1.25,-0.75,0.0],[1.25,-0.25,0.0],[1.25,0.25,0.0],[1.25,0.75,0.0],[1.25,1.25,0.0],[-0.25,0.800213166022788,0],[-0.40454030910048633,0.75,0],[-0.12063956169213375,0.75,0],[0.26262883963300576,0.75,0],[0.2965814223961207,0.7499999999999999,0],[0.25,0.7458966431104911,0],[-0.0474418321740453,0.6492519768745146,0],[-0.062098786234855645,0.38370365439655,0],[-0.5252540111541748,0.3789262284612263,0],[-0.43037602605864467,0.31152374714887837,0],[-0.75,0.30590181943224076,0],[-0.25,0.2529160365351583,0],[-0.7829381227493286,0.25,0],[-0.38567640434751044,0.25,0],[-0.2410253603741596,0.25,0],[-0.15924018540562482,0.25,0],[0.012285735781981902,0.25,0],[0.3556794207710358,0.25,0],[0.25,0.17276194803173134,0],[-0.75,-0.025498786533337918,0],[-0.7499999999999999,-0.11386034516947105,0],[-1.1689947421603755,-0.25,0],[-0.6488216908912678,-0.25,0],[-0.25,-0.43029696307232007,0],[-1.0319489771248536,-0.75,0],[-0.6811738136736784,-0.75,0],[-0.46747201681137085,-0.75,0],[0.1054406072992803,-0.75,0],[-0.75,-0.8416107830558557,0],[-0.25,-0.9679424454780529,0]]
        sv_faces = [[0,65,64,66,46,78],[1,64,65],[2,68,15,67],[3,77,46,66,67,15,68],[4,78,46,79],[5,83,29,27,86],[6,81,29,83],[7,8,85,27,29,82],[9,86,27,89],[10,11,84,38,88],[12,87,38,84],[13,90,38,87],[14,88,38,90],[16,61,41,63],[17,69,21,70,72,41,62],[18,70,21,69],[19,76,75],[20,74,31,80,24],[22,73,31,71],[23,80,31,73],[25,26,89,85,28],[27,85,89],[28,85,8,7,82,30],[30,82,81,80,23,73,32],[29,81,82],[32,73,22,71,33,34],[34,33,35,36],[26,37,90,13,87,86,9,89],[39,83,5,86,87,12,84],[39,40,74,20,24,80,81,6,83],[31,74,70,18,69,71],[40,72,70,74],[33,71,69,17,62],[33,62,61,42,35],[41,61,62],[37,43,44,88,14,90],[44,45,39,84,11,10,88],[39,45,79,77,76,19,75,40],[46,77,79],[40,75,72],[47,63,67,66],[41,72,75,76,68,2,67,63],[47,48,42,61,16,63],[43,49,50,44],[44,50,51,45],[45,51,52,78,4,79],[52,53,65,0,78],[47,66,64],[47,64,1,65,53,54,48],[49,55,56,50],[50,56,57,51],[51,57,58,52],[52,58,59,53],[53,59,60,54],[3,68,76,77]]
        face_mask = [1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,0,1,0,0,1,0,0,0,0,0,1,0,0,0,1,0,0,0,1,0,0,1,0,0,0,0,0,0,0,0,0,0,0,0,0]

        expected_points = [[0.29009705781936646,0.7589249610900879,0.0],[0.26262883963300576,0.75,0],[0.25,0.7458966431104911,0],[-0.0474418321740453,0.6492519768745146,0],[-0.12063956169213375,0.75,0],[-0.17485976219177246,0.824627697467804,0.0],[-0.25,0.800213166022788,0],[-0.40454030910048633,0.75,0],[-0.5252540111541748,0.710777759552002,0.0],[-0.5252540111541748,0.3789262284612263,0],[-0.75,0.30590181943224076,0],[-0.7829381227493286,0.29519957304000854,0.0],[-0.7829381227493286,0.25,0],[-0.7829381227493286,-0.014796540141105652,0.0],[-0.75,-0.025498786533337918,0],[-0.48811429738998413,-0.1105906218290329,0.0],[-0.3059031367301941,0.14020152390003204,0.0],[-0.38567640434751044,0.25,0],[-0.43037602605864467,0.31152374714887837,0],[-0.25,0.2529160365351583,0],[-0.2410253603741596,0.25,0],[-0.17485976219177246,0.22850149869918823,0.0],[-0.15924018540562482,0.25,0],[-0.062098786234855645,0.38370365439655,0],[-0.06209878623485565,0.274168998003006,0.0],[0.012285735781981902,0.25,0],[0.25,0.17276194803173134,0],[0.29009705781936646,0.1597336232662201,0.0],[0.3556794207710358,0.25,0],[0.50776606798172,0.4593293070793152,0.0],[0.2965814223961207,0.7499999999999999,0],[-0.7481659054756165,-0.11326441168785095,0.0],[-0.7499999999999999,-0.11386034516947105,0],[-1.1689947421603755,-0.25,0],[-1.1766273975372314,-0.25248000025749207,0.0],[-1.1766273975372314,-0.7029911279678345,0.0],[-1.0319489771248536,-0.75,0],[-0.75,-0.8416107830558557,0],[-0.7481659054756165,-0.8422067165374756,0.0],[-0.6811738136736784,-0.75,0],[-0.4833621382713318,-0.4777355492115021,0.0],[-0.6488216908912678,-0.25,0],[-0.46747201681137085,-0.5009579062461853,0.0],[-0.46747201681137085,-0.75,0],[-0.46747201681137085,-0.8972814679145813,0.0],[-0.25,-0.9679424454780529,0],[-0.09054585546255112,-1.0197522640228271,0.0],[0.1054406072992803,-0.75,0],[0.14240731298923492,-0.6991196870803833,0.0],[-0.09054585546255112,-0.37848716974258423,0.0],[-0.25,-0.43029696307232007,0],[-1.25,-1.25,0.0],[-0.75,-1.25,0.0],[-1.25,-0.75,0.0],[-1.25,-0.25,0.0],[-1.25,0.25,0.0],[-0.75,0.75,0.0],[-1.25,0.75,0.0],[-0.75,1.25,0.0],[-1.25,1.25,0.0],[-0.25,-1.25,0.0],[-0.25,-0.25,0.0],[-0.25,0.25,0.0],[-0.25,1.25,0.0],[0.25,-1.25,0.0],[0.25,-0.75,0.0],[0.25,-0.25,0.0],[0.25,0.75,0.0],[0.25,1.25,0.0],[0.75,-1.25,0.0],[0.75,-0.75,0.0],[0.75,-0.25,0.0],[0.75,0.25,0.0],[0.75,0.75,0.0],[0.75,1.25,0.0],[1.25,-1.25,0.0],[1.25,-0.75,0.0],[1.25,-0.25,0.0],[1.25,0.25,0.0],[1.25,0.75,0.0],[1.25,1.25,0.0]]
        expected_faces = [[0,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19,20,21,22,23,24,25,26,27,28,29,30],[31,32,33,34,35,36,37,38,39,40,41],[42,43,44,45,46,47,48,49,50],[51,52,37,36,53],[53,36,35,34,33,54],[54,33,32,14,13,12,55],[55,12,11,10,56,57],[57,56,58,59],[52,60,45,44,43,39,38,37],[61,41,40,39,43,42,50],[61,62,17,16,15,14,32,31,41],[62,19,18,17],[56,10,9,8,7],[56,7,6,63,58],[60,64,65,47,46,45],[65,66,61,50,49,48,47],[61,66,26,25,22,21,20,62],[62,20,19],[67,4,3,2],[67,68,63,6,5,4],[64,69,70,65],[65,70,71,66],[66,71,72,28,27,26],[72,73,30,29,28],[67,2,1],[67,1,0,30,73,74,68],[69,75,76,70],[70,76,77,71],[71,77,78,72],[72,78,79,73],[73,79,80,74],[24,23,22,25]]
        expected_face_mask = [1,1,1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0]
        expected_index_mask = [0,5,9,20,22,23,25,26,27,28,29,31,32,33,35,36,37,39,40,42,43,44,45,46,47,48,49,50,51,52,53,54]

        result_points, result_faces, result_face_mask, result_index_mask = dissolve_faces(sv_points, sv_faces,
                                                                                          face_mask, True, True)

        self.assert_sverchok_data_equal(expected_points, result_points, precision=5)
        self.assert_sverchok_data_equal(expected_faces, result_faces)
        self.assert_sverchok_data_equal(expected_face_mask, result_face_mask)
        self.assert_sverchok_data_equal(expected_index_mask, result_index_mask)
