#ifndef CONFIGURATION_H
#define CONFIGURATION_H

#include <QMap>
#include <QObject>
#include <QString>

class QVariant;
class QSettings;

class Configuration : QObject
{
    Q_OBJECT

    const QString SYSTEM_CONFIG_PATH = QStringLiteral("/usr/lib");
    const QString LOCAL_CONFIG_PATH = QStringLiteral("/etc");

    enum class Param {
        guestEnabled,
        mosAuthBtnEnabled,
        loginLowercaseOnly,
        wifiEapDomain,
        wifiEapSsid,
    };

public:
    explicit Configuration(QObject *parent, const QString &configPath);

    // Value getters

    bool guestEnabled() const;
    bool mosAuthBtnEnabled() const;
    bool loginLowercaseOnly() const;
    QString wifiEapDomain() const;
    QString wifiEapSsid() const;

private:
    // Object with settings values.
    QMap<Param, QVariant> settings;

    // Convert table from Enum value to INI-label of param.
    const QMap<Param, QString> paramsMap = {
        {Param::guestEnabled, "sddm/guest-enabled"},
        {Param::mosAuthBtnEnabled, "sddm/mos-auth-btn-enabled"},
        {Param::loginLowercaseOnly, "sddm/login-lowercase-only"},
        {Param::wifiEapDomain, "sddm/wifi-eap-domain"},
        {Param::wifiEapSsid, "sddm/wifi-eap-ssid"},
    };

    /**
     * @brief readFromQSettings - read all supported settings from `q`
     * @param q - QSettings object
     */
    void readFromQSettings(const QSettings &q);

    /**
     * @brief readFromDropInDir - read all drop-in configs from directory `.d`
     * @param dirPath - path for dir with drop-in configs
     * @param extension - extension with dot for drop-in configs (e.g. '.conf'),
     * zero value ("") - for files without extension,
     * and "*" value for any files.
     */
    void readFromDropInDir(const QString &dirPath, const QString &extension = ".conf");

    /**
     * @brief parseBoolValue - parse string value to boolean
     * @param value
     * @param defaultValue
     *
     * This function supports values: "yes", "true", "1", "no", "false", "0"
     * and return default value for other cases.
     *
     * @return
     */
    bool parseBoolValue(const QString &value, bool defaultValue) const;
};

#endif // CONFIGURATION_H
