#include "configuration.h"

#include <QDebug>
#include <QDir>
#include <QSettings>

Configuration::Configuration(QObject *parent, const QString &configPath)
    : QObject(parent)
{
    if (configPath.isEmpty()) {
        qWarning() << "Config path is empty.";
        return;
    }

    // First - read from system drop-in configs.
    // Second - read from local config, then drop-in configs.
    this->readFromDropInDir(SYSTEM_CONFIG_PATH + configPath + ".d");
    this->readFromQSettings({LOCAL_CONFIG_PATH + configPath, QSettings::IniFormat});
    this->readFromDropInDir(LOCAL_CONFIG_PATH + configPath + ".d");
}

void Configuration::readFromQSettings(const QSettings &q)
{
    for (auto it = paramsMap.keyValueBegin(); it != paramsMap.keyValueEnd(); ++it) {
        const auto [key, label] = *it;
        if (q.contains(label)) {
            settings[key] = q.value(label);
        }
    }
}

void Configuration::readFromDropInDir(const QString &dirPath, const QString &extension)
{
    // Dir with drop-in files.
    QDir dir(dirPath);
    dir.setNameFilters({"*" + extension});
    const auto files = dir.entryList(QDir::Files, QDir::Name);
    for (const auto &file : files) {
        // Speical case - if empty extension, then skip files with dot.
        if (extension == "" && file.contains(".")) {
            continue;
        }
        this->readFromQSettings({dirPath + "/" + file, QSettings::IniFormat});
    }
}

bool Configuration::parseBoolValue(const QString &input, bool defaultValue) const
{
    // Get trimmed value.
    const auto value = input.trimmed();

    if (value == QLatin1String("yes") || value == QLatin1String("true") || value == QLatin1String("1")) {
        return true;
    } else if (value == QLatin1String("no") || value == QLatin1String("false") || value == QLatin1String("0")) {
        return false;
    }

    return defaultValue;
}

bool Configuration::guestEnabled() const
{
    return this->parseBoolValue(settings.value(Param::guestEnabled).toString(), true);
}

bool Configuration::mosAuthBtnEnabled() const
{
    return this->parseBoolValue(settings.value(Param::mosAuthBtnEnabled).toString(), true);
}

bool Configuration::loginLowercaseOnly() const
{
    return this->parseBoolValue(settings.value(Param::loginLowercaseOnly).toString(), false);
}

QString Configuration::wifiEapDomain() const
{
    return settings.value(Param::wifiEapDomain).toString();
}

QString Configuration::wifiEapSsid() const
{
    return settings.value(Param::wifiEapSsid).toString();
}
