#include "authservice.h"
#include "authdbusservice.h"
#include "configuration.h"
#include "utils.h"

#include <QDebug>
#include <QMap>
#include <QProcess>
#include <QStringList>
#include <QtDBus/QtDBus>

#include <algorithm>

AuthService::AuthService(QObject *parent)
    : QObject(parent)
    , config(new Configuration(this, "/mos-auth/mos-auth.conf"))
    , registerUserProcess(new QProcess(this))
    , prepareGuestProcess(new QProcess(this))
{
    // Удаляем гостя, если он выключен в настройках.
    if (!guestEnabled()) {
        QProcess *deleteGuest = new QProcess(this);
        const QString cmd = R"( sudo /bin/loginctl kill-user --signal KILL 830 ;
                                sudo /usr/sbin/userdel -rf guest
                            )";

        deleteGuest->start("/bin/sh", QStringList() << "-c" << cmd);
    }

    // If our previous enterprise connection is existed, then delete it.
    {
        QProcess *deleteWifi = new QProcess(this);

        const QString deleteIfExist = QString("nmcli connection delete %1").arg(this->enterpriseConnectionName);
        deleteWifi->start("/bin/sh", QStringList() << "-c" << deleteIfExist);
    }

    // clang-format off
    connect(registerUserProcess, QOverload<int, QProcess::ExitStatus>::of(&QProcess::finished),
            [=](int exitCode, QProcess::ExitStatus /*exitStatus*/)
    {
        Q_EMIT registerUserFinished(exitCode == 0);
    });

    connect(prepareGuestProcess, QOverload<int, QProcess::ExitStatus>::of(&QProcess::finished),
            [=](int exitCode, QProcess::ExitStatus /*exitStatus*/)
    {
        Q_EMIT prepareGuestFinished(exitCode == 0);
    });
    // clang-format on

    this->dbusConnect();
}

void AuthService::dbusConnect()
{
    if (!QDBusConnection::systemBus().isConnected()) {
        qWarning() << "Cannot connect to the D-Bus system bus.";
        return;
    }

    if (!QDBusConnection::systemBus().registerService("org.mos.auth")) {
        qWarning() << qPrintable(QDBusConnection::systemBus().lastError().message());
        return;
    }

    AuthDBusService *dbus = new AuthDBusService(this);
    QDBusConnection::systemBus().registerObject("/", dbus, QDBusConnection::ExportAllSlots);

    connect(dbus, &AuthDBusService::loginCalled, this, &AuthService::login);
}

QMap<int, QString> AuthService::initMapRoleToGroup()
{
    QMap<int, QString> res;

    res.insert(Role::Admin, "mos-admin");
    res.insert(Role::Teacher, "mos-teacher");
    res.insert(Role::Student, "mos-student");

    return res;
}

QString AuthService::getGroupsFromRoles(const QList<int> &roles, const QString &separator)
{
    QString groups = "";

    for (int role : roles) {
        if (groups.length() > 0) {
            groups += separator;
        }

        groups += mapRoleToGroup.value(role);
    }

    return groups;
}

QString AuthService::getExcludeGroupsFromRoles(const QList<int> &userRoles, const QString &separator)
{
    // Все роли.
    QList<int> allRoles = mapRoleToGroup.keys();

    // Роли, не относящиеся к пользователю (группы, к которым он не принадлежит).
    QList<int> excludedRoles;

    std::set_difference(allRoles.begin(), //
                        allRoles.end(),
                        userRoles.begin(),
                        userRoles.end(),
                        std::inserter(excludedRoles, excludedRoles.begin()));

    return getGroupsFromRoles(excludedRoles, separator);
}

void AuthService::registerUser(const QString &obrId, //
                               const QString &_name,
                               const QString &_login,
                               const QString &_pin,
                               QList<int> roles)
{
    bool obrIdToNumberResult = false;
    const qint64 numericObrId = obrId.toLongLong(&obrIdToNumberResult);

    if (!obrIdToNumberResult) {
        return Q_EMIT registerUserFinished(false);
    }

    const qint64 uid = 10'000'000 + numericObrId;

    const QString uid_arg = QString::number(uid);
    const QString login = Utils::shellQuote(_login);
    const QString pin = Utils::shellQuote(_pin);
    const QString name = Utils::shellQuote(_name);

    if (roles.empty()) {
        roles.push_back(Role::Student);
    }

    // Вообще такого не должно быть, что список групп от ролей пустой,
    // так как выше в случае пустого массива с ролями добавляется группа студента (как fallback),
    // но на всякий случай обработаем и этот случай при добавлении commonGroup
    const QString groupsListFromRoles = getGroupsFromRoles(roles);
    const QString groups = Utils::shellQuote(groupsListFromRoles != "" //
                                                 ? groupsListFromRoles + "," + commonGroup
                                                 : commonGroup);

    const QString exclude_groups = Utils::shellQuote(getExcludeGroupsFromRoles(roles));

    // Создаем группы mos-teacher, mos-student и mos-admin, если их вдруг нет.
    const QString _cmd_admin_group = "(getent group 310 || sudo /usr/sbin/groupadd -g 310 mos-admin)";
    const QString _cmd_teacher_group = "(getent group 311 || sudo /usr/sbin/groupadd -g 311 mos-teacher)";
    const QString _cmd_student_group = "(getent group 312 || sudo /usr/sbin/groupadd -g 312 mos-student)";
    const QString cmd_create_groups = _cmd_admin_group + " && " + _cmd_teacher_group + " && " + _cmd_student_group;

    // Создаем группу для пользователя с gid = `uid_arg`, если её не существует.
    const QString _cmd_groupadd = "sudo /usr/sbin/groupadd -g " + uid_arg + " " + login;
    const QString cmd_create_group = "(getent group " + login + " || " + _cmd_groupadd + ")";

    // Проверяем, существует ли пользователь.
    const QString cmd_check_user_exist = "/bin/id -u " + login;

    // Обновляем группы пользователя (на тот случай, если он уже существует в системе).
    const QString _cmd_exclude_user_groups =
        "sudo /usr/sbin/usermod -G $(/bin/id -nGz " + login + " | grep -zxv " + exclude_groups + " | tr '\\0' ',' | sed 's/.$/\\n/') " + login;
    const QString cmd_update_user_groups = "(" + _cmd_exclude_user_groups + " && " + "sudo /usr/sbin/usermod -a -G " + groups + " " + login + ")";

    // Создаем нового пользователя (если его не было в системе).
    const QString cmd_create_user = "sudo /usr/sbin/useradd -c " + name + " -u " + uid_arg + " -g " + uid_arg + " -G " + groups + " " + login;

    // Задаем пароль.
    const QString cmd_pass = "sudo /usr/sbin/chpasswd <<< " + Utils::shellQuote(_login + ":" + _pin);

    // Сбрасываем блокировку пароля (если была).
    const QString cmd_reset_faillock = "sudo /sbin/faillock --user " + login + " --reset";

    const QString cmd = cmd_create_groups + " && " + cmd_create_group + " && " + "(" + cmd_check_user_exist + " && " + cmd_update_user_groups + " || "
        + cmd_create_user + ")" + " && " + cmd_pass + " && " + cmd_reset_faillock;

    registerUserProcess->start("/bin/sh", QStringList() << "-c" << cmd);
}

void AuthService::prepareGuest()
{
    const QString _cmd_create_guest = R"(sudo /usr/sbin/useradd -u 830 -c "Гость" -M guest -G )" + commonGroup;

    const QString cmd =
        "sudo /bin/loginctl kill-user --signal KILL 830 ; "
        "sudo /usr/sbin/userdel -rf guest ; "
        + _cmd_create_guest + " && " + "sudo /usr/bin/passwd -d guest";

    prepareGuestProcess->start("/bin/sh", QStringList() << "-c" << cmd);
}

void AuthService::handleEapWifi(const QString &_ssid, const QString &_login, const QString &_pass)
{
    QProcess *createWifi = new QProcess(this);

    const auto ssid = Utils::shellQuote(_ssid);
    const auto login = Utils::shellQuote(_login);
    const auto pass = Utils::shellQuote(_pass);

    // Включаем Wi-Fi перед попыткой соединения, на тот случай, если Wi-Fi был выключен.
    // Ожидаем включения Wi-Fi таким образом,
    // так как `nmcli radio wifi on` не дожидается включения и сразу возвращает результат.
    const QString enableWifi = QStringLiteral("nmcli radio wifi on && while [[ $(nmcli radio wifi) == 'disabled' ]]; do sleep 1; done");
    const QString deleteIfExist = QString("nmcli connection delete %1").arg(this->enterpriseConnectionName);
    // Если после попытки удаления, оно все еще существует, значит у нас нет прав на это соединение,
    // поэтому не будем подключаться в целях безопасности.
    const QString checkExist = QString("nmcli connection show %1").arg(this->enterpriseConnectionName);
    const QString create = QString("nmcli connection add save no type wifi con-name %1 "
                                   "autoconnect no connection.permissions sddm "
                                   "wifi-sec.key-mgmt wpa-eap 802-1x.password-flags 2 "
                                   "802-1x.eap peap 802-1x.phase2-auth mschapv2 "
                                   "802-1x.identity %2 802-1x.domain-suffix-match %3 ssid %4")
                            .arg(this->enterpriseConnectionName).arg(login).arg(config->wifiEapDomain()).arg(ssid);
    const QString connectTo = QString("echo %1 | nmcli connection up %2 --ask").arg(pass).arg(this->enterpriseConnectionName);

    const QString cmd = deleteIfExist + " ; " + checkExist + " || (" + create + " && " + enableWifi + " && " + connectTo + ")";
    createWifi->start("/bin/sh", QStringList() << "-c" << cmd);
}

bool AuthService::guestEnabled() const
{
    return config->guestEnabled();
}

bool AuthService::mosAuthBtnEnabled() const
{
    return config->mosAuthBtnEnabled();
}

bool AuthService::loginLowercaseOnly() const
{
    return config->loginLowercaseOnly();
}

bool AuthService::wifiEapDomainIsSet() const
{
    return config->wifiEapDomain().length() > 0;
}

QString AuthService::wifiEapSsid() const
{
    return config->wifiEapSsid();
}