import QtQuick 2.15
import QtQuick.Controls 2.15 as QQC2

import org.kde.kirigami 2.13 as Kirigami
import org.kde.plasma.core 2.0 as PlasmaCore

/*
 * A model (UserModel) with a list of users to show in the view.
 *
 * type: {
 *  name: string,
 *  realName: string,
 *  homeDir: string,
 *  icon: string,
 *  needsPassword?: bool,
 * }
 */
QQC2.ComboBox {
    id: userSelect

    /*
     * HACK: заметка по поводу присутствия здесь скрытого ListView, который не используется для отображения.
     * из ComboBox довольно проблематично прочесть свойство текущего элемента из модели
     * (например needsPassword или icon), поскольку у ComboBox нет `currentItem` (в отличие от ListView).
     *
     * Прочитать свойство можно напрямую через метод model.data(index, dataRole), однако
     * SDDM не экспортирует Enum с DataRole свойств модели, поэтому читать просто по ручному int,
     * тоже нежелательно, так как при изменении Enum в SDDM (UserModel.h) может измениться и это значение.
     *
     * Соответственно, тут применяется костыль с привязкой `currentIndex` у UserSelect и UserList (dataAccessList),
     * и таким образом читать `currentItem` у UserList, чтобы получить доступ к свойству `needsPassword` и `icon`.
     *
     * Наверное, попробовать решить это можно или предложением добавить `currentItem` в `ComboBox` в Qt,
     * либо добавить в SDDM метод .get(index) для модели UserModel (или экспортированием из SDDM Enum с DataRole).
     *
     */
    ListView {
        id: dataAccessList

        visible: false
        cacheBuffer: 0

        currentIndex: userSelect.currentIndex
        model: userSelect.model
        delegate: Item {
            property string avatarPath: model.icon
            property string homeDir: model.homeDir
            property bool needsPassword: model.needsPassword !== undefined ? model.needsPassword : true

            visible: false
        }
    }

    property alias currentItem: dataAccessList.currentItem

    readonly property bool currentUserNeedsPassword: currentItem ? currentItem.needsPassword : true
    readonly property string selectedUser: currentValue ?? ""

    /*
     * Signals that a user was explicitly selected
     */
    signal userSelected()

    QtObject
    {
        id: customColors

        property var highlightBackground: Kirigami.ColorUtils.adjustColor(PlasmaCore.Theme.viewHoverColor, {alpha: 0.4*255})
        property var openedComboBoxBackground: Kirigami.ColorUtils.adjustColor(PlasmaCore.Theme.backgroundColor, {alpha: 0.8*255})
    }

    textRole: "realName"
    valueRole: "name"

    // Colors settings.
    palette: PlasmaCore.Theme.palette
    Binding {
        target: userSelect.indicator
        property: 'color'
        value: PlasmaCore.Theme.textColor
    }
    Binding {
        target: userSelect.background
        property: 'color'
        value: down ? customColors.openedComboBoxBackground : PlasmaCore.Theme.backgroundColor
    }

    // Display selected user in ComboBox.
    contentItem: UserSelectItem {
        text: currentText || currentValue || ""
        subtitle: (config.showLoginInUserSelect === "true") ? (currentValue || "") : ""
        avatarPath: currentItem ? currentItem.avatarPath : ""
        homeDir: currentItem ? currentItem.homeDir : ""
        tooltipVisible: Kirigami.Settings.tabletMode ? pressed : hovered

        rightPadding: userSelect.indicator.width
        bold: true
    }

    // Delegate for listItem in ComboBox popup list.
    delegate: QQC2.ItemDelegate
    {
        property bool selected: index === userSelect.currentIndex
        highlighted: index === userSelect.highlightedIndex

        implicitWidth: userSelect.width
        implicitHeight: userSelect.height

        background: Rectangle
        {
            color: {
                if (selected) {
                    PlasmaCore.Theme.buttonHoverColor;
                } else if (highlighted) {
                    customColors.highlightBackground;
                } else {
                    PlasmaCore.Theme.backgroundColor;
                }
            }
        }

        contentItem: UserSelectItem {
            text: model.realName || model.name
            subtitle: config.showLoginInUserSelect === "true" ? model.name : ""
            avatarPath: model.icon
            homeDir: model.homeDir
            tooltipVisible: Kirigami.Settings.tabletMode ? pressed : hovered
            textColor: selected ? PlasmaCore.Theme.complementaryTextColor : PlasmaCore.Theme.textColor
        }
    }

    onCurrentIndexChanged: userSelect.userSelected()

    Keys.onUpPressed: {
        if (userSelect.popup.visible) {
            userSelect.decrementCurrentIndex();
        } else {
            userSelect.popup.open();
        }
    }

    Keys.onDownPressed: {
        if (userSelect.popup.visible) {
            userSelect.incrementCurrentIndex();
        } else {
            userSelect.popup.open();
        }
    }
}
