/*
    SPDX-FileCopyrightText: 2016 David Edmundson <davidedmundson@kde.org>

    SPDX-License-Identifier: LGPL-2.0-or-later
*/

import QtQuick 2.15

import QtQuick.Layouts 1.15

import org.kde.plasma.core 2.0 as PlasmaCore
import org.kde.plasma.components 3.0 as PlasmaComponents3

FocusScope {
    id: root

    /*
     * Any message to be displayed to the user, visible above the text fields
     */
    property alias notificationMessage: notificationsLabel.text

    /*
     * A list of Items (typically ActionButtons) to be shown in a Row beneath the prompts
     */
    property alias actionItems: actionItemsLayout.children

    /*
     * Whether to show or hide the list of action items as a whole.
     */
    property alias actionItemsVisible: actionItemsLayout.visible

    /*
     * A model with a list of users to show in the view.
     * There are different implementations in sddm greeter (UserModel) and
     * KScreenLocker (SessionsModel), so some roles will be missing.
     *
     * type: {
     *  name: string,
     *  realName: string,
     *  homeDir: string,
     *  icon: string,
     *  iconName?: string,
     *  needsPassword?: bool,
     *  displayNumber?: string,
     *  vtNumber?: int,
     *  session?: string
     *  isTty?: bool,
     * }
     */
    property var userListModel

    /*
     * Self explanatory
     */
    property int userListCurrentIndex: 0

    enum UserListDisplayType {
        ListView,
        Select,
        None
    }

    property int userListDisplayType: SessionManagementScreen.UserListDisplayType.ListView
    property bool showUserListView: userListDisplayType === SessionManagementScreen.UserListDisplayType.ListView
    property bool showUserSelect: userListDisplayType === SessionManagementScreen.UserListDisplayType.Select

    property var userList: showUserListView ? userListView : userSelect

    property alias prompts: prompts

    property int fontSize: PlasmaCore.Theme.defaultFont.pointSize + 2

    default property alias _children: innerLayout.children

    signal userSelected()

    ColumnLayout {
        anchors.fill: parent
        ColumnLayout {
            Layout.preferredHeight: showUserListView ? root.height * 0.5 : root.height * 0.3
            Layout.maximumHeight: root.height * 0.5
            UserList {
                id: userListView

                property bool isSmallSize: root.height < 650

                // Не уверен, что настоящая модель для скрытого списка как-то
                // влияет на производительность, но на всякий случай передаем пустую
                // модель-заглушку, если этот список не отображается (аналогично и для userSelect).
                model: showUserListView ? userListModel : null
                currentIndex: userListCurrentIndex

                visible: showUserListView
                Layout.alignment: Qt.AlignBottom
                Layout.fillWidth: true
                Layout.topMargin: PlasmaCore.Units.gridUnit * 0.5
                Layout.bottomMargin:
                {
                    if (constrainText && !isSmallSize) {
                        PlasmaCore.Units.gridUnit * 3
                    }
                    else if (constrainText && isSmallSize) {
                        PlasmaCore.Units.gridUnit * 1
                    }
                    else {
                        0
                    }
                }
                fontSize: root.fontSize
                // bubble up the signal
                onUserSelected: root.userSelected()

                smallSize: isSmallSize
            }
        }

        //goal is to show the prompts, in ~16 grid units high, then the action buttons
        //but collapse the space between the prompts and actions if there's no room
        //ui is constrained to 16 grid units wide, or the screen
        ColumnLayout {
            id: prompts
            Layout.alignment: Qt.AlignCenter
            Layout.topMargin: PlasmaCore.Units.gridUnit * 0.5
            Layout.minimumHeight: implicitHeight
            Layout.preferredHeight: root.height * 0.5
            PlasmaComponents3.Label {
                id: notificationsLabel
                font.pointSize: root.fontSize
                Layout.minimumHeight: PlasmaCore.Units.gridUnit * 2.5
                Layout.maximumWidth: PlasmaCore.Units.gridUnit * 24
                Layout.alignment: Qt.AlignHCenter
                Layout.fillWidth: true
                Layout.bottomMargin: PlasmaCore.Units.gridUnit * 0.5
                horizontalAlignment: Text.AlignHCenter
                wrapMode: Text.WordWrap
                font.italic: true
            }
            ColumnLayout {
                Layout.minimumHeight: implicitHeight
                Layout.maximumHeight: implicitHeight + PlasmaCore.Units.gridUnit * 3
                Layout.maximumWidth: PlasmaCore.Units.gridUnit * 16
                Layout.alignment: Qt.AlignHCenter

                ColumnLayout {
                    id: innerLayout
                    Layout.alignment: Qt.AlignHCenter | Qt.AlignTop
                    Layout.bottomMargin: PlasmaCore.Units.gridUnit

                    UserSelect {
                        id: userSelect

                        model: showUserSelect ? userListModel : null
                        currentIndex: userListCurrentIndex

                        visible: showUserSelect
                        Layout.fillWidth: true
                        // bubble up the signal
                        onUserSelected:
                        {
                            root.userSelected()
                        }
                    }

                }
            }
            Row { //deliberately not rowlayout as I'm not trying to resize child items
                id: actionItemsLayout
                Layout.alignment: Qt.AlignHCenter
                spacing: PlasmaCore.Units.largeSpacing * 0.5
                Layout.minimumHeight: implicitHeight * 2
            }
            Item {
                Layout.fillHeight: true
            }
        }
    }
}
