import "components"

import QtQuick 2.15
import QtQuick.Layouts 1.15
import QtQuick.Controls 2.15 as QQC2

import org.kde.plasma.core 2.0 as PlasmaCore
import org.kde.plasma.components 3.0 as PlasmaComponents3
import org.kde.plasma.extras 2.0 as PlasmaExtras
import org.kde.kitemmodels 1.0 as KItemModels

import org.kde.kirigami 2.19 as Kirigami

import org.kde.plasma.networkmanagement 0.2 as PlasmaNM

import QtGraphicalEffects 1.15

import org.mos.auth 0.1 as MosAuthPlugin

FocusScope {
    id: root

    readonly property int defaultOffset: PlasmaCore.Units.gridUnit * 10
    readonly property int fontSize: PlasmaCore.Theme.defaultFont.pointSize + 2

    readonly property int selectedConnectionSecurityType: getSelectedConnectionSecurityType()

    readonly property bool predictableWirelessPassword: isConnectionHasPassword()
    readonly property bool selectedConnectionActivated: isSelectedConnectionActivated()
    readonly property bool selectedConnectionIsEnterprise: isSelectedConnectionHasEnterpriseSecurity()
    readonly property bool selectedConnectionIsAllowedEap: isSelectedConnectionHasAllowedEap()
    readonly property bool selectedConnectionSaved: isSelectedConnectionSaved()
    readonly property bool waitingConnectionFinish: (wirelessStatus === PlasmaNM.Enums.Deactivating
                                                                 || wirelessStatus === PlasmaNM.Enums.Activating)

    readonly property string wirelessName: getProcessedWifiName()

    property string userSelectedConnection: ""
    property bool modelInitialized: false

    // From parent
    required property int wirelessStatus
    required property string wirelessStatusMsg
    required property MosAuthPlugin.AuthService authService

    function getDataByRole(role: PlasmaNM.NetworkModel, index = -1) {
        index = (index === -1) ? wifiSelect.currentIndex : index;
        return filteredModel.data(filteredModel.index(index, 0), role);
    }

    function getSelectedConnectionSecurityType(): int {
        return getDataByRole(PlasmaNM.NetworkModel.SecurityTypeRole);
    }

    function getProcessedWifiName(): string {
        const noname = "...";

        if (wirelessStatus === PlasmaNM.Enums.Deactivated || !modelInitialized) {
            return noname;
        }

        let N = filteredModel.rowCount();
        for (let i = 0; i < N; ++i) {
            const connectionState = getDataByRole(PlasmaNM.NetworkModel.ConnectionStateRole, i);
            const deactivating = (wirelessStatus === PlasmaNM.Enums.Deactivating
                                        && connectionState === PlasmaNM.Enums.Activated);
            if (connectionState === wirelessStatus || deactivating) {
                return getDataByRole(PlasmaNM.NetworkModel.ItemUniqueNameRole, i);
            }
        }
        return noname;
    }

    function isConnectionHasPassword(): bool {
        const devicePath = getDataByRole(PlasmaNM.NetworkModel.DevicePathRole);
        if (devicePath === undefined) {
            return false;
        }
        const securityType = selectedConnectionSecurityType;
        const passwordIsStatic = (securityType === PlasmaNM.Enums.StaticWep
                                || securityType === PlasmaNM.Enums.WpaPsk
                                || securityType === PlasmaNM.Enums.Wpa2Psk
                                || securityType === PlasmaNM.Enums.Wpa2Eap
                                || securityType === PlasmaNM.Enums.SAE);
        const uuid = getDataByRole(PlasmaNM.NetworkModel.UuidRole) ?? "";
        return !uuid && passwordIsStatic;
    }

    function isSelectedConnectionActivated(): bool {
        const selectedItemName = getDataByRole(PlasmaNM.NetworkModel.ItemUniqueNameRole);
        return (wirelessStatus === PlasmaNM.Enums.Activated && selectedItemName === wirelessName);
    }

    function isSelectedConnectionHasEnterpriseSecurity(): bool {
        const securityType = selectedConnectionSecurityType;
        const enterpriseSecurity = (securityType === PlasmaNM.Enums.DynamicWep
                                  || securityType === PlasmaNM.Enums.Leap
                                  || securityType === PlasmaNM.Enums.WpaEap
                                  || securityType === PlasmaNM.Enums.Wpa2Eap
                                  || securityType === PlasmaNM.Enums.Wpa3SuiteB192);
        return enterpriseSecurity;
    }

    function isSelectedConnectionHasAllowedEap(): bool {
        return (selectedConnectionSecurityType === PlasmaNM.Enums.Wpa2Eap
                && authService.wifiEapDomainIsSet);
    }

    function isSelectedConnectionSaved(): bool {
        const uuid = getDataByRole(PlasmaNM.NetworkModel.UuidRole) ?? "";
        return uuid;
    }

    function handleWifiConnection() {
        const devicePath = getDataByRole(PlasmaNM.NetworkModel.DevicePathRole);
        if (devicePath === undefined) {
            return;
        }

        const connectionState = getDataByRole(PlasmaNM.NetworkModel.ConnectionStateRole);
        const connectionPath = getDataByRole(PlasmaNM.NetworkModel.ConnectionPathRole);
        const specificPath = getDataByRole(PlasmaNM.NetworkModel.SpecificPathRole);
        const uuid = getDataByRole(PlasmaNM.NetworkModel.UuidRole) ?? "";
        const password = wifiPasswordInput.text.trim();

        if (selectedConnectionIsAllowedEap) {
            const login = eapLoginInput.text.trim();
            const ssid = getDataByRole(PlasmaNM.NetworkModel.SsidRole);
            authService.handleEapWifi(ssid, login, password);
        } else if (connectionState === PlasmaNM.Enums.Deactivated) {
            if (!predictableWirelessPassword && !uuid) {
                // Новое соединение без пароля
                handler.addAndActivateConnection(devicePath, specificPath);
            } else if (password !== "" && !uuid) {
                // Новое соединение с паролем
                handler.addAndActivateConnection(devicePath, specificPath, password);
            } else {
                // Старое соединение
                handler.activateConnection(connectionPath, devicePath, specificPath);
            }
        } else if (connectionState === PlasmaNM.Enums.Activated
                   || connectionState === PlasmaNM.Enums.Activating) {
            handler.deactivateConnection(connectionPath, devicePath);
        }

        wifiPasswordInput.clear();
    }

    function removeWifiConnection() {
        const connectionPath = getDataByRole(PlasmaNM.NetworkModel.ConnectionPathRole);
        if (connectionPath === undefined) {
            return;
        }

        handler.removeConnection(connectionPath);
    }

    objectName: "WirelessNetworkManager"

    Binding {
        target: wifiSelect.popup.contentItem
        property: "implicitHeight"
        value: Math.min(root.height / 3, wifiSelect.popup.contentItem.contentHeight)
    }

    PlasmaNM.Handler {
        id: handler
    }

    PlasmaNM.NetworkModel {
        id: connectionModel
    }

    // For sort like in plasma-nm
    PlasmaNM.AppletProxyModel {
        id: appletProxyModel

        sourceModel: connectionModel
    }

    // This proxy filters only Wi-Fi connections.
    KItemModels.KSortFilterProxyModel {
        id: filteredModel

        sourceModel: appletProxyModel
        filterRowCallback: (sourceRow, sourceParent) => {
            const value = sourceModel.data(
                sourceModel.index(sourceRow, 0, sourceParent),
                PlasmaNM.NetworkModel.TypeRole
            );
            return value === PlasmaNM.Enums.Wireless;
        }

        // When the index of the selected connection changes due to sorting,
        // we need to change the current index for the drop-down list.
        onDataChanged: {
            const currentName = getDataByRole(PlasmaNM.NetworkModel.ItemUniqueNameRole);
            if (userSelectedConnection === "" || userSelectedConnection === currentName) {
                return;
            }

            let N = filteredModel.rowCount();
            for (let i = 0; i < N; ++i) {
                const name = getDataByRole(PlasmaNM.NetworkModel.ItemUniqueNameRole, i);
                if (name === userSelectedConnection) {
                    wifiSelect.currentIndex = i;
                    return;
                }
            }
            wifiSelect.currentIndex = -1;
        }

        Component.onCompleted: {
            modelInitialized = true;

            // First in list as initial selected connection.
            const initialName = getDataByRole(PlasmaNM.NetworkModel.ItemUniqueNameRole);
            if (initialName !== undefined) {
                userSelectedConnection = initialName;
            }
        }
    }

    Timer {
        id: scanTimer

        interval: 10000
        repeat: true
        running: true

        onTriggered: handler.requestScan()
    }

    ColumnLayout {
        anchors.fill: parent

        FocusScope {
            Layout.preferredHeight: parent.height - root.defaultOffset
            Layout.preferredWidth: parent.width

            ColumnLayout {
                anchors.fill: parent

                ColumnLayout {
                    Layout.alignment: Qt.AlignCenter
                    Layout.maximumWidth: PlasmaCore.Units.gridUnit * 16
                    Layout.topMargin: root.defaultOffset

                    PlasmaComponents3.Label {
                        Layout.bottomMargin: PlasmaCore.Units.gridUnit
                        Layout.fillWidth: true
                        font.bold: true
                        font.pointSize: root.fontSize + 1
                        horizontalAlignment: Text.AlignHCenter
                        text: "Настройка беспроводной сети"
                    }

                    QQC2.ProgressBar {
                        id: progress

                        Layout.bottomMargin: PlasmaCore.Units.gridUnit
                        Layout.fillWidth: true
                        indeterminate: true
                        visible: root.waitingConnectionFinish
                    }

                    RowLayout {
                        id: connectionStatus

                        Layout.bottomMargin: PlasmaCore.Units.gridUnit / 2
                        Layout.fillWidth: true
                        visible: true

                        ColumnLayout {
                            PlasmaComponents3.Label {
                                id: connectionNameLabel

                                Layout.fillWidth: true
                                Layout.maximumHeight: PlasmaCore.Units.gridUnit * 8
                                elide: Text.ElideRight
                                font.pointSize: root.fontSize + 2
                                horizontalAlignment: Text.AlignRight
                                text: (root.wirelessStatus !== PlasmaNM.Enums.Deactivated)
                                        ? root.wirelessName : "..."
                                wrapMode: Text.Wrap
                            }

                            PlasmaComponents3.Label {
                                id: connectionStatusLabel

                                Layout.fillWidth: true
                                font.pointSize: root.fontSize - 4
                                horizontalAlignment: Text.AlignRight
                                text: root.wirelessStatusMsg
                            }
                        }

                        PlasmaCore.IconItem {
                            Layout.minimumHeight: PlasmaCore.Units.gridUnit * 2
                            Layout.minimumWidth: PlasmaCore.Units.gridUnit * 2
                            colorGroup: PlasmaCore.ColorScope.colorGroup
                            source: "network-wireless"
                        }
                    }

                    PlasmaComponents3.Label {
                        id: enterpriseConnectionNotification

                        Layout.bottomMargin: PlasmaCore.Units.gridUnit
                        Layout.fillWidth: true
                        font.italic: true
                        font.pointSize: root.fontSize
                        horizontalAlignment: Text.AlignHCenter
                        text: "До входа в систему нельзя подключаться к корпоративной сети.\n\nПопробуйте другое соединение."
                        visible: root.selectedConnectionIsEnterprise && !root.selectedConnectionIsAllowedEap
                        wrapMode: Text.Wrap
                    }

                    QQC2.ComboBox {
                        id: wifiSelect

                        function popupOpenOnNavigate(event) {
                            event.accepted = false;
                            wifiSelect.popup.open();
                        }

                        Layout.fillWidth: true
                        model: filteredModel
                        textRole: "ItemUniqueName"

                        enabled: !root.waitingConnectionFinish

                        displayText: {
                            if (count == 0) {
                                return "Нет доступных сетей";
                            }

                            return (currentIndex !== -1) ? currentText : "Выберите сеть";
                        }

                        delegate: QQC2.ItemDelegate {
                            highlighted: wifiSelect.highlightedIndex === index
                            // HACK: The system temporarily does not have the necessary icons.
                            // So we change the name to those icons that are in the system for sure.
                            icon.name: model.ConnectionIcon
                                .replace("-locked", "")
                                .replace("wireless-", "wireless-connected-")
                                .replace(/40|60/g, "50")
                                .replace("20", "25")
                                .replace("80", "75")

                            text: ItemUniqueName
                            width: wifiSelect.width

                            QQC2.ToolTip.delay: 1000
                            QQC2.ToolTip.visible: hovered
                            QQC2.ToolTip.text: ItemUniqueName
                        }

                        Keys.onDownPressed: event => popupOpenOnNavigate(event)
                        Keys.onUpPressed: event => popupOpenOnNavigate(event)

                        onActivated: (index) => {
                            userSelectedConnection = root.getDataByRole(PlasmaNM.NetworkModel.ItemUniqueNameRole, index);
                            const primaryInput = eapLoginInput.visible ? eapLoginInput : wifiPasswordInput;
                            primaryInput.forceActiveFocus();
                        }
                    }

                    PlasmaComponents3.TextField {
                        id: eapLoginInput

                        Layout.fillWidth: true
                        visible: root.selectedConnectionIsAllowedEap
                        focus: true
                        font.pointSize: fontSize + 1
                        placeholderText: i18nd("plasma_lookandfeel_org.kde.lookandfeel", "Username")

                        onAccepted: wifiPasswordInput.forceActiveFocus()
                    }

                    RowLayout {
                        PlasmaExtras.PasswordField {
                            id: wifiPasswordInput

                            Layout.fillWidth: true
                            enabled: root.predictableWirelessPassword
                                     && (!root.selectedConnectionIsEnterprise
                                        || root.selectedConnectionIsAllowedEap)
                                     && !root.waitingConnectionFinish
                            focus: !eapLoginInput.visible
                            font.pointSize: root.fontSize + 1
                            placeholderText: i18nd("plasma_lookandfeel_org.kde.lookandfeel", "Password")

                            onAccepted: root.handleWifiConnection()
                        }

                        PlasmaComponents3.Button {
                            icon.name: "delete"
                            display: QQC2.AbstractButton.IconOnly
                            Layout.preferredHeight: wifiPasswordInput.implicitHeight
                            visible: root.selectedConnectionSaved
                            enabled: !root.waitingConnectionFinish
                            onClicked: removeConnectionDialog.open()
                        }
                    }

                    PlasmaComponents3.Button {
                        id: wifiConnectButton

                        readonly property bool activatingConnection:
                            root.wirelessStatus === PlasmaNM.Enums.Activating

                        Layout.fillWidth: true
                        Layout.preferredHeight: wifiPasswordInput.implicitHeight

                        enabled: (!root.selectedConnectionIsEnterprise
                                    || root.selectedConnectionIsAllowedEap)
                            && root.wirelessStatus !== PlasmaNM.Enums.Deactivating

                        text: {
                            if (activatingConnection) {
                                return "Остановить подключение"
                            }

                            return !root.selectedConnectionActivated ? "Подключить" : "Отключить";
                        }

                        background: Rectangle {
                            Layout.fillHeight: true
                            Layout.fillWidth: true
                            color: {
                                if (parent.hovered) {
                                    !parent.activatingConnection ? "#0074ff" : "#d86200";
                                } else if (!parent.enabled) {
                                    "#a3a3a3";
                                } else {
                                    !parent.activatingConnection ? "#1d99f3" : "#ff7400";
                                }
                            }
                            radius: 6
                        }
                        contentItem: Text {
                            color: "#ffffff"
                            elide: Text.ElideRight
                            font: parent.font
                            horizontalAlignment: Text.AlignHCenter
                            text: parent.text
                            verticalAlignment: Text.AlignVCenter
                        }

                        Keys.onEnterPressed: clicked()
                        Keys.onReturnPressed: clicked()
                        onClicked: root.handleWifiConnection()
                    }
                }
            }
        }

        /// Кнопка с действием для возврата к списку пользователей.
        ActionButton {
            Layout.alignment: Qt.AlignHCenter
            Layout.minimumHeight: root.defaultOffset
            fontSize: root.fontSize - 1
            iconSource: "system-user-list"
            text: i18nd("plasma_lookandfeel_org.kde.lookandfeel", "List Users")

            onClicked: mainStack.pop()
        }
    }

    Kirigami.PromptDialog {
        id: removeConnectionDialog
        title: i18nd("plasmanetworkmanagement-libs", "Remove Connection")
        subtitle: i18nd("plasmanetworkmanagement-libs",
                        "Do you want to remove the connection '%1'?",
                        root.getDataByRole(PlasmaNM.NetworkModel.ItemUniqueNameRole));

        standardButtons: Kirigami.Dialog.Ok | Kirigami.Dialog.Cancel
        onAccepted: root.removeWifiConnection()
    }

    Connections {
        target: root

        onWirelessStatusChanged: {
            if (root.wirelessStatus === PlasmaNM.Enums.Activated) {
                mainStack.pop();
            }
        }
    }

    Component.onCompleted: {
        handler.enableWireless(true);
        handler.requestScan();
        wifiPasswordInput.forceActiveFocus();
    }
}
