import "components"

import QtQuick 2.15
import QtQuick.Layouts 1.15
import QtQuick.Controls 2.15 as QQC2

import org.kde.plasma.core 2.0 as PlasmaCore
import org.kde.plasma.components 3.0 as PlasmaComponents3
import org.kde.plasma.extras 2.0 as PlasmaExtras

import QtWebEngine 1.3
import QtGraphicalEffects 1.15

import org.mos.auth 0.1 as MosAuthPlugin

FocusScope {
    id: root

    // From parent
    required property MosAuthPlugin.AuthService authService

    readonly property int defaultOffset: PlasmaCore.Units.gridUnit * 10
    readonly property int fontSize: PlasmaCore.Theme.defaultFont.pointSize + 2
    readonly property int pinLength: 5

    property bool pageAuthCompleted: false
    property bool userDataLoaded: false

    signal loginRequest(string username, string password)
    signal displayRegisterNotification(string message)

    signal waitRegistration()
    signal stopWaitingRegistration()

    // Глобальный роли с сайта
    enum MosRoleType {
        Student = 1,

        Employee = 3,
        Head_Teacher = 8,
        Teacher = 9,

        Admin = 4,
        Admin_Sys = 16,
        Admin_IB = 17,
        Admin_School = 26,
        Admin_OS = 27
    }

    QtObject {
        id: credentials

        property string obrId
        property string login
        property string fullName
        property string pin

        // type: QList<int>
        property var roles: []
    }

    Connections {
        target: authService
        function onRegisterUserFinished(result: boolean)
        {
            debugInfo.log(result);

            if (result)
            {
                loginRequest(credentials.login, credentials.pin);
            }
            else
            {
                root.stopWaitingRegistration();
                root.displayRegisterNotification(`Непредвиденная ошибка\nпри попытке регистрации пользователя!`);
            }
        }
    }

    // Данные в localStorage появляются не сразу (AJAX).
    Timer {
        id: userDataCheckingTimer
        interval: 500
        repeat: true
        onTriggered: function()
        {
            page.runJavaScript('localStorage.getItem("user_data")', function(result)
            {
                if (result !== "" && result !== undefined)
                {
                    userDataCheckingTimer.stop();

                    if (!userDataLoaded)
                    {
                        userDataLoaded = true;
                        parseUserDataFromLS(result);
                    }
                }
            });
        }
    }

    Timer {
        id: reconnectPageTimer
        interval: 5000
        repeat: false
        onTriggered: function()
        {
            page.reload();
        }
    }

    function getRoleForUser(roleId)
    {
        switch (roleId)
        {
        case MosAuth.MosRoleType.Student:
            return MosAuthPlugin.AuthService.Student;

        case MosAuth.MosRoleType.Employee:
        case MosAuth.MosRoleType.Head_Teacher:
        case MosAuth.MosRoleType.Teacher:
            return MosAuthPlugin.AuthService.Teacher;

        case MosAuth.MosRoleType.Admin:
        case MosAuth.MosRoleType.Admin_Sys:
        case MosAuth.MosRoleType.Admin_IB:
        case MosAuth.MosRoleType.Admin_School:
        case MosAuth.MosRoleType.Admin_OS:
            return MosAuthPlugin.AuthService.Admin;

        default:
            return undefined;
        }
    }

    // Вытягиваем данные из объекта "user_data" из localStorage
    function parseUserDataFromLS(result)
    {
        debugInfo.log("parseUserDataFromLS()");

        const userData = JSON.parse(result);

        credentials.obrId = userData.userId;

        let nameRegex = new RegExp(/^[a-z][-a-z0-9]{0,30}$/);

        // Получаем логин.
        if (userData.login !== undefined
                && userData.login !== null
                && userData.login !== ""
                && nameRegex.test(userData.login))
        {
            credentials.login = userData.login;
        }
        else
        {
            credentials.login = `mos-${credentials.obrId}`;
        }

        // Получаем имя.
        const firstName = userData.info.FirstName;
        const lastName = userData.info.LastName;

        credentials.fullName = firstName;

        if (lastName)
        {
            credentials.fullName += " " + lastName;
        }

        debugInfo.log(`${credentials.obrId}: ${credentials.login}, ${credentials.fullName}`);

        // Получаем роли.
        credentials.roles = [];

        for (const role of userData.roles)
        {
            if (role.subsystems.length <= 0)
            {
                continue;
            }

            const userRole = getRoleForUser(role.id);
            if (userRole !== undefined)
            {
                credentials.roles.push(userRole);
            }
        }

        // Если ролей с сайта так и не было получено, присвоим Student
        if (credentials.roles.length == 0)
        {
            credentials.roles.push(MosAuthPlugin.AuthService.Student);
        }

        // Переходим на форму регистрации, если был получен obrId.
        if (credentials.obrId !== undefined)
        {
            mosAuthStack.replace(registerForm);
        }
    }

    // Проверяем, авторизовались ли мы.
    function checkIsAuthDone(rawCookie)
    {
        debugInfo.log("checkIsAuthDone()");

        let cookie = {};

        rawCookie.split(';').forEach(function(el) {
            let [key, value] = el.split('=');
            cookie[key.trim()] = value;
        });

        let token = cookie["aupd_token"];
        let obrId = cookie["obr_id"];

        // TODO: for test
        /*credentials.login = "testik";
        credentials.fullName = "Иван Иванов";
        credentials.obrId = "55555";
        mosAuthStack.replace(registerForm);*/

        if (token !== undefined && obrId !== undefined)
        {
            pageAuthCompleted = true;
            debugInfo.log("authDone");

            // Ожидаем появление данных в localStorage.
            userDataCheckingTimer.start();
        }
    }

    // Запрос на создание локальной учетной записи через authService.
    function requestRegisterUser(pin, pinConfirm)
    {
        debugInfo.log("requestRegisterUser()");

        const pinTrimmed = pin.trim();
        const pinConfirmTrimmed = pinConfirm.trim();

        if (pinTrimmed.length < pinLength)
        {
            root.displayRegisterNotification(`Цифровой пароль должен быть не меньше\n ${root.pinLength} символов!`);
        }
        else if (pinTrimmed !== pinConfirmTrimmed)
        {
            root.displayRegisterNotification(`Введенные пароли не совпадают!`);
        }
        else
        {
            debugInfo.log("request to register user: " + credentials.login);
            root.waitRegistration();

            credentials.pin = pin;
            authService.registerUser(credentials.obrId,
                                     credentials.fullName,
                                     credentials.login,
                                     credentials.pin,
                                     credentials.roles);
        }
    }

    ColumnLayout
    {
        anchors.fill: parent

        // Используем laoyut для центрирования нашего StackView.
        ColumnLayout
        {
            Layout.preferredHeight: parent.height - root.defaultOffset
            Layout.preferredWidth: parent.width

            QQC2.StackView
            {
                id: mosAuthStack

                Layout.preferredWidth: Math.max( Math.min(800, root.width - root.defaultOffset), root.width / 1.75 )
                Layout.preferredHeight: parent.height - root.height * 0.05
                Layout.alignment: Qt.AlignCenter

                // Это по аналогии с mainStack из KDE Breeze.
                hoverEnabled: loginScreenRoot.uiVisible ? undefined : false
                focus: true

                Timer {
                    running: true
                    repeat: false
                    interval: 200
                    onTriggered: mosAuthStack.forceActiveFocus()
                }

                // Начальный элемент - веб-страница.
                initialItem: Item
                {
                    function setShowProgressBar(show)
                    {
                        progress.visible = show;
                        page.visible = !show;
                    }

                    Rectangle {
                        color: "white"
                        anchors.fill: parent
                    }

                    ColumnLayout
                    {
                        anchors.centerIn: parent

                        QQC2.ProgressBar {
                            id: progress
                            visible: true
                            indeterminate: true
                        }

                        Text {
                            id: name
                            text: "Подключаемся к mos.ru"
                            Layout.topMargin: PlasmaCore.Units.gridUnit / 4
                            Layout.alignment: Qt.AlignHCenter
                        }
                    }

                    WebEngineView
                    {
                        id: page
                        url: "https://login.mos.ru/sps/login/methods/password?bo=%2Fsps%2Foauth%2Fae%3Fresponse_type%3Dcode%26access_type%3Doffline%26client_id%3Ddnevnik.mos.ru%26scope%3Dopenid%2Bprofile%2Bbirthday%2Bcontacts%2Bsnils%2Bblitz_user_rights%2Bblitz_change_password%26redirect_uri%3Dhttps%253A%252F%252Fschool.mos.ru%252Fv3%252Fauth%252Fsudir%252Fcallback"

                        profile: profile

                        visible: false

                        anchors.fill: parent

                        WebEngineProfile {
                            id: profile

                            persistentCookiesPolicy: WebEngineProfile.NoPersistentCookies
                            httpCacheType: WebEngineProfile.NoCache
                        }

                        onLoadingChanged: function(loadingRequest)
                        {
                            if (pageAuthCompleted)
                            {
                                return;
                            }

                            if (loadingRequest.status === WebEngineLoadRequest.LoadSucceededStatus)
                            {
                                parent.setShowProgressBar(false);
                                page.runJavaScript('document.cookie', checkIsAuthDone);
                            }
                            else if (loadingRequest.status === WebEngineLoadRequest.LoadFailedStatus)
                            {
                                parent.setShowProgressBar(true);
                                debugInfo.log("try reconnect again");
                                reconnectPageTimer.start();
                            }
                        }
                    }
                }

                Component
                {
                    id: registerForm

                    FocusScope
                    {
                        ColumnLayout
                        {
                            anchors.fill: parent
                            ColumnLayout
                            {
                                Layout.minimumHeight: implicitHeight
                                Layout.maximumWidth: PlasmaCore.Units.gridUnit * 16
                                Layout.alignment: Qt.AlignCenter

                                Layout.topMargin: PlasmaCore.Units.gridUnit * 10

                                PlasmaComponents3.Label {
                                    font.pointSize: fontSize + 1
                                    font.bold: true
                                    Layout.fillWidth: true
                                    horizontalAlignment: Text.AlignHCenter
                                    text: "Установка пароля, состоящего из цифр, для учетной записи"
                                    Layout.bottomMargin: PlasmaCore.Units.gridUnit
                                }

                                ColumnLayout
                                {
                                    id: busyIndicatorLayout
                                    visible: false

                                    PlasmaComponents3.Label {
                                        visible: true
                                        Layout.fillWidth: true
                                        font.italic: true
                                        font.pointSize: fontSize
                                        horizontalAlignment: Text.AlignHCenter
                                        wrapMode: Text.WordWrap
                                        text: `Идет регистрация пользователя.\nПожалуйста, подождите...`
                                        Layout.bottomMargin: PlasmaCore.Units.gridUnit / 2
                                    }

                                    PlasmaComponents3.BusyIndicator
                                    {
                                        Layout.fillWidth: true
                                    }

                                    Layout.bottomMargin: PlasmaCore.Units.gridUnit
                                }

                                PlasmaComponents3.Label {
                                    id: registerNotification
                                    visible: false
                                    font.pointSize: fontSize + 1
                                    font.italic: true
                                    Layout.fillWidth: true
                                    horizontalAlignment: Text.AlignHCenter
                                    wrapMode: Text.WordWrap
                                    text: ""
                                    Layout.bottomMargin: PlasmaCore.Units.gridUnit
                                }

                                PlasmaComponents3.Label {
                                    id: nameLabel
                                    font.pointSize: fontSize + 2
                                    Layout.fillWidth: true
                                    horizontalAlignment: Text.AlignHCenter
                                    wrapMode: Text.WordWrap
                                    text: credentials.fullName
                                    Layout.bottomMargin: PlasmaCore.Units.gridUnit / 2
                                }

                                PlasmaComponents3.TextField
                                {
                                    id: loginInput
                                    font.pointSize: fontSize + 1
                                    Layout.fillWidth: true

                                    placeholderText: "Введите логин"

                                    text: credentials.login

                                    readOnly: true
                                    enabled: false

                                    onAccepted: {
                                        pinInput.forceActiveFocus()
                                    }
                                }

                                PlasmaExtras.PasswordField
                                {
                                    id: pinInput
                                    font.pointSize: fontSize + 1
                                    Layout.fillWidth: true

                                    placeholderText: "Пароль цифрами"
                                    focus: true

                                    validator: RegExpValidator { regExp: /[0-9]+/ }

                                    Keys.onEscapePressed: {
                                        mosAuthStack.currentItem.forceActiveFocus();
                                    }
                                }

                                PlasmaExtras.PasswordField
                                {
                                    id: pinInputConfirm
                                    font.pointSize: fontSize + 1
                                    Layout.fillWidth: true

                                    placeholderText: "Повторно введите пароль"

                                    validator: RegExpValidator { regExp: /[0-9]+/ }

                                    Keys.onEscapePressed: {
                                        mosAuthStack.currentItem.forceActiveFocus();
                                    }

                                    onAccepted: {
                                        if (loginButton.visible) {
                                            loginButton.clicked();
                                        }
                                    }
                                }

                                PlasmaComponents3.Button {
                                    id: loginButton
                                    Layout.preferredHeight: pinInput.implicitHeight
                                    Layout.fillWidth: true

                                    contentItem: Text {
                                        text: parent.text
                                        font: parent.font
                                        color: "#ffffff"
                                        horizontalAlignment: Text.AlignHCenter
                                        verticalAlignment: Text.AlignVCenter
                                        elide: Text.ElideRight
                                    }

                                    background: Rectangle {
                                        Layout.fillWidth: true
                                        Layout.fillHeight: true
                                        color: parent.hovered ?  "#0074ff" : "#1d99f3"
                                        radius: 6
                                    }

                                    text: "Сохранить"

                                    onClicked: requestRegisterUser(pinInput.text,
                                                                   pinInputConfirm.text)
                                    Keys.onEnterPressed: clicked()
                                    Keys.onReturnPressed: clicked()
                                }

                                Connections {
                                    target: root
                                    function onDisplayRegisterNotification(message) {
                                        pinInput.forceActiveFocus();
                                        registerNotification.text = message;
                                        registerNotification.visible = true;
                                    }

                                    function onWaitRegistration() {
                                        registerNotification.visible = false;

                                        busyIndicatorLayout.visible = true;
                                        footer.enabled = false;
                                        mainStack.enabled = false;

                                        loginButton.forceActiveFocus();
                                    }

                                    function onStopWaitingRegistration() {
                                        busyIndicatorLayout.visible = false;
                                        footer.enabled = true;
                                        mainStack.enabled = true;
                                    }
                                }
                            }
                        }
                    }
                }

                replaceEnter: Transition {
                    PropertyAnimation {
                        property: "opacity"
                        from: 0
                        to:1
                        duration: 200
                    }
                }
                replaceExit: Transition {
                    PropertyAnimation {
                        property: "opacity"
                        from: 1
                        to:0
                        duration: 200
                    }
                }
            }
        }

        // Кнопка с действием для возврата к списку пользователей.
        ActionButton {
            Layout.minimumHeight: root.defaultOffset;
            Layout.alignment: Qt.AlignHCenter;
            iconSource: "system-user-list";
            text: i18nd("plasma_lookandfeel_org.kde.lookandfeel", "List Users");
            fontSize: parseInt(config.fontSize) + 1;
            onClicked: mainStack.pop();
        }
    }
}
