/*
    SPDX-FileCopyrightText: 2016 David Edmundson <davidedmundson@kde.org>

    SPDX-License-Identifier: LGPL-2.0-or-later
*/

import QtQuick 2.15
import QtQuick.Layouts 1.15
import QtQuick.Controls 2.15 as QQC2
import QtGraphicalEffects 1.15

import org.kde.plasma.core 2.0 as PlasmaCore
import org.kde.plasma.components 3.0 as PlasmaComponents3
import org.kde.plasma.extras 2.0 as PlasmaExtras

import org.kde.plasma.networkmanagement 0.2 as PlasmaNM

import org.mos.auth 0.1 as MosAuthPlugin

import "components"
import "components/animation"

// TODO: Once SDDM 0.19 is released and we are setting the font size using the
// SDDM KCM's syncing feature, remove the `config.fontSize` overrides here and
// the fontSize properties in various components, because the theme's default
// font size will be correctly propagated to the login screen

PlasmaCore.ColorScope {
    id: root

    // If we're using software rendering, draw outlines instead of shadows
    // See https://bugs.kde.org/show_bug.cgi?id=398317
    readonly property bool softwareRendering: GraphicsInfo.api === GraphicsInfo.Software

    colorGroup: PlasmaCore.Theme.ComplementaryColorGroup

    width: 1600
    height: 900

    readonly property string i18_networkConnectedMsg: i18ndc("plasmanetworkmanagement-libs", "A network device is connected, with global network connectivity", "Connected");
    readonly property string i18_networkDisconnectedMsg: i18ndc("plasmanetworkmanagement-libs", "There is no active network connection", "Disconnected");
    readonly property string i18_networkConnectingMsg: i18ndc("plasmanetworkmanagement-libs", "A network device is connecting to a network and there is no other available network connection", "Connecting");
    readonly property string i18_networkDisconnectingMsg: i18ndc("plasmanetworkmanagement-libs", "Network connections are being cleaned up", "Disconnecting");

    readonly property int wiredStatus: getWiredStatus()
    readonly property int wirelessStatus: getWirelessStatus()
    readonly property string wirelessStatusMsg: getConnectionStatusString(wirelessStatus)

    property string notificationMessage
    property int loginAttempts: 0

    property alias activeConnections: networkStatus.activeConnections

    function getWirelessStatus()
    {
        const strArr = activeConnections.split("\n");

        for (let i = 0; i < strArr.length; ++i)
        {
            const wifiStr = i18nd("plasmanetworkmanagement-libs", "Wi-Fi");

            if (strArr[i].search(`${wifiStr}: ${i18_networkConnectedMsg}`) !== -1)
            {
                return PlasmaNM.Enums.Activated;
            }

            if (strArr[i].search(`${wifiStr}: ${i18_networkConnectingMsg}`) !== -1
                    || strArr[i].search(i18_networkConnectingMsg) !== -1)
            {
                return PlasmaNM.Enums.Activating;
            }

            if (strArr[i].search(`${wifiStr}:`) !== -1 || strArr[i].search(i18_networkDisconnectingMsg) !== -1)
            {
                return PlasmaNM.Enums.Deactivating;
            }

            if (strArr[i].search(i18_networkDisconnectedMsg) !== -1)
            {
                return PlasmaNM.Enums.Deactivated;
            }
        }

        return PlasmaNM.Enums.Deactivated;
    }

    function getWiredStatus()
    {
        const strArr = activeConnections.split("\n");

        for (let i = 0; i < strArr.length; ++i)
        {
            const wiredEthernetStr = i18nd("plasmanetworkmanagement-libs", "Wired Ethernet");

            if (strArr[i].search(`${wiredEthernetStr}: ${i18_networkConnectedMsg}`) !== -1)
            {
                return PlasmaNM.Enums.Activated;
            }
        }

        return PlasmaNM.Enums.Deactivated;
    }

    function getConnectionStatusString(status)
    {
        switch (status)
        {
        case PlasmaNM.Enums.Activated:
            return i18_networkConnectedMsg;
        case PlasmaNM.Enums.Deactivated:
            return i18_networkDisconnectedMsg;
        case PlasmaNM.Enums.Activating:
            return i18_networkConnectingMsg;
        case PlasmaNM.Enums.Deactivating:
            return i18_networkDisconnectingMsg;
        }
    }

    LayoutMirroring.enabled: Qt.application.layoutDirection === Qt.RightToLeft
    LayoutMirroring.childrenInherit: true

    PlasmaCore.DataSource {
        id: keystateSource
        engine: "keystate"
        connectedSources: "Caps Lock"
    }

    MosAuthPlugin.AuthService {
        id: mosAuthService
    }

    PlasmaNM.NetworkStatus {
        id: networkStatus
    }

    PlasmaNM.AvailableDevices {
        id: availableDevices
    }

    Item {
        id: wallpaper
        anchors.fill: parent
        Repeater {
            model: screenModel

            Background {
                x: geometry.x; y: geometry.y; width: geometry.width; height: geometry.height
                sceneBackgroundType: config.type
                sceneBackgroundColor: config.color
                sceneBackgroundImage: config.background
            }
        }
    }

    RejectPasswordAnimation {
        id: rejectPasswordAnimation
        target: mainStack
    }

    MouseArea {
        id: loginScreenRoot
        anchors.fill: parent

        property bool uiVisible: true
        property bool blockUI: mainStack.depth > 1 || userListComponent.mainPasswordBox.text.length > 0 || inputPanel.keyboardActive || config.type !== "image"

        hoverEnabled: true
        drag.filterChildren: true
        onPressed: uiVisible = true;
        onPositionChanged: uiVisible = true;
        onUiVisibleChanged: {
            if (blockUI) {
                fadeoutTimer.running = false;
            } else if (uiVisible) {
                fadeoutTimer.restart();
            }
        }
        onBlockUIChanged: {
            if (blockUI) {
                fadeoutTimer.running = false;
                uiVisible = true;
            } else {
                fadeoutTimer.restart();
            }
        }

        Keys.onPressed: {
            uiVisible = true;
            event.accepted = false;
        }

        //takes one full minute for the ui to disappear
        Timer {
            id: fadeoutTimer
            running: true
            interval: 60000
            onTriggered: {
                if (!loginScreenRoot.blockUI) {
                    loginScreenRoot.uiVisible = false;
                }
            }
        }
        WallpaperFader {
            visible: config.type === "image"
            anchors.fill: parent
            state: loginScreenRoot.uiVisible ? "on" : "off"
            source: wallpaper
            mainStack: mainStack
            footer: footer
            clock: clock
        }

        DropShadow {
            id: clockShadow
            anchors.fill: clock
            source: clock
            visible: !softwareRendering
            radius: 6
            samples: 14
            spread: 0.3
            color : "black" // shadows should always be black
            Behavior on opacity {
                OpacityAnimator {
                    duration: PlasmaCore.Units.veryLongDuration * 2
                    easing.type: Easing.InOutQuad
                }
            }
        }

        Clock {
            id: clock
            property Item shadow: clockShadow
            property int coords: ((userListComponent.userList.y > 0
                                   ? userListComponent.userList.y
                                   : userListComponent.prompts.y) + mainStack.y) * 0.5
            visible: y > 0
            anchors.horizontalCenter: parent.horizontalCenter
            y: loginScreenRoot.uiVisible ? (coords - height * 0.5) : coords
            Layout.alignment: Qt.AlignBaseline
        }

        QQC2.ScrollView {
            visible: false;

            width: 400;
            height: parent.height;

            ListView {
                id: debugInfo;
                anchors.fill: parent

                function log(text)
                {
                    text = String(text)
                    debugInfo.model.append({value: text});
                }

                model: ListModel {

                }

                QQC2.ScrollBar.vertical: QQC2.ScrollBar { visible: true }

                delegate: Text {
                    wrapMode: Text.Wrap;
                    width: parent.width
                    leftPadding: 20;
                    rightPadding: 20;
                    text: value + "\n"
                }
            }

            Rectangle
            {
                anchors.fill: parent
                color: "#ffccaa"
            }

            MouseArea {
                anchors.fill: parent;
                propagateComposedEvents: true;
                onClicked: debugInfo.model.clear()
            }
        }

        QQC2.StackView {
            id: mainStack

            anchors {
                left: parent.left
                right: parent.right
            }
            height: root.height + PlasmaCore.Units.gridUnit * 3

            // If true (depends on the style and environment variables), hover events are always accepted
            // and propagation stopped. This means the parent MouseArea won't get them and the UI won't be shown.
            // Disable capturing those events while the UI is hidden to avoid that, while still passing events otherwise.
            // One issue is that while the UI is visible, mouse activity won't keep resetting the timer, but when it
            // finally expires, the next event should immediately set uiVisible = true again.
            hoverEnabled: loginScreenRoot.uiVisible ? undefined : false

            focus: true //StackView is an implicit focus scope, so we need to give this focus so the item inside will have it

            Timer {
                //SDDM has a bug in 0.13 where even though we set the focus on the right item within the window, the window doesn't have focus
                //it is fixed in 6d5b36b28907b16280ff78995fef764bb0c573db which will be 0.14
                //we need to call "window->activate()" *After* it's been shown. We can't control that in QML so we use a shoddy timer
                //it's been this way for all Plasma 5.x without a huge problem
                running: true
                repeat: false
                interval: 200
                onTriggered: mainStack.forceActiveFocus()
            }

            initialItem: Login {
                id: userListComponent
                userListModel: userModel
                loginScreenUiVisible: loginScreenRoot.uiVisible
                userListCurrentIndex: userModel.lastIndex >= 0 ? userModel.lastIndex : 0
                lastUserName: userModel.lastUser
                userListDisplayType: {
                    if (!userListModel.hasOwnProperty("count")
                            || !userListModel.hasOwnProperty("disableAvatarsThreshold")) {
                        return SessionManagementScreen.UserListDisplayType.None
                    }

                    if (userListModel.count === 0) {
                        return SessionManagementScreen.UserListDisplayType.None
                    }

                    if (userListModel.hasOwnProperty("containsAllUsers") && !userListModel.containsAllUsers) {
                        return SessionManagementScreen.UserListDisplayType.None
                    }

                    if (config.userListDisplayType === "none") {
                        return SessionManagementScreen.UserListDisplayType.None
                    }

                    if (config.userListDisplayType === "listview") {
                        return SessionManagementScreen.UserListDisplayType.ListView
                    }

                    if (userListModel.count > userListModel.disableAvatarsThreshold
                            || config.userListDisplayType === "select") {
                        return SessionManagementScreen.UserListDisplayType.Select
                    }

                    return SessionManagementScreen.UserListDisplayType.ListView
                }

                showMosButtons: true
                authService: mosAuthService
                wirelessStatus: root.wirelessStatus
                wirelessDeviceAvailable: availableDevices.wirelessDeviceAvailable

                notificationMessage: {
                    const parts = [];
                    if (keystateSource.data["Caps Lock"]["Locked"]) {
                        parts.push(i18nd("plasma_lookandfeel_org.kde.lookandfeel", "Caps Lock is on"));
                    }
                    if (root.notificationMessage) {
                        parts.push(root.notificationMessage);
                    }

                    return parts.join(" • ");
                }

                onDisplayNotificationMessage: {
                    root.notificationMessage = msg;
                }

                actionItemsVisible: !inputPanel.keyboardActive
                actionItems: [
                    ActionButton {
                        iconSource: "system-suspend"
                        text: i18ndc("plasma_lookandfeel_org.kde.lookandfeel", "Suspend to RAM", "Sleep")
                        fontSize: parseInt(config.fontSize) + 1
                        onClicked: sddm.suspend()
                        enabled: sddm.canSuspend
                    },
                    ActionButton {
                        iconSource: "system-reboot"
                        text: i18nd("plasma_lookandfeel_org.kde.lookandfeel", "Restart")
                        fontSize: parseInt(config.fontSize) + 1
                        onClicked: sddm.reboot()
                        enabled: sddm.canReboot
                    },
                    ActionButton {
                        iconSource: "system-shutdown"
                        text: i18nd("plasma_lookandfeel_org.kde.lookandfeel", "Shut Down")
                        fontSize: parseInt(config.fontSize) + 1
                        onClicked: sddm.powerOff()
                        enabled: sddm.canPowerOff
                    },
                    ActionButton {
                        iconSource: "system-user-prompt"
                        text: i18ndc("plasma_lookandfeel_org.kde.lookandfeel", "For switching to a username and password prompt", "Other…")
                        fontSize: parseInt(config.fontSize) + 1
                        onClicked: mainStack.push(userPromptComponent)
                        enabled: true
                        visible: !userListComponent.showUsernamePrompt
                    }]

                onLoginRequest: {
                    root.notificationMessage = ""
                    sddm.login(username, password, sessionButton.currentIndex)
                }
            }

            Behavior on opacity {
                OpacityAnimator {
                    duration: PlasmaCore.Units.longDuration
                }
            }

            readonly property real zoomFactor: 3

            popEnter: Transition {
                ScaleAnimator {
                    from: mainStack.zoomFactor
                    to: 1
                    duration: PlasmaCore.Units.longDuration * (mainStack.zoomFactor / 2)
                    easing.type: Easing.OutCubic
                }
                OpacityAnimator {
                    from: 0
                    to: 1
                    duration: PlasmaCore.Units.longDuration * (mainStack.zoomFactor / 2)
                    easing.type: Easing.OutCubic
                }
            }

            popExit: Transition {
                ScaleAnimator {
                    from: 1
                    to: 0
                    duration: PlasmaCore.Units.longDuration * (mainStack.zoomFactor / 2)
                    easing.type: Easing.OutCubic
                }
                OpacityAnimator {
                    from: 1
                    to: 0
                    duration: PlasmaCore.Units.longDuration * (mainStack.zoomFactor / 2)
                    easing.type: Easing.OutCubic
                }
            }

            pushEnter: Transition {
                ScaleAnimator {
                    from: 0
                    to: 1
                    duration: PlasmaCore.Units.longDuration * (mainStack.zoomFactor / 2)
                    easing.type: Easing.OutCubic
                }
                OpacityAnimator {
                    from: 0
                    to: 1
                    duration: PlasmaCore.Units.longDuration * (mainStack.zoomFactor / 2)
                    easing.type: Easing.OutCubic
                }
            }

            pushExit: Transition {
                ScaleAnimator {
                    from: 1
                    to: mainStack.zoomFactor
                    duration: PlasmaCore.Units.longDuration * (mainStack.zoomFactor / 2)
                    easing.type: Easing.OutCubic
                }
                OpacityAnimator {
                    from: 1
                    to: 0
                    duration: PlasmaCore.Units.longDuration * (mainStack.zoomFactor / 2)
                    easing.type: Easing.OutCubic
                }
            }
        }

        Loader {
            id: inputPanel
            state: "hidden"
            property bool keyboardActive: item ? item.active : false
            onKeyboardActiveChanged: {
                if (keyboardActive) {
                    state = "visible"
                    // Otherwise the password field loses focus and virtual keyboard
                    // keystrokes get eaten
                    userListComponent.mainPasswordBox.forceActiveFocus();
                } else {
                    state = "hidden";
                }
            }

            source: Qt.platform.pluginName.includes("wayland") ? "components/VirtualKeyboard_wayland.qml" : "components/VirtualKeyboard.qml"

            width: (root.width > 1280) ? parent.width * 0.5 : parent.width * 0.75

            anchors {
                horizontalCenter: parent.horizontalCenter
                bottom: parent.bottom
            }

            function showHide() {
                state = state === "hidden" ? "visible" : "hidden";
            }

            states: [
                State {
                    name: "visible"
                    PropertyChanges {
                        target: mainStack
                        y: Math.min(0, root.height - inputPanel.height - userListComponent.visibleBoundary)
                    }
                    PropertyChanges {
                        target: inputPanel
                        y: root.height - inputPanel.height
                        opacity: 1
                    }
                },
                State {
                    name: "hidden"
                    PropertyChanges {
                        target: mainStack
                        y: 0
                    }
                    PropertyChanges {
                        target: inputPanel
                        y: root.height - root.height/4
                        opacity: 0
                    }
                }
            ]
            transitions: [
                Transition {
                    from: "hidden"
                    to: "visible"
                    SequentialAnimation {
                        ScriptAction {
                            script: {
                                inputPanel.item.activated = true;
                                Qt.inputMethod.show();
                            }
                        }
                        ParallelAnimation {
                            NumberAnimation {
                                target: mainStack
                                property: "y"
                                duration: PlasmaCore.Units.longDuration
                                easing.type: Easing.InOutQuad
                            }
                            NumberAnimation {
                                target: inputPanel
                                property: "y"
                                duration: PlasmaCore.Units.longDuration
                                easing.type: Easing.OutQuad
                            }
                            OpacityAnimator {
                                target: inputPanel
                                duration: PlasmaCore.Units.longDuration
                                easing.type: Easing.OutQuad
                            }
                        }
                    }
                },
                Transition {
                    from: "visible"
                    to: "hidden"
                    SequentialAnimation {
                        ParallelAnimation {
                            NumberAnimation {
                                target: mainStack
                                property: "y"
                                duration: PlasmaCore.Units.longDuration
                                easing.type: Easing.InOutQuad
                            }
                            NumberAnimation {
                                target: inputPanel
                                property: "y"
                                duration: PlasmaCore.Units.longDuration
                                easing.type: Easing.InQuad
                            }
                            OpacityAnimator {
                                target: inputPanel
                                duration: PlasmaCore.Units.longDuration
                                easing.type: Easing.InQuad
                            }
                        }
                        ScriptAction {
                            script: {
                                inputPanel.item.activated = false;
                                Qt.inputMethod.hide();
                            }
                        }
                    }
                }
            ]
        }

        Component {
            id: userPromptComponent
            Login {
                showUsernamePrompt: true
                notificationMessage: {
                    const parts = [];
                    if (keystateSource.data["Caps Lock"]["Locked"]) {
                        parts.push(i18nd("plasma_lookandfeel_org.kde.lookandfeel", "Caps Lock is on"));
                    }
                    if (root.notificationMessage) {
                        parts.push(root.notificationMessage);
                    }

                    return parts.join(" • ");
                }
                loginScreenUiVisible: loginScreenRoot.uiVisible
                fontSize: parseInt(config.fontSize) + 2

                // using a model rather than a QObject list to avoid QTBUG-75900
                userListModel: ListModel {
                    ListElement {
                        name: ""
                        iconSource: ""
                    }
                    Component.onCompleted: {
                        // as we can't bind inside ListElement
                        setProperty(0, "name", i18nd("plasma_lookandfeel_org.kde.lookandfeel", "Type in Username and Password"));
                    }
                }

                authService: mosAuthService
                wirelessStatus: root.wirelessStatus
                wirelessDeviceAvailable: availableDevices.wirelessDeviceAvailable

                onLoginRequest: {
                    root.notificationMessage = ""
                    sddm.login(username, password, sessionButton.currentIndex)
                }

                actionItemsVisible: !inputPanel.keyboardActive
                actionItems: [
                    ActionButton {
                        iconSource: "system-suspend"
                        text: i18ndc("plasma_lookandfeel_org.kde.lookandfeel", "Suspend to RAM", "Sleep")
                        fontSize: parseInt(config.fontSize) + 1
                        onClicked: sddm.suspend()
                        enabled: sddm.canSuspend
                    },
                    ActionButton {
                        iconSource: "system-reboot"
                        text: i18nd("plasma_lookandfeel_org.kde.lookandfeel", "Restart")
                        fontSize: parseInt(config.fontSize) + 1
                        onClicked: sddm.reboot()
                        enabled: sddm.canReboot
                    },
                    ActionButton {
                        iconSource: "system-shutdown"
                        text: i18nd("plasma_lookandfeel_org.kde.lookandfeel", "Shut Down")
                        fontSize: parseInt(config.fontSize) + 1
                        onClicked: sddm.powerOff()
                        enabled: sddm.canPowerOff
                    },
                    ActionButton {
                        iconSource: "system-user-list"
                        text: i18nd("plasma_lookandfeel_org.kde.lookandfeel", "List Users")
                        fontSize: parseInt(config.fontSize) + 1
                        onClicked: mainStack.pop()
                    }
                ]
            }
        }

        Component {
            id: mosAuthComponent
            MosAuth {
                authService: mosAuthService

                onLoginRequest: {
                    sddm.login(username, password, sessionButton.currentIndex)
                }

                onDisplayRegisterNotification: {
                    rejectPasswordAnimation.start()
                }
            }
        }

        Component {
            id: wirelessNmComponent

            WirelessNetworkManager {
                wirelessStatus: root.wirelessStatus
                wirelessStatusMsg: root.wirelessStatusMsg
                authService: mosAuthService
            }
        }

        DropShadow {
            id: logoShadow
            anchors.fill: logo
            source: logo
            visible: !softwareRendering && config.showlogo === "shown"
            horizontalOffset: 1
            verticalOffset: 1
            radius: 6
            samples: 14
            spread: 0.3
            color : "black" // shadows should always be black
            opacity: loginScreenRoot.uiVisible ? 0 : 1
            Behavior on opacity {
                //OpacityAnimator when starting from 0 is buggy (it shows one frame with opacity 1)"
                NumberAnimation {
                    duration: PlasmaCore.Units.longDuration
                    easing.type: Easing.InOutQuad
                }
            }
        }

        Image {
            id: logo
            visible: config.showlogo === "shown"
            source: config.logo
            anchors.horizontalCenter: parent.horizontalCenter
            anchors.bottom: footer.top
            anchors.bottomMargin: PlasmaCore.Units.largeSpacing
            asynchronous: true
            sourceSize.height: height
            opacity: loginScreenRoot.uiVisible ? 0 : 1
            fillMode: Image.PreserveAspectFit
            height: Math.round(PlasmaCore.Units.gridUnit * 3.5)
            Behavior on opacity {
                // OpacityAnimator when starting from 0 is buggy (it shows one frame with opacity 1)"
                NumberAnimation {
                    duration: PlasmaCore.Units.longDuration
                    easing.type: Easing.InOutQuad
                }
            }
        }

        //Footer
        RowLayout {
            id: footer
            anchors {
                bottom: parent.bottom
                left: parent.left
                right: parent.right
                margins: PlasmaCore.Units.smallSpacing
            }

            Behavior on opacity {
                OpacityAnimator {
                    duration: PlasmaCore.Units.longDuration
                }
            }

            PlasmaComponents3.ToolButton {
                text: i18ndc("plasma_lookandfeel_org.kde.lookandfeel", "Button to show/hide virtual keyboard", "Virtual Keyboard")
                font.pointSize: config.fontSize
                icon.name: inputPanel.keyboardActive ? "input-keyboard-virtual-on" : "input-keyboard-virtual-off"
                onClicked: {
                    // Otherwise the password field loses focus and virtual keyboard
                    // keystrokes get eaten
                    userListComponent.mainPasswordBox.forceActiveFocus();
                    inputPanel.showHide()
                }
                visible: inputPanel.status === Loader.Ready
            }

            KeyboardButton {
                font.pointSize: config.fontSize

                onKeyboardLayoutChanged: {
                    // Otherwise the password field loses focus and virtual keyboard
                    // keystrokes get eaten
                    userListComponent.mainPasswordBox.forceActiveFocus();
                }
            }

            SessionButton {
                id: sessionButton
                font.pointSize: config.fontSize

                onSessionChanged: {
                    // Otherwise the password field loses focus and virtual keyboard
                    // keystrokes get eaten
                    userListComponent.mainPasswordBox.forceActiveFocus();
                }
            }

            Item {
                Layout.fillWidth: true
            }

            PlasmaComponents3.ToolButton {
                visible: availableDevices.wirelessDeviceAvailable
                text: wirelessStatusMsg
                font.pointSize: config.fontSize
                icon.name: wirelessStatus === PlasmaNM.Enums.Activated ? "network-wireless" : "network-wireless-disconnected"
                Layout.rightMargin: PlasmaCore.Units.gridUnit * 0.5

                onClicked: {
                    if (mainStack.currentItem.objectName !== "WirelessNetworkManager")
                    {
                        mainStack.push(wirelessNmComponent);
                    }
                    else
                    {
                        mainStack.pop();
                    }
                }
            }

            RowLayout {
                visible: wiredStatus === PlasmaNM.Enums.Activated

                Layout.rightMargin: PlasmaCore.Units.gridUnit * 0.5

                PlasmaCore.IconItem {
                    source: "network-wired"
                    colorGroup: PlasmaCore.ColorScope.colorGroup
                }

                PlasmaComponents3.Label {
                    text: i18_networkConnectedMsg
                    font.pointSize: config.fontSize
                }
            }

            Battery {
                fontSize: config.fontSize
                Layout.rightMargin: PlasmaCore.Units.gridUnit * 0.5
            }
        }
    }

    Connections {
        target: sddm
        function onLoginFailed() {
            notificationMessage = userListComponent.mosAuthBtnEnabled ?
                "Забыли пароль? Вы можете восстановить его, авторизовавшись через mos.ru"
                : "Неправильное имя пользователя или пароль."

            ++loginAttempts
            /*
             * HACK: почему-то эта функция вызывается два раза на одну неудачную попытку входа
             * поэтому умножаем тут на 2.
             */
            if (loginAttempts >= 3 * 2 && !userListComponent.mosAuthBtnEnabled)
            {
                notificationMessage = "Вы предприняли несколько неудачных попыток входа. Возможно, учетная запись была временно заблокирована. Попробуйте позже или обратитесь к администратору."
            }

            footer.enabled = true
            mainStack.enabled = true
            userListComponent.userList.opacity = 1
            rejectPasswordAnimation.start()
        }
        function onLoginSucceeded() {
            //note SDDM will kill the greeter at some random point after this
            //there is no certainty any transition will finish, it depends on the time it
            //takes to complete the init
            mainStack.opacity = 0
            footer.opacity = 0
        }
    }

    onNotificationMessageChanged: {
        if (notificationMessage) {
            notificationResetTimer.start();
        }
    }

    Timer {
        id: notificationResetTimer
        interval: 10000
        onTriggered: notificationMessage = ""
    }
}
