import "components"

import QtQuick 2.15
import QtQuick.Layouts 1.15
import QtQuick.Controls 2.15 as QQC2

import org.kde.plasma.core 2.0 as PlasmaCore
import org.kde.plasma.components 3.0 as PlasmaComponents3
import org.kde.plasma.extras 2.0 as PlasmaExtras
import org.kde.kirigami 2.13 as Kirigami

import org.kde.plasma.networkmanagement 0.2 as PlasmaNM
import org.mos.auth 0.1 as MosAuthPlugin


SessionManagementScreen {
    id: root

    // From parent
    required property MosAuthPlugin.AuthService authService
    required property int wirelessStatus
    required property bool wirelessDeviceAvailable

    property Item mainPasswordBox: passwordBox

    property bool showUsernamePrompt: (userListDisplayType === SessionManagementScreen.UserListDisplayType.None)

    property bool showMosButtons: false

    property string lastUserName
    property bool loginScreenUiVisible: false

    //the y position that should be ensured visible when the on screen keyboard is visible
    property int visibleBoundary: mapFromItem(loginButton, 0, 0).y
    onHeightChanged: visibleBoundary = mapFromItem(loginButton, 0, 0).y + loginButton.height + PlasmaCore.Units.smallSpacing

    property int fontSize: parseInt(config.fontSize)

    signal loginRequest(string username, string password)

    signal displayNotificationMessage(string msg)

    signal waitGuestPreparation()
    signal stopWaitingGuestPreparation()

    onWaitGuestPreparation: {
        authControls.visible = false;
        guestBusyIndicatorLayout.visible = true;
        footer.enabled = false;
        mainStack.enabled = false;

        loginAsGuestButton.forceActiveFocus();
    }

    onStopWaitingGuestPreparation: {
        authControls.visible = true;
        guestBusyIndicatorLayout.visible = false;
        footer.enabled = true;
        mainStack.enabled = true;
    }

    onShowUsernamePromptChanged: {
        if (!showUsernamePrompt) {
            lastUserName = ""
        }
    }

    onUserSelected: {
        // Don't startLogin() here, because the signal is connected to the
        // Escape key as well, for which it wouldn't make sense to trigger
        // login.
        focusFirstVisibleFormControl();
    }

    QQC2.StackView.onActivating: {
        // Controls are not visible yet.
        Qt.callLater(focusFirstVisibleFormControl);
    }

    function focusFirstVisibleFormControl() {
        const nextControl = (userNameInput.visible
                             ? userNameInput
                             : (passwordBox.visible
                                ? passwordBox
                                : loginButton));
        // Using TabFocusReason, so that the loginButton gets the visual highlight.
        nextControl.forceActiveFocus(Qt.TabFocusReason);
    }

    /*
     * Login has been requested with the following username and password
     * If username field is visible, it will be taken from that, otherwise from the "name" property of the currentIndex
     */
    function startLogin() {
        const username = showUsernamePrompt ? userNameInput.text : userList.selectedUser
        const password = passwordBox.text

        footer.enabled = false
        mainStack.enabled = false
        userListComponent.userList.opacity = 0.5

        // This is partly because it looks nicer, but more importantly it
        // works round a Qt bug that can trigger if the app is closed with a
        // TextField focused.
        //
        // See https://bugreports.qt.io/browse/QTBUG-55460
        loginButton.forceActiveFocus();

        // Autoconnect to enterprise Wi-Fi:
        // If SSID and domain are set in config and now Wi-Fi is deactivated.
        if (wirelessDeviceAvailable
            && authService.wifiEapSsid !== ""
            && authService.wifiEapDomainIsSet
            && wirelessStatus === PlasmaNM.Enums.Deactivated)
        {
            // Do async login request,
            // after the connection is completed (successfully or not).
            const oneshot = () => {
                if (root.wirelessStatus === PlasmaNM.Enums.Activated
                    || root.wirelessStatus === PlasmaNM.Enums.Deactivated)
                {
                    root.onWirelessStatusChanged.disconnect(oneshot);
                    loginRequest(username, password);
                }
            };

            // Create enterprise connection and try to connect to it.
            authService.handleEapWifi(authService.wifiEapSsid, username, password);
            // Asynchronously handle changes of the connection status.
            root.onWirelessStatusChanged.connect(oneshot);
        }
        else {
            loginRequest(username, password);
        }
    }

    Item {
        Connections {
            target: authService
            function onPrepareGuestFinished(result: bool)
            {
                if (result)
                {
                    const username = "guest";
                    const password = "";

                    root.loginRequest(username, password);
                }
                else
                {
                    root.stopWaitingGuestPreparation();
                    displayNotificationMessage("Непредвиденная ошибка\nпри попытке входа в гостя!")
                }
            }

            function onLogin(user, pass)
            {
                root.loginRequest(user, pass);
            }
        }
    }

    // Очищаем профиль гостя, перед логином в него.
    function prepareGuestAccount()
    {
        debugInfo.log("prepareGuestAccount()");
        root.waitGuestPreparation();
        authService.prepareGuest();
    }

    function openMosAuthComponent()
    {
        mainStack.push(mosAuthComponent);
    }

    ColumnLayout
    {
        id: guestBusyIndicatorLayout
        visible: false

        PlasmaComponents3.Label {
            visible: true
            Layout.fillWidth: true
            font.italic: true
            font.pointSize: fontSize + 1
            horizontalAlignment: Text.AlignHCenter
            wrapMode: Text.WordWrap
            text: `Идет подготовка профиля гостя.\nПожалуйста, подождите...`
            Layout.bottomMargin: PlasmaCore.Units.gridUnit * 0.5
        }

        PlasmaComponents3.BusyIndicator
        {
            Layout.fillWidth: true
        }

        Layout.bottomMargin: PlasmaCore.Units.gridUnit
    }

    ColumnLayout {
        id: authControls

        PlasmaComponents3.TextField {
            id: userNameInput
            font.pointSize: fontSize + 1
            Layout.fillWidth: true

            text: lastUserName
            visible: showUsernamePrompt
            focus: showUsernamePrompt && !lastUserName //if there's a username prompt it gets focus first, otherwise password does
            placeholderText: i18nd("plasma_lookandfeel_org.kde.lookandfeel", "Username")

            validator: RegExpValidator {
                regExp: authService.loginLowercaseOnly ? /[a-z0-9_.@$-]+/ : /[a-zA-Z0-9_.@$\\-]+/
            }
            inputMethodHints: Qt.ImhPreferLowercase | Qt.ImhNoAutoUppercase | Qt.ImhNoPredictiveText

            onAccepted: {
                if (root.loginScreenUiVisible) {
                    passwordBox.forceActiveFocus()
                }
            }
        }

        ColumnLayout {
            visible: showUsernamePrompt || userList.selectedUser !== "guest"
            PlasmaExtras.PasswordField {
                id: passwordBox
                font.pointSize: fontSize + 1
                Layout.fillWidth: true

                placeholderText: i18nd("plasma_lookandfeel_org.kde.lookandfeel", "Password")
                focus: !showUsernamePrompt || lastUserName

                // Включил обратно, так как кажется все в порядке

                // Disable reveal password action because SDDM does not have the breeze icon set loaded
                // rightActions: []

                onAccepted: {
                    if (root.loginScreenUiVisible) {
                        startLogin();
                    }
                }

                visible: root.showUsernamePrompt || userList.currentUserNeedsPassword

                Keys.onEscapePressed: {
                    mainStack.currentItem.forceActiveFocus();
                }

                //if empty and left or right is pressed change selection in user switch
                //this cannot be in keys.onLeftPressed as then it doesn't reach the password box
                Keys.onPressed: {
                    if (event.key === Qt.Key_Left && !text) {
                        userList.decrementCurrentIndex();
                        event.accepted = true
                    }
                    if (event.key === Qt.Key_Right && !text) {
                        userList.incrementCurrentIndex();
                        event.accepted = true
                    }
                }

                Connections {
                    target: sddm
                    function onLoginFailed() {
                        passwordBox.selectAll()
                        passwordBox.forceActiveFocus()
                    }
                }
            }

            PlasmaComponents3.Button {
                id: loginButton
                Accessible.name: i18nd("plasma_lookandfeel_org.kde.lookandfeel", "Log In")
                Layout.preferredHeight: passwordBox.implicitHeight
                Layout.fillWidth: true

                contentItem: Text {
                    text: parent.text
                    font.pointSize: fontSize + 1
                    color: "#ffffff"
                    horizontalAlignment: Text.AlignHCenter
                    verticalAlignment: Text.AlignVCenter
                    elide: Text.ElideRight
                }

                background: Rectangle {
                    Layout.fillWidth: true
                    Layout.fillHeight: true
                    color: parent.hovered ? "#0074ff" : "#1d99f3"
                    radius: 6
                }

                text: i18nd("plasma_lookandfeel_org.kde.lookandfeel", "Log In")

                onClicked: startLogin()
                Keys.onEnterPressed: clicked()
                Keys.onReturnPressed: clicked()
            }
        }

        ColumnLayout {
            id: mosButtonsLayout

            visible: showMosButtons

            Layout.preferredWidth: passwordBox.width
            Layout.fillWidth: true
            Layout.fillHeight: true
            Layout.topMargin: PlasmaCore.Units.gridUnit

            PlasmaComponents3.Button {
                id: mosAuthButton
                visible: authService.mosAuthBtnEnabled
                Layout.preferredHeight: passwordBox.implicitHeight
                Layout.fillWidth: true

                contentItem: Text {
                    text: parent.text
                    font.pointSize: fontSize + 1
                    color: "#1d99f3"
                    horizontalAlignment: Text.AlignHCenter
                    verticalAlignment: Text.AlignVCenter
                    elide: Text.ElideRight
                }

                background: Rectangle {
                    Layout.fillWidth: true
                    Layout.fillHeight: true
                    radius: 6
                    color: parent.hovered ? Kirigami.ColorUtils.linearInterpolation(PlasmaCore.Theme.backgroundColor,
                                                                                    "#1d99f3", 0.15)
                                          : PlasmaCore.Theme.backgroundColor
                    border.color: "#1d99f3"
                    border.width: 1
                }

                text: "Авторизация через mos.ru"

                onClicked: openMosAuthComponent()
                Keys.onEnterPressed: clicked()
                Keys.onReturnPressed: clicked()
            }

            PlasmaComponents3.Button {
                id: loginAsGuestButton
                visible: authService.guestEnabled
                Layout.preferredHeight: passwordBox.implicitHeight
                Layout.fillWidth: true

                background: Rectangle {
                    Layout.fillWidth: true
                    Layout.fillHeight: true
                    radius: 6
                    color: parent.hovered ? Kirigami.ColorUtils.linearInterpolation(PlasmaCore.Theme.backgroundColor,
                                                                                    PlasmaCore.Theme.textColor, 0.1)
                                          : PlasmaCore.Theme.backgroundColor
                    border.color: parent.hovered ? "#1d99f3" : "#e0e0e0"
                    border.width: 1
                }

                text: "Войти как гость"
                font.pointSize: fontSize + 1

                onClicked: prepareGuestAccount()
                Keys.onEnterPressed: clicked()
                Keys.onReturnPressed: clicked()
            }
        }
    }
}
