#!/bin/bash

set -e
set -u

trap 'echo exit' EXIT

# Удалить / отмонтировать home гостя, если папка существует.
_remove_home_if_exist()
{
    { test -d "${1}" && { umount -f "${1}" || umount -fl "${1}" || rm -rf "${1}"; } ; } || true
}

# Для считывания значений из конфига.
_parse_config_true_val()
{
    [[ "$1" == "true" || "$1" == "yes" || "$1" == "1" ]]
}

_parse_config_empty_val()
{
    [[ "$1" == "" ]]
}

# Основной код.
if [[ $PAM_USER == "guest"
    && $PAM_SERVICE != "systemd-user"
    && ($PAM_TYPE == "open_session" || $PAM_TYPE == "close_session") ]]
then
    HOME="$( getent passwd "$PAM_USER" | cut -d: -f6 )"

    # Если сессия завершается.
    if [[ $PAM_TYPE == "close_session" ]]
    then
        if { loginctl show-user guest ; }
        then
            loginctl kill-user --signal KILL guest
        fi
        exit
    fi

    # Если уже есть активная сессия, то ничего не делаем.
    if [[ $(loginctl show-user -p State guest) == "State=active"
            || $(loginctl show-user -p State guest) == "State=online" ]]
    then
        exit
    fi

    # Иначе state of session = closing.
    if { loginctl show-user guest ; }
    then
        loginctl kill-user --signal KILL guest
    fi

    _remove_home_if_exist "$HOME"

    # Создаем папку home.
    mkdir --parents "${HOME}"

    # Если в конфиге включена опция 'guest-home-tmpfs', то монтируем как tmpfs.
    declare -l guest_home_tmpfs
    # py-ini-config возвращает != 0 и пустую строку, если значение не найдено
    guest_home_tmpfs="$(py-ini-config get sddm guest-home-tmpfs -a /usr/lib/mos-auth/mos-auth.conf -a /etc/mos-auth/mos-auth.conf -e conf || true)"
    if { _parse_config_true_val "$guest_home_tmpfs" ; }
    then
        mount --types tmpfs --options mode=700 none "${HOME}"
    fi

    # Если в конфиге переопределена директория skel для гостя, то считываем это значение.
    # Иначе, считываем стандартное значение из `/etc/default/useradd`.
    # Если оно отсутствует или равно "", тогда используем `/etc/skel`.
    # Также проверяем, чтобы указанные директории существовали.
    declare -l guest_skel_directory
    guest_skel_directory="$(py-ini-config get sddm guest-skel-directory -a /usr/lib/mos-auth/mos-auth.conf -a /etc/mos-auth/mos-auth.conf -e conf || true)"

    if { _parse_config_empty_val "$guest_skel_directory" || ! test -d "$guest_skel_directory" ; }
    then
        guest_skel_directory="$(py-ini-config get /etc/default/useradd --no-section SKEL || true)"

        if { _parse_config_empty_val "$guest_skel_directory" || ! test -d "$guest_skel_directory" ; }
        then
            guest_skel_directory="/etc/skel"
        fi
    fi

    # Копируем содержимое папки skel в домашний профиль гостя.
    cp --reflink=auto --recursive --no-target-directory "${guest_skel_directory}" "${HOME}"

    # Если в конфиге включена опция 'guest-dbus-service', то ставим в автозагрузку DBus службу.
    declare -l guest_dbus_service
    guest_dbus_service="$(py-ini-config get sddm guest-dbus-service -a /usr/lib/mos-auth/mos-auth.conf -a /etc/mos-auth/mos-auth.conf -e conf || true)"
    if { _parse_config_true_val "$guest_dbus_service" || _parse_config_empty_val "$guest_dbus_service" ; }
    then
        mkdir --parents "${HOME}/.config/autostart"
        cat << EOF > "${HOME}/.config/autostart/mos-guest-dbus-service.desktop"
[Desktop Entry]
Name=MOS Guest DBus Service
Exec=mos-guest-dbus-service
Type=Application
EOF
    fi

    chown --recursive "$PAM_USER:" "${HOME}"
fi
