#!/bin/bash
set -e
set -u
set -f

TMPDIR="$(mktemp -d)"
export TMPDIR
trap 'rm -fr "$TMPDIR"' EXIT

if [ "$UID" -ne 0 ] && command -v sudo >/dev/null 2>&1
    then SUDO="sudo "
    else SUDO=""
fi

_help(){
    cat << EOF
Команда mos-auth-config включает и отключает вход в систему через учетную запись на mos.ru.
Требуется ее запуск от root.

Для включения входа через mos.ru выполните:
  ${SUDO}mos-auth-config enable
Для отключения входа через mos.ru выполните:
  ${SUDO}mos-auth-config disable
После чего перезагрузите компьютер (или перезапустите DM).

При включении (enable) выполняются следующие действия:
* устанавливается тема mos-auth в SDDM,
* устанавливается PAM-механизм автоочистки профиля гостя,
* включается поддержка сетевых папок (если установлен пакет 'mos-auth-folders').

При выключении (disable) выполняются следующие действия:
* возвращается тема SDDM, которая была настроена до включения,
* выключается PAM-механизм автоочистки профиля гостя,
* выключается поддержка сетевых папок.

Вы можете настраивать поведение авторизации через mos.ru путем изменения конфигурационного файла.
Для редактирования файла с конфигурацией (настройками) выполните:
  ${SUDO}mos-auth-config edit

Список команд:
  enable		Включение авторизации через mos.ru
  disable		Выключение авторизации через mos.ru
  status		Вывод текущего статуса авторизации через mos.ru в системе
  edit			Редактирование конфигурационного файла
  help			Вызов текущей справки
EOF
}

_check_root(){
    if [ "$UID" -ne 0 ]; then
        echo "Ошибка: необходимы root-права!"
        echo ""
        _help
        return 1
    fi
}

# $1: путь к конфигу SDDM для редактирования
_sddm_config(){
    local file
    file="$1"
    local tmp
    tmp="$(mktemp)"
    cat "$file" > "$tmp"
    py-ini-config set "$tmp" Theme Current mos-auth
    py-ini-config set "$tmp" Users MaximumUid 2000000000
    py-ini-config set "$tmp" Users MinimumUid 10000000
    grep -q 'Current=mos-auth' "$tmp"
    grep -q 'MaximumUid=2000000000' "$tmp"
    grep -q 'MinimumUid=10000000' "$tmp"
    cat "$tmp" > "$file"
    rm -f "$tmp"
}

_enable(){
    local fsddm
    local fold
    fsddm="/etc/sddm.conf"
    fold="/var/lib/mos-auth/mos-auth-old-settings"
    if test -f /var/lib/mos-auth/enabled
    then
        echo "Уже настроено. Выполните disable и снова enable"
        return 1
    fi
    mkdir -p /var/lib/mos-auth
    if test -f "$fsddm"
    then
        cp -f "$fsddm" /var/lib/mos-auth/sddm.conf.mos-auth-bak
        touch "$fold"
        if grep -q 'Current=' "$fsddm"
        then
            py-ini-config set "$fold" Theme Current "$(py-ini-config get $fsddm Theme Current)"
        fi
        if grep -q 'MinimumUid=' "$fsddm"
        then
            py-ini-config set "$fold" Users MinimumUid "$(py-ini-config get $fsddm Users MinimumUid)"
        fi
        if grep -q 'MaximumUid=' "$fsddm"
        then
            py-ini-config set "$fold" Users MaximumUid "$(py-ini-config get $fsddm Users MaximumUid)"
        fi
        if ! _sddm_config "$fsddm"; then
            echo "Ошибка правки конфига SDDM"
            return 1
        fi
    else
        echo "Нет файла /etc/sddm.conf, пропускаем настройку SDDM"
    fi

    local system_auth="/var/lib/mos-auth/system-auth.latest"

    if test -f /etc/pam.d/system-auth
    then
        cp -f /etc/pam.d/system-auth /var/lib/mos-auth/system-auth.mos-auth-bak
        cp -f /etc/pam.d/system-auth "$system_auth"
        if grep -q 'prepare-guest-account' "$system_auth"
        then
            echo "PAM-config уже активирован."
        else
            echo "Активируем PAM-config..."

            if grep -q 'pam_sss.so' "$system_auth"
            then
                echo "Обнаружены настройки sssd в PAM-конфигурации. Пропускаем настройку faillock."
            else
                echo "Активируем настройку faillock..."
                awk -i inplace '{sub(/auth        sufficient    pam_unix.so try_first_pass likeauth nullok/,"auth        [success=1 default=bad] pam_unix.so try_first_pass likeauth nullok\nauth        [default=die] pam_faillock.so authfail\nauth        sufficient    pam_faillock.so authsucc"); print}' "$system_auth"
            fi
            echo "Активируем настройку для автоочистки профиля гостя..."
            line_number=$(awk '/session/{ print NR; exit }' "$system_auth")
            awk -i inplace -v n="$line_number" -v s="session     optional      pam_exec.so /usr/bin/prepare-guest-account" 'NR == n {print s} {print}' "$system_auth"

            cat "$system_auth" > /etc/pam.d/system-auth
        fi
    else
        echo "Нет файла /etc/pam.d/system-auth, пропускаем настройку PAM"
    fi
    if test -f /etc/pam.d/postlogin
    then
        cp -f /etc/pam.d/postlogin /var/lib/mos-auth/postlogin.mos-auth-bak
        echo "session     optional      pam_exec.so quiet /usr/bin/mount-shared-folder-on-login" >> /etc/pam.d/postlogin
    else
        echo "Нет файла /etc/pam.d/postlogin, пропускаем настройку PAM"
    fi
    touch /var/lib/mos-auth/enabled
}

_disable(){
    local fsddm
    local fold
    fold="/var/lib/mos-auth/mos-auth-old-settings"
    for fsddm in "/etc/sddm.conf" "/etc/sddm.conf.d/kde_settings.conf"
    do
        echo "$fsddm"
        if test -f "$fsddm"
        then
            if test -f "$fold" && grep -q 'Current=' "$fold"
            then
                py-ini-config set "$fsddm" Theme Current "$(py-ini-config get $fold Theme Current)"
            else
                py-ini-config del "$fsddm" Theme Current --flush || echo "Восстановление не требуется"
            fi
            if test -f "$fold" && grep -q 'MinimumUid=' "$fold"
            then
                py-ini-config set "$fsddm" Users MinimumUid "$(py-ini-config get $fold Users MinimumUid)"
            else
                py-ini-config del "$fsddm" Users MinimumUid --flush || echo "Восстановление не требуется"
            fi
            if test -f "$fold" && grep -q 'MaximumUid=' "$fold"
            then
                py-ini-config set "$fsddm" Users MaximumUid "$(py-ini-config get $fold Users MaximumUid)"
            else
                py-ini-config del "$fsddm" Users MaximumUid --flush || echo "Восстановление не требуется"
            fi
        fi
    done
    rm -f "$fold"
    sed -i '/mount-shared-folder-on-login/d' /etc/pam.d/postlogin

    local system_auth="/var/lib/mos-auth/system-auth.latest"

    if test -f /etc/pam.d/system-auth
    then
        cp -f /etc/pam.d/system-auth "$system_auth"
        if grep -q 'prepare-guest-account' "$system_auth"
        then
            echo "Деактивируем PAM-config..."

            if grep -q 'pam_sss.so' "$system_auth"
            then
                echo "Обнаружены настройки sssd в PAM-конфигурации. Пропускаем настройку faillock."
            else
                echo "Деактивируем настройку faillock..."
                awk -i inplace '{sub(/auth        \[success=1 default=bad\] pam_unix.so try_first_pass likeauth nullok/,"auth        sufficient    pam_unix.so try_first_pass likeauth nullok"); print}' "$system_auth"
                sed -i '/auth        \[default=die\] pam_faillock.so authfail/d' "$system_auth"
                sed -i '/auth        sufficient    pam_faillock.so authsucc/d' "$system_auth"
            fi

            echo "Деактивируем настройку для автоочистки профиля гостя..."
            sed -i '/prepare-guest-account/d' "$system_auth"

            cat "$system_auth" > /etc/pam.d/system-auth
        else
            echo "PAM-config уже деактивирован."
        fi
    else
        if test -f /var/lib/mos-auth/system-auth.mos-auth-bak
        then
            cat /var/lib/mos-auth/system-auth.mos-auth-bak > /etc/pam.d/system-auth
        else
            echo "Нет резервной копии файла '/etc/pam.d/system-auth'! Пропущено восстановление настроек PAM!"
        fi
    fi
    rm -f /var/lib/mos-auth/enabled
    echo "Конфигурация mos-auth успешно отключена."
}

_status(){
    if test -f /var/lib/mos-auth/enabled
    then
        echo "В данный момент авторизация через mos.ru включена."
    else
        echo "В данный момент авторизация через mos.ru выключена."
    fi
}

_edit(){
    command ${VISUAL:-${EDITOR:-$(command -v nano)}} /etc/mos-auth/mos-auth.conf
}

_main(){
    case "${1:-X}" in
        enable )
            _check_root
            _enable
        ;;
        disable )
            _check_root
            _disable
        ;;
        help )
            _help
        ;;
        status )
            _status
        ;;
        edit )
            _check_root
            _edit
        ;;
        * )
            _help
            exit 1
        ;;
    esac
}

if [ "${SOURCING:-0}" -ne 1 ]; then
    _main "$@"
fi
