# frozen_string_literal: true

require 'fast_spec_helper'

require 're2'

require_relative '../../app/services/service_response'
require_relative '../../lib/gitlab/error_tracking'

RSpec.describe ServiceResponse do
  describe '.success' do
    it 'creates a successful response without a message' do
      expect(described_class.success).to be_success
    end

    it 'creates a successful response with a message' do
      response = described_class.success(message: 'Good orange')

      expect(response).to be_success
      expect(response.message).to eq('Good orange')
    end

    it 'creates a successful response with payload' do
      response = described_class.success(payload: { good: 'orange' })

      expect(response).to be_success
      expect(response.payload).to eq(good: 'orange')
    end

    it 'creates a successful response with default HTTP status' do
      response = described_class.success

      expect(response).to be_success
      expect(response.http_status).to eq(:ok)
    end

    it 'creates a successful response with custom HTTP status' do
      response = described_class.success(http_status: 204)

      expect(response).to be_success
      expect(response.http_status).to eq(204)
    end
  end

  describe '.error' do
    it 'creates an error response without HTTP status' do
      response = described_class.error(message: 'Bad apple')

      expect(response).to be_error
      expect(response.message).to eq('Bad apple')
    end

    it 'creates an error response with HTTP status' do
      response = described_class.error(message: 'Bad apple', http_status: 400)

      expect(response).to be_error
      expect(response.message).to eq('Bad apple')
      expect(response.http_status).to eq(400)
    end

    it 'creates an error response with payload' do
      response = described_class.error(message: 'Bad apple',
                                       payload: { bad: 'apple' })

      expect(response).to be_error
      expect(response.message).to eq('Bad apple')
      expect(response.payload).to eq(bad: 'apple')
    end

    it 'creates an error response with a reason' do
      response = described_class.error(message: 'Bad apple',
                                       reason: :permission_denied)

      expect(response).to be_error
      expect(response.message).to eq('Bad apple')
      expect(response.reason).to eq(:permission_denied)
    end
  end

  describe '#success?' do
    it 'returns true for a successful response' do
      expect(described_class.success.success?).to eq(true)
    end

    it 'returns false for a failed response' do
      expect(described_class.error(message: 'Bad apple').success?).to eq(false)
    end
  end

  describe '#error?' do
    it 'returns false for a successful response' do
      expect(described_class.success.error?).to eq(false)
    end

    it 'returns true for a failed response' do
      expect(described_class.error(message: 'Bad apple').error?).to eq(true)
    end
  end

  describe '#errors' do
    it 'returns an empty array for a successful response' do
      expect(described_class.success.errors).to be_empty
    end

    it 'returns an array with a correct message for an error response' do
      expect(described_class.error(message: 'error message').errors).to eq(['error message'])
    end
  end

  describe '#track_and_raise_exception' do
    context 'when successful' do
      let(:response) { described_class.success }

      it 'returns self' do
        expect(response.track_and_raise_exception).to be response
      end
    end

    context 'when an error' do
      let(:response) { described_class.error(message: 'bang') }

      it 'tracks and raises' do
        expect(::Gitlab::ErrorTracking).to receive(:track_and_raise_exception)
          .with(StandardError.new('bang'), {})

        response.track_and_raise_exception
      end

      it 'allows specification of error class' do
        error = Class.new(StandardError)
        expect(::Gitlab::ErrorTracking).to receive(:track_and_raise_exception)
          .with(error.new('bang'), {})

        response.track_and_raise_exception(as: error)
      end

      it 'allows extra data for tracking' do
        expect(::Gitlab::ErrorTracking).to receive(:track_and_raise_exception)
          .with(StandardError.new('bang'), { foo: 1, bar: 2 })

        response.track_and_raise_exception(foo: 1, bar: 2)
      end
    end
  end

  describe '#track_exception' do
    context 'when successful' do
      let(:response) { described_class.success }

      it 'returns self' do
        expect(response.track_exception).to be response
      end
    end

    context 'when an error' do
      let(:response) { described_class.error(message: 'bang') }

      it 'tracks' do
        expect(::Gitlab::ErrorTracking).to receive(:track_exception)
          .with(StandardError.new('bang'), {})

        expect(response.track_exception).to be response
      end

      it 'allows specification of error class' do
        error = Class.new(StandardError)
        expect(::Gitlab::ErrorTracking).to receive(:track_exception)
          .with(error.new('bang'), {})

        expect(response.track_exception(as: error)).to be response
      end

      it 'allows extra data for tracking' do
        expect(::Gitlab::ErrorTracking).to receive(:track_exception)
          .with(StandardError.new('bang'), { foo: 1, bar: 2 })

        expect(response.track_exception(foo: 1, bar: 2)).to be response
      end
    end
  end

  describe '#log_and_raise_exception' do
    context 'when successful' do
      let(:response) { described_class.success }

      it 'returns self' do
        expect(response.log_and_raise_exception).to be response
      end
    end

    context 'when an error' do
      let(:response) { described_class.error(message: 'bang') }

      it 'logs' do
        expect(::Gitlab::ErrorTracking).to receive(:log_and_raise_exception)
          .with(StandardError.new('bang'), {})

        response.log_and_raise_exception
      end

      it 'allows specification of error class' do
        error = Class.new(StandardError)
        expect(::Gitlab::ErrorTracking).to receive(:log_and_raise_exception)
          .with(error.new('bang'), {})

        response.log_and_raise_exception(as: error)
      end

      it 'allows extra data for tracking' do
        expect(::Gitlab::ErrorTracking).to receive(:log_and_raise_exception)
          .with(StandardError.new('bang'), { foo: 1, bar: 2 })

        response.log_and_raise_exception(foo: 1, bar: 2)
      end
    end
  end
end
