# frozen_string_literal: true

require 'spec_helper'

RSpec.describe VulnerabilityIssueLinks::DeleteService do
  include AccessMatchersGeneric

  before do
    stub_licensed_features(security_dashboard: true)
  end

  let_it_be(:project) { create(:project) }
  let_it_be(:user) { create(:user) }
  let_it_be(:vulnerability_issue_link, refind: true) { create(:vulnerabilities_issue_link, project: project) }

  let(:service) { described_class.new(user, vulnerability_issue_link) }

  subject(:delete_issue_link) { service.execute }

  context 'with an authorized user with proper permissions' do
    before do
      project.add_developer(user)
    end

    context 'with valid params' do
      it 'deletes the specified vulnerability-issue link' do
        expect { delete_issue_link }.to change { Vulnerabilities::IssueLink.count }.by(-1)

        response = delete_issue_link
        expect(response).to be_success
        expect(response.http_status).to eq 200

        issue_link = response.payload[:record]
        expect(issue_link).to have_attributes(vulnerability_issue_link.attributes)
      end
    end

    context 'when security dashboard feature is disabled' do
      before do
        stub_licensed_features(security_dashboard: false)
      end

      it 'raises an "access denied" error' do
        expect { delete_issue_link }.to raise_error(Gitlab::Access::AccessDeniedError)
      end
    end
  end

  describe 'permissions' do
    context 'when admin mode is enabled', :enable_admin_mode do
      it { expect { delete_issue_link }.to be_allowed_for(:admin) }
    end

    context 'when admin mode is disabled' do
      it { expect { delete_issue_link }.to be_denied_for(:admin) }
    end

    it { expect { delete_issue_link }.to be_allowed_for(:owner).of(project) }
    it { expect { delete_issue_link }.to be_allowed_for(:maintainer).of(project) }
    it { expect { delete_issue_link }.to be_allowed_for(:developer).of(project) }

    it { expect { delete_issue_link }.to be_denied_for(:auditor) }
    it { expect { delete_issue_link }.to be_denied_for(:reporter).of(project) }
    it { expect { delete_issue_link }.to be_denied_for(:guest).of(project) }
    it { expect { delete_issue_link }.to be_denied_for(:anonymous) }
  end
end
