# frozen_string_literal: true

# Attribute mapping for alerts via generic alerting integration.
module EE
  module Gitlab
    module AlertManagement
      module Payload
        module Generic
          extend ::Gitlab::Utils::Override

          EXCLUDED_PAYLOAD_FINGERPRINT_PARAMS = %w(start_time end_time hosts).freeze
          CUSTOM_MAPPING_PATH_KEY = 'path'

          private

          # Currently we use full payloads, when generating a fingerprint.
          # This results in a quite strict fingerprint.
          # Over time we can relax these rules.
          # See https://gitlab.com/gitlab-org/gitlab/-/issues/214557#note_362795447
          override :plain_gitlab_fingerprint
          def plain_gitlab_fingerprint
            strong_memoize(:plain_gitlab_fingerprint) do
              next super if super.present?
              next unless generic_alert_fingerprinting_enabled?

              payload_excluding_params = payload.excluding(EXCLUDED_PAYLOAD_FINGERPRINT_PARAMS)

              next if payload_excluding_params.none? { |_, v| v.present? }

              payload_excluding_params
            end
          end

          def generic_alert_fingerprinting_enabled?
            project.feature_available?(:generic_alert_fingerprinting)
          end

          override :value_for_paths
          def value_for_paths(paths)
            custom_mapping_value_for_paths(paths) || super(paths)
          end

          def custom_mapping_value_for_paths(paths)
            return unless ::Gitlab::AlertManagement.custom_mapping_available?(project)
            return unless integration&.active?

            custom_mapping_path = integration.payload_attribute_mapping.dig(*paths.first, CUSTOM_MAPPING_PATH_KEY)

            payload&.dig(*custom_mapping_path) if custom_mapping_path
          end
        end
      end
    end
  end
end
