# frozen_string_literal: true

require 'spec_helper'

RSpec.describe Gitlab::BackgroundMigration::MigrateLinksForVulnerabilityFindings,
  feature_category: :vulnerability_management do
  let(:vulnerability_occurrences) { table(:vulnerability_occurrences) }
  let(:vulnerability_finding_links) { table(:vulnerability_finding_links) }
  let(:link_hash) { { url: 'http://test.com' } }
  let(:namespace1) { table(:namespaces).create!(name: 'namespace 1', path: 'namespace1') }
  let(:project1) { table(:projects).create!(namespace_id: namespace1.id, project_namespace_id: namespace1.id) }
  let(:user) { table(:users).create!(email: 'test1@example.com', projects_limit: 5) }

  let(:scanner1) do
    table(:vulnerability_scanners).create!(project_id: project1.id, external_id: 'test 1', name: 'test scanner 1')
  end

  let(:stating_id) { vulnerability_occurrences.pluck(:id).min }
  let(:end_id) { vulnerability_occurrences.pluck(:id).max }

  let(:migration) do
    described_class.new(
      start_id: stating_id,
      end_id: end_id,
      batch_table: :vulnerability_occurrences,
      batch_column: :id,
      sub_batch_size: 2,
      pause_ms: 2,
      connection: ApplicationRecord.connection
    )
  end

  subject(:perform_migration) { migration.perform }

  context 'without the presence of links key' do
    before do
      create_finding!(project1.id, scanner1.id, { other_keys: 'test' })
    end

    it 'does not create any link' do
      expect(Gitlab::AppLogger).not_to receive(:error)

      expect { perform_migration }.not_to change { vulnerability_finding_links.count }
    end
  end

  context 'with links equals to an array of nil element' do
    before do
      create_finding!(project1.id, scanner1.id, { links: [nil] })
    end

    it 'does not create any link' do
      expect(Gitlab::AppLogger).not_to receive(:error)

      expect { perform_migration }.not_to change { vulnerability_finding_links.count }
    end
  end

  context 'with links equals to an array of duplicated elements' do
    let!(:finding) do
      create_finding!(project1.id, scanner1.id, { links: [link_hash, link_hash] })
    end

    it 'creates one new link' do
      expect(Gitlab::AppLogger).not_to receive(:error)

      expect { perform_migration }.to change { vulnerability_finding_links.count }.by(1)
    end
  end

  context 'with existing links within raw_metadata' do
    let!(:finding1) { create_finding!(project1.id, scanner1.id, { links: [link_hash] }) }
    let!(:finding2) { create_finding!(project1.id, scanner1.id, { links: [link_hash] }) }

    it 'creates new link for each finding' do
      expect(Gitlab::AppLogger).not_to receive(:error)

      expect { perform_migration }.to change { vulnerability_finding_links.count }.by(2)
    end

    context 'when create throws exception ActiveRecord::RecordNotUnique' do
      before do
        allow(migration).to receive(:create_links).and_raise(ActiveRecord::RecordNotUnique)
      end

      it 'does not log this error nor create new records' do
        expect(Gitlab::AppLogger).not_to receive(:error)

        expect { perform_migration }.not_to change { vulnerability_finding_links.count }
      end
    end

    context 'when create throws exception StandardError' do
      before do
        allow(migration).to receive(:create_links).and_raise(StandardError)
      end

      it 'logs StandardError' do
        expect(Gitlab::AppLogger).to receive(:error).with({
          class: described_class.name, message: StandardError.to_s, model_id: finding1.id
        })
        expect(Gitlab::AppLogger).to receive(:error).with({
          class: described_class.name, message: StandardError.to_s, model_id: finding2.id
        })
        expect { perform_migration }.not_to change { vulnerability_finding_links.count }
      end
    end
  end

  context 'with existing link records' do
    let!(:finding) { create_finding!(project1.id, scanner1.id, { links: [link_hash] }) }

    before do
      vulnerability_finding_links.create!(vulnerability_occurrence_id: finding.id, url: link_hash[:url])
    end

    it 'does not create new link' do
      expect(Gitlab::AppLogger).not_to receive(:error)

      expect { perform_migration }.not_to change { vulnerability_finding_links.count }
    end
  end

  private

  def create_finding!(project_id, scanner_id, raw_metadata)
    vulnerability = table(:vulnerabilities).create!(project_id: project_id, author_id: user.id, title: 'test',
      severity: 4, confidence: 4, report_type: 0)

    identifier = table(:vulnerability_identifiers).create!(project_id: project_id, external_type: 'uuid-v5',
      external_id: 'uuid-v5', fingerprint: OpenSSL::Digest::SHA256.hexdigest(vulnerability.id.to_s),
      name: 'Identifier for UUIDv5 2 2')

    table(:vulnerability_occurrences).create!(
      vulnerability_id: vulnerability.id, project_id: project_id, scanner_id: scanner_id,
      primary_identifier_id: identifier.id, name: 'test', severity: 4, confidence: 4, report_type: 0,
      uuid: SecureRandom.uuid, project_fingerprint: '123qweasdzxc', location: { "image" => "alpine:3.4" },
      location_fingerprint: 'test', metadata_version: 'test',
      raw_metadata: raw_metadata.to_json)
  end
end
