# frozen_string_literal: true

require 'spec_helper'

RSpec.describe VulnerabilityIssueLinks::CreateService do
  include AccessMatchersGeneric

  before do
    stub_licensed_features(security_dashboard: true)
  end

  let(:user) { create(:user) }
  let(:project) { create(:project) }
  let(:vulnerability) { create(:vulnerability, project: project) }
  let(:issue) { create(:issue, project: vulnerability.project) }
  let(:service) { described_class.new(user, vulnerability, issue) }

  subject(:create_issue_link) { service.execute }

  context 'with an authorized user with proper permissions' do
    before do
      project.add_developer(user)
    end

    context 'with valid params' do
      it 'creates a new vulnerability-issue link' do
        expect { create_issue_link }.to change { Vulnerabilities::IssueLink.count }.by(1)

        response = create_issue_link
        expect(response).to be_success

        issue_link = response.payload[:record]
        expect(issue_link).to be_persisted
        expect(issue_link).to have_attributes(vulnerability: vulnerability, issue: issue, link_type: 'related')
      end
    end

    context 'with missing vulnerability' do
      let(:service) { described_class.new(user, nil, issue) }

      it 'responds with an error' do
        expect { create_issue_link }.to raise_error(Gitlab::Access::AccessDeniedError)
      end
    end

    context 'with missing issue' do
      let(:service) { described_class.new(user, vulnerability, nil) }

      it 'responds with an error', :aggregate_failures do
        expect { create_issue_link }.not_to change { Vulnerabilities::IssueLink.count }

        response = create_issue_link

        expect(response).to be_error
        expect(response.http_status).to eq 422
        expect(response.message).to eq "Issue can't be blank"
      end
    end

    context 'when a link between these issue and vulnerability already exists' do
      before do
        create(:vulnerabilities_issue_link, vulnerability: vulnerability, issue: issue)
      end

      it 'responds with an error about a conflicting data' do
        expect { create_issue_link }.not_to change { Vulnerabilities::IssueLink.count }

        response = create_issue_link

        expect(response).to be_error
        expect(response.http_status).to eq 422
        expect(response.message).to eq 'Issue has already been linked to another vulnerability'
      end
    end

    context 'when a "created" link already exists for a vulnerability' do
      before do
        create(:vulnerabilities_issue_link, vulnerability: vulnerability, issue: create(:issue), link_type: 'created')
      end

      let(:service) { described_class.new(user, vulnerability, issue, link_type: 'created') }

      it 'responds with an error about a conflicting data' do
        expect { create_issue_link }.not_to change { Vulnerabilities::IssueLink.count }

        response = create_issue_link

        expect(response).to be_error
        expect(response.http_status).to eq 422
        expect(response.message).to eq 'Vulnerability already has a "created" issue link'
      end
    end

    context 'when trying to relate an issue of a different project' do
      let(:issue) { create(:issue) }

      it 'creates a vulnerability-issue link' do
        expect { create_issue_link }.to change { Vulnerabilities::IssueLink.count }.by(1)
      end
    end

    context 'when trying to relate a confidential issue of the same project' do
      it 'creates a vulnerability-issue link' do
        expect { create_issue_link }.to change { Vulnerabilities::IssueLink.count }.by(1)
      end
    end

    context 'when security dashboard feature is disabled' do
      before do
        stub_licensed_features(security_dashboard: false)
      end

      it 'raises an "access denied" error' do
        expect { create_issue_link }.to raise_error(Gitlab::Access::AccessDeniedError)
      end
    end
  end

  describe 'permissions' do
    context 'when admin mode enabled', :enable_admin_mode do
      it { expect { create_issue_link }.to be_allowed_for(:admin) }
    end

    context 'when admin mode disabled' do
      it { expect { create_issue_link }.to be_denied_for(:admin) }
    end

    it { expect { create_issue_link }.to be_allowed_for(:owner).of(project) }
    it { expect { create_issue_link }.to be_allowed_for(:maintainer).of(project) }
    it { expect { create_issue_link }.to be_allowed_for(:developer).of(project) }

    it { expect { create_issue_link }.to be_denied_for(:auditor) }
    it { expect { create_issue_link }.to be_denied_for(:reporter).of(project) }
    it { expect { create_issue_link }.to be_denied_for(:guest).of(project) }
    it { expect { create_issue_link }.to be_denied_for(:anonymous) }
  end
end
