# frozen_string_literal: true

require 'spec_helper'

RSpec.describe Gitlab::BackgroundMigration::RemoveVulnerabilityFindingLinks, :migration, schema: 20211202041233 do
  let(:vulnerability_findings) { table(:vulnerability_occurrences) }
  let(:finding_links) { table(:vulnerability_finding_links) }

  let(:namespace) { table(:namespaces).create!(name: 'user', path: 'user', type: Namespaces::UserNamespace.sti_name) }
  let(:project) { table(:projects).create!(namespace_id: namespace.id) }
  let(:scanner) { table(:vulnerability_scanners).create!(project_id: project.id, external_id: 'scanner', name: 'scanner') }
  let(:vulnerability_identifier) do
    table(:vulnerability_identifiers).create!(
      project_id: project.id,
      external_type: 'vulnerability-identifier',
      external_id: 'vulnerability-identifier',
      fingerprint: '7e394d1b1eb461a7406d7b1e08f057a1cf11287a',
      name: 'vulnerability identifier')
  end

  # vulnerability findings
  let!(:findings) do
    Array.new(2) do |id|
      vulnerability_findings.create!(
        project_id: project.id,
        name: 'Vulnerability Name',
        severity: 7,
        confidence: 7,
        report_type: 0,
        project_fingerprint: '123qweasdzxc',
        scanner_id: scanner.id,
        primary_identifier_id: vulnerability_identifier.id,
        location_fingerprint: "location_fingerprint_#{id}",
        metadata_version: 'metadata_version',
        raw_metadata: 'raw_metadata',
        uuid: SecureRandom.uuid
      )
    end
  end

  # vulnerability finding links
  let!(:links) do
    {
      findings.first => Array.new(5) { |id| finding_links.create!(vulnerability_occurrence_id: findings.first.id, name: "Link Name 1", url: "link_url1_#{id}.example") },
      findings.second => Array.new(5) { |id| finding_links.create!(vulnerability_occurrence_id: findings.second.id, name: "Link Name 2", url: "link_url2_#{id}.example") }
    }
  end

  it 'removes vulnerability links' do
    expect do
      subject.perform(links[findings.first].first.id, links[findings.second].last.id)
    end.to change { finding_links.count }.from(10).to(0)

    expect(finding_links.all).to be_empty
  end

  it 'only deletes vulnerability links for the current batch' do
    expected_links = [finding_links.where(vulnerability_occurrence_id: findings.second.id)].flatten

    expect do
      subject.perform(links[findings.first].first.id, links[findings.first].last.id)
    end.to change { finding_links.count }.from(10).to(5)

    expect(finding_links.all).to match_array(expected_links)
  end
end
