# frozen_string_literal: true

require 'spec_helper'

RSpec.describe Gitlab::Llm::Templates::ExplainVulnerability, feature_category: :vulnerability_management do
  let_it_be(:source_code) do
    <<~SOURCE
    #include <stdio.h>

    int main(int argc, char *argv[])
    {
      char buf[8];
      memcpy(&buf, "123456789");
      printf("hello, world!");
    }
    SOURCE
  end

  let_it_be(:project) do
    create(:project, :custom_repo, files: {
      'src/main.c' => source_code
    })
  end

  let_it_be(:vulnerability) do
    create(:vulnerability, :with_finding, project: project)
  end

  before do
    vulnerability.finding.project = project
    vulnerability.finding.clear_memoization(:source_code)
    vulnerability.finding.clear_memoization(:vulnerable_code)
  end

  subject { described_class.new(vulnerability) }

  describe '#options' do
    context 'for OpenAI' do
      let(:client) { ::Gitlab::Llm::OpenAi::Client }

      it 'returns max tokens' do
        expect(subject.options(client)).to match(hash_including({
          max_tokens: described_class::MAX_TOKENS
        }))
      end
    end

    context 'for VertexAI' do
      let(:client) { ::Gitlab::Llm::VertexAi::Client }

      it 'returns max tokens' do
        expect(subject.options(client)).to be_empty
      end
    end
  end

  describe '#to_prompt' do
    let(:identifiers) { vulnerability.finding.identifiers.pluck(:name).join(", ") }

    context 'when a file is provided' do
      context 'when the file exists' do
        before do
          vulnerability.finding.location['file'] = 'src/main.c'
          vulnerability.finding.location['start_line'] = 5
          vulnerability.finding.location['end_line'] = 6
        end

        it 'includes the title' do
          expect(subject.to_prompt).to include(vulnerability.title)
        end

        it 'includes the description' do
          expect(subject.to_prompt).to include(vulnerability.description)
        end

        it 'includes the identifiers' do
          expect(subject.to_prompt).to include(identifiers)
        end

        it 'includes the file name' do
          expect(subject.to_prompt).to include('"main.c"')
        end

        it 'includes the vulnerable code' do
          vulnerable_code = source_code.lines[4..5].join
          expect(subject.to_prompt).to include(vulnerable_code)
        end
      end

      context 'when the vulnerability is for a secret detection' do
        before do
          vulnerability.report_type = :secret_detection
          vulnerability.finding.location['file'] = 'src/main.c'
          vulnerability.finding.location['start_line'] = 5
        end

        it 'does not include the vulnerable code' do
          vulnerable_code = source_code.lines[4..5].join
          expect(subject.to_prompt).not_to include(vulnerable_code)
        end
      end

      context 'when there is more vulnerable code than the maximum allowed' do
        let_it_be(:source_code) { "a" * (described_class::MAX_CODE_LENGTH + 1) }

        let_it_be(:project) do
          create(:project, :custom_repo, files: {
            'jquery.min.js' => source_code
          })
        end

        before do
          vulnerability.finding.location['file'] = 'jquery.min.js'
          vulnerability.finding.location['start_line'] = 1
        end

        it 'does not include vulnerable code' do
          expect(subject.to_prompt).not_to include(source_code)
        end
      end

      context 'when the file does not exist' do
        before do
          vulnerability.finding.location['file'] = 'missing.c'
        end

        it 'customizes the prompt' do
          expect(subject.to_prompt).to eq(<<~PROMPT)
          You are a software vulnerability developer.
          Explain the vulnerability "#{vulnerability.title} - #{vulnerability.description} (#{identifiers})".
          The vulnerable code is in the file "#{vulnerability.file}".
          Provide a code example with syntax highlighting on how to exploit it.
          Provide a code example with syntax highlighting on how to fix it.
          Provide the response in markdown format with headers.
          PROMPT
        end
      end
    end

    context 'when a file is not provided' do
      before do
        vulnerability.finding.location.delete('file')
      end

      it 'customizes the prompt' do
        expected = <<~PROMPT
        You are a software vulnerability developer.
        Explain the vulnerability "#{vulnerability.title} - #{vulnerability.description} (#{identifiers})".
        Provide a code example with syntax highlighting on how to exploit it.
        Provide a code example with syntax highlighting on how to fix it.
        Provide the response in markdown format with headers.
        PROMPT

        expect(subject.to_prompt).to eq(expected)
      end
    end
  end
end
