# frozen_string_literal: true

module Gitlab
  module Llm
    module VertexAi
      module Completions
        class ExplainVulnerability < Gitlab::Llm::Completions::Base
          DEFAULT_ERROR = 'An unexpected error has occurred.'

          def execute(user, vulnerability, options)
            unless vertex_ai?(vulnerability)
              return ::Gitlab::Llm::OpenAi::Completions::ExplainVulnerability
                .new(ai_prompt_class)
                .execute(user, vulnerability, options)
            end

            response = response_for(user, vulnerability)
            response_modifier = ::Gitlab::Llm::VertexAi::ResponseModifiers::Predictions.new(response)

            ::Gitlab::Llm::GraphqlSubscriptionResponseService.new(
              user, vulnerability, response_modifier, options: { request_id: params[:request_id] }
            ).execute
          rescue StandardError => error
            Gitlab::ErrorTracking.track_exception(error)

            response_modifier = ::Gitlab::Llm::VertexAi::ResponseModifiers::Predictions.new(
              { error: { message: DEFAULT_ERROR } }.to_json
            )

            ::Gitlab::Llm::GraphqlSubscriptionResponseService.new(
              user, vulnerability, response_modifier, options: { request_id: params[:request_id] }
            ).execute
          end

          private

          def response_for(user, vulnerability)
            key = [vulnerability.cache_key, 'explain'].join('/')
            Rails.cache.fetch(key, expires_in: 5.minutes, skip_nil: true) do
              template = ai_prompt_class.new(vulnerability)
              request(user, template)
            end
          end

          def request(user, template)
            client_class = ::Gitlab::Llm::VertexAi::Client
            client_class
              .new(user)
              .chat(content: template.to_prompt, **template.options(client_class))
          end

          def vertex_ai?(vulnerability)
            Feature.enabled?(:explain_vulnerability_vertex, vulnerability.project)
          end
        end
      end
    end
  end
end
