# frozen_string_literal: true

module Gitlab
  module Llm
    module Templates
      class ExplainVulnerability
        include Gitlab::Utils::StrongMemoize

        MAX_CODE_LENGTH = 1_024
        MAX_TOKENS = 3_000

        def initialize(vulnerability)
          @vulnerability = vulnerability
        end

        def options(client)
          return {} unless client == ::Gitlab::Llm::OpenAi::Client

          { max_tokens: MAX_TOKENS }
        end

        def to_prompt
          return prompt_without_file_or_code unless file.present?
          return prompt_without_code unless eligible_code?

          default_prompt
        end

        private

        attr_reader :vulnerability

        delegate :title, :description, :file, to: :vulnerability
        delegate :source_code?, :vulnerable_code, to: :finding

        # rubocop: disable CodeReuse/ActiveRecord
        def identifiers
          finding.identifiers.pluck(:name).join(", ")
        end
        # rubocop: enable CodeReuse/ActiveRecord

        def finding
          vulnerability.finding
        end
        strong_memoize_attr :finding

        def filename
          File.basename(file)
        end

        def eligible_code?
          return false unless source_code?
          return false if vulnerable_code.length > MAX_CODE_LENGTH
          return false if vulnerability.secret_detection?

          true
        end

        def default_prompt
          <<~PROMPT
          You are a software vulnerability developer.
          Explain the vulnerability "#{title} - #{description} (#{identifiers})".
          The file "#{filename}" has this vulnerable code:

          ```
          #{vulnerable_code}"
          ```

          Provide a code example with syntax highlighting on how to exploit it.
          Provide a code example with syntax highlighting on how to fix it.
          Provide the response in markdown format with headers.
          PROMPT
        end

        def prompt_without_code
          <<~PROMPT
          You are a software vulnerability developer.
          Explain the vulnerability "#{title} - #{description} (#{identifiers})".
          The vulnerable code is in the file "#{filename}".
          Provide a code example with syntax highlighting on how to exploit it.
          Provide a code example with syntax highlighting on how to fix it.
          Provide the response in markdown format with headers.
          PROMPT
        end

        def prompt_without_file_or_code
          <<~PROMPT
          You are a software vulnerability developer.
          Explain the vulnerability "#{title} - #{description} (#{identifiers})".
          Provide a code example with syntax highlighting on how to exploit it.
          Provide a code example with syntax highlighting on how to fix it.
          Provide the response in markdown format with headers.
          PROMPT
        end
      end
    end
  end
end
