# frozen_string_literal: true

require 'spec_helper'
require_migration!

RSpec.describe ScheduleRecalculateVulnerabilityFindingSignaturesForFindings, :migration,
feature_category: :vulnerability_management do
  before do
    allow(Gitlab).to receive(:ee?).and_return(ee?)
    stub_const("#{described_class.name}::BATCH_SIZE", 2)
  end

  context 'when the Gitlab instance is FOSS' do
    let(:ee?) { false }

    it 'does not run the migration' do
      expect { migrate! }.not_to change { BackgroundMigrationWorker.jobs.size }
    end
  end

  context 'when the Gitlab instance is EE' do
    let(:ee?) { true }

    let!(:namespaces) { table(:namespaces) }
    let!(:projects) { table(:projects) }
    let!(:findings) { table(:vulnerability_occurrences) }
    let!(:scanners) { table(:vulnerability_scanners) }
    let!(:identifiers) { table(:vulnerability_identifiers) }
    let!(:vulnerability_finding_signatures) { table(:vulnerability_finding_signatures) }

    let!(:namespace) { namespaces.create!(name: 'test', path: 'test') }
    let!(:project) { projects.create!(namespace_id: namespace.id, name: 'gitlab', path: 'gitlab') }

    let!(:scanner) do
      scanners.create!(project_id: project.id, external_id: 'trivy', name: 'Security Scanner')
    end

    let!(:identifier) do
      identifiers.create!(project_id: project.id,
                          fingerprint: 'd432c2ad2953e8bd587a3a43b3ce309b5b0154c123',
                          external_type: 'SECURITY_ID',
                          external_id: 'SECURITY_0',
                          name: 'SECURITY_IDENTIFIER 0')
    end

    let!(:finding1) { findings.create!(finding_params) }
    let!(:signature1) { vulnerability_finding_signatures.create!(finding_id: finding1.id, algorithm_type: 0, signature_sha: ::Digest::SHA1.digest(SecureRandom.hex(50))) }

    let!(:finding2) { findings.create!(finding_params) }
    let!(:signature2) { vulnerability_finding_signatures.create!(finding_id: finding2.id, algorithm_type: 0, signature_sha: ::Digest::SHA1.digest(SecureRandom.hex(50))) }

    let!(:finding3) { findings.create!(finding_params) }
    let!(:signature3) { vulnerability_finding_signatures.create!(finding_id: finding3.id, algorithm_type: 0, signature_sha: ::Digest::SHA1.digest(SecureRandom.hex(50))) }

    # this migration is now a no-op
    it 'does not schedule the background jobs', :aggregate_failure do
      Sidekiq::Testing.fake! do
        freeze_time do
          migrate!

          expect(BackgroundMigrationWorker.jobs.size).to eq(0)
          expect(described_class::MIGRATION)
            .not_to be_scheduled_migration_with_multiple_args(signature1.id, signature2.id)
          expect(described_class::MIGRATION)
            .not_to be_scheduled_migration_with_multiple_args(signature3.id, signature3.id)
        end
      end
    end

    def finding_params
      uuid = SecureRandom.uuid

      {
        severity: 0,
        confidence: 5,
        report_type: 2,
        project_id: project.id,
        scanner_id: scanner.id,
        primary_identifier_id: identifier.id,
        location: nil,
        project_fingerprint: SecureRandom.hex(20),
        location_fingerprint: Digest::SHA1.hexdigest(SecureRandom.hex(10)),
        uuid: uuid,
        name: "Vulnerability Finding #{uuid}",
        metadata_version: '1.3',
        raw_metadata: '{}'
      }
    end
  end
end
