# frozen_string_literal: true

module QA
  RSpec.describe 'Govern', :runner, product_group: :threat_insights do
    describe 'project vulnerability report' do
      let!(:project) do
        Resource::Project.fabricate_via_api! do |project|
          project.name = 'project-with-secure'
          project.description = 'Project with secure'
          project.auto_devops_enabled = false
          project.initialize_with_readme = true
        end
      end

      let(:vulnerabilities) do
        { "Greyhound vulnerability": "Describes vulnerabilities of greyhounds in detail",
          "Scenthound vulnerability": "Describes vulnerabilities with scents",
          "Spellbound vulnerability": "Describes harry potter spells",
          "Sonicsound vulnerability": "Describes if lightening is faster than thunder" }
      end

      let!(:vulnerability_report) do
        vulnerabilities.each do |name, description|
          QA::EE::Resource::VulnerabilityItem.fabricate_via_api! do |vulnerability|
            vulnerability.id = project.id
            vulnerability.severity = vuln_severity
            vulnerability.name = name
            vulnerability.description = description
          end
        end
      end

      let(:vuln_name) { "Greyhound vulnerability" }
      let(:vuln_status_hash) { { 'dismissed': 'dismiss', 'resolved': 'resolve', 'confirmed': 'confirm' } }
      let(:vuln_status_sample) { vuln_status_hash.keys.sample.to_s }
      let(:change_state) { vuln_status_hash[vuln_status_sample.to_sym] }
      let(:vuln_severity) { :CRITICAL }

      before do
        Flow::Login.sign_in
        project.visit!
      end

      it 'can successfully change status of a vulnerability in vulnerability details page', :reliable,
         testcase: 'https://gitlab.com/gitlab-org/gitlab/-/quality/test_cases/361048' do
        Page::Project::Menu.perform(&:click_on_vulnerability_report)

        EE::Page::Project::Secure::SecurityDashboard.perform do |security_dashboard|
          expect(security_dashboard).to have_vulnerability(description: vuln_name)

          security_dashboard.click_vulnerability(description: vuln_name)
        end

        EE::Page::Project::Secure::VulnerabilityDetails.perform do |vulnerability_details|
          vulnerability_details.change_status(vuln_status_sample)

          expect(vulnerability_details).to have_vulnerability_status(vuln_status_sample)
        end
        Page::Project::Menu.perform(&:click_on_vulnerability_report)
        EE::Page::Project::Secure::Show.perform do |security_dashboard|
          security_dashboard.filter_by_status([vuln_status_sample])

          expect(security_dashboard).to have_vulnerability(vuln_name)
        end
      end

      it 'can successfully bulk change status in vulnerability report', :reliable,
         testcase: 'https://gitlab.com/gitlab-org/gitlab/-/quality/test_cases/361405' do
        Page::Project::Menu.perform(&:click_on_vulnerability_report)
        EE::Page::Project::Secure::SecurityDashboard.perform do |security_dashboard|
          security_dashboard.select_all_vulnerabilities
          security_dashboard.change_state(change_state)
        end

        EE::Page::Project::Secure::Show.perform do |security_dashboard|
          security_dashboard.filter_by_status([vuln_status_sample])
          vulnerabilities.each do |name, _description|
            expect(security_dashboard).to have_vulnerability(name)
            expect(security_dashboard).to have_status(vuln_status_sample, name)
          end
        end
      end
    end
  end
end
