# frozen_string_literal: true

require 'spec_helper'

RSpec.describe Gitlab::BackgroundMigration::UpdateVulnerabilityOccurrencesLocation, schema: 20211102114802 do
  let!(:namespaces) { table(:namespaces) }
  let!(:group) { namespaces.create!(name: 'foo', path: 'foo', type: Group.sti_name) }
  let!(:projects) { table(:projects) }
  let!(:findings) { table(:vulnerability_occurrences) }
  let!(:scanners) { table(:vulnerability_scanners) }
  let!(:identifiers) { table(:vulnerability_identifiers) }

  let!(:project) { projects.create!(id: 123, namespace_id: group.id, name: 'gitlab', path: 'gitlab') }

  let!(:scanner) do
    scanners.create!(id: 6, project_id: project.id, external_id: 'trivy', name: 'Security Scanner')
  end

  let!(:identifier) do
    identifiers.create!(id: 123,
                        project_id: 123,
                        fingerprint: 'd432c2ad2953e8bd587a3a43b3ce309b5b0154c123',
                        external_type: 'SECURITY_ID',
                        external_id: 'SECURITY_0',
                        name: 'SECURITY_IDENTIFIER 0')
  end

  let!(:location) do
    { "file" => "maven/src/main/java/com/gitlab/security_products/tests/App.java", "start_line" => 29, "end_line" => 29, "class" => "com.gitlab.security_products.tests.App", "method" => "insecureCypher" }
  end

  let!(:raw_metadata) do
    { "description" => "The cipher does not provide data integrity update 1",
      "message" => "The cipher does not provide data integrity",
      "cve" => "818bf5dacb291e15d9e6dc3c5ac32178:CIPHER",
      "solution" => "GCM mode introduces an HMAC into the resulting encrypted data, providing integrity of the result.",
      "location" => location,
      "links" => [{ "name" => "Cipher does not check for integrity first?", "url" => "https://crypto.stackexchange.com/questions/31428/pbewithmd5anddes-cipher-does-not-check-for-integrity-first" }],
      "assets" => [{ "type" => "postman", "name" => "Test Postman Collection", "url" => "http://localhost/test.collection" }],
      "evidence" =>
        { "summary" => "Credit card detected",
          "request" => { "headers" => [{ "name" => "Accept", "value" => "*/*" }], "method" => "GET", "url" => "http://goat:8080/WebGoat/logout", "body" => nil },
          "response" => { "headers" => [{ "name" => "Content-Length", "value" => "0" }], "reason_phrase" => "OK", "status_code" => 200, "body" => nil },
          "source" => { "id" => "assert:Response Body Analysis", "name" => "Response Body Analysis", "url" => "htpp://hostname/documentation" },
          "supporting_messages" =>
            [{ "name" => "Origional", "request" => { "headers" => [{ "name" => "Accept", "value" => "*/*" }], "method" => "GET", "url" => "http://goat:8080/WebGoat/logout", "body" => "" } },
             { "name" => "Recorded",
               "request" => { "headers" => [{ "name" => "Accept", "value" => "*/*" }], "method" => "GET", "url" => "http://goat:8080/WebGoat/logout", "body" => "" },
               "response" => { "headers" => [{ "name" => "Content-Length", "value" => "0" }], "reason_phrase" => "OK", "status_code" => 200, "body" => "" } }] } }
  end

  shared_examples 'does not update location' do
    it 'does not update location' do
      expect(findings.where(location: nil).count).to eq(2)

      described_class.new.perform(vul1.id, vul2.id)

      expect(findings.where(location: nil).count).to eq(2)
      expect(vul1.reload.read_attribute(:location)).to be_nil
      expect(vul2.reload.read_attribute(:location)).to be_nil
    end
  end

  context 'when raw_metadata is a valid json' do
    let!(:vul1) { findings.create!(finding_params) }
    let!(:vul2) { findings.create!(finding_params) }
    let!(:vul3) { findings.create!(finding_params.merge(location: { "cluster_id" => "1" })) }

    it 'updates location' do
      expect(findings.where(location: nil).count).to eq(2)

      described_class.new.perform(vul1.id, vul2.id)

      expect(findings.where(location: nil).count).to eq(0)
      expect(vul1.reload.location).to eq(raw_metadata['location'])
      expect(vul2.reload.location).to eq(raw_metadata['location'])
      expect(vul3.reload.location).to eq({ "cluster_id" => "1" })
    end
  end

  context 'when raw_metadata is invalid json' do
    let!(:raw_metadata) { 'invalid' }
    let!(:vul1) { findings.create!(finding_params) }
    let!(:vul2) { findings.create!(finding_params) }

    it_behaves_like 'does not update location'
  end

  context 'when raw_metadata is empty' do
    let!(:raw_metadata) { '' }
    let!(:vul1) { findings.create!(finding_params) }
    let!(:vul2) { findings.create!(finding_params) }

    it_behaves_like 'does not update location'
  end

  def finding_params
    uuid = SecureRandom.uuid

    {
      severity: 0,
      confidence: 5,
      report_type: 2,
      project_id: 123,
      scanner_id: 6,
      primary_identifier_id: 123,
      location: nil,
      project_fingerprint: SecureRandom.hex(20),
      location_fingerprint: Digest::SHA1.hexdigest(SecureRandom.hex(10)),
      uuid: uuid,
      name: "Vulnerability Finding #{uuid}",
      metadata_version: '1.3',
      raw_metadata: raw_metadata.to_json
    }
  end
end
