# frozen_string_literal: true

require 'spec_helper'

RSpec.describe GitlabSchema.types['VulnerabilityEvidence'] do
  let_it_be(:project) { create(:project) }
  let_it_be(:user) { create(:user) }
  let_it_be(:pipeline) { create(:ee_ci_pipeline, :with_api_fuzzing_report, project: project) }

  before do
    stub_licensed_features(api_fuzzing: true, security_dashboard: true)

    project.add_developer(user)
  end

  let(:fields) do
    %i[summary supporting_messages source request response]
  end

  subject { GitlabSchema.execute(query, context: { current_user: user }).as_json }

  specify { expect(described_class.graphql_name).to eq('VulnerabilityEvidence') }

  it { expect(described_class).to have_graphql_fields(fields) }

  describe 'checking field contents' do
    let(:query) do
      %(
        query {
          project(fullPath: "#{project.full_path}") {
            pipeline(iid: "#{pipeline.iid}") {
              securityReportFindings {
                nodes {
                  title
                  evidence {
                    summary
                  }
                }
              }
            }
          }
        }
      )
    end

    it 'checks the contents of the evidence.summary field' do
      vulnerabilities = subject.dig('data', 'project', 'pipeline', 'securityReportFindings', 'nodes')

      vulnerability = vulnerabilities.find { |v| v['title'] == "CORS misconfiguration at 'http://127.0.0.1:7777/api/users'" }

      expect(vulnerability['evidence']['summary']).to eq("The Origin header was changed to an invalid value of http://peachapisecurity.com and the response contained an Access-Control-Allow-Origin header which included this invalid Origin, indicating that the CORS configuration on the server is overly permissive.\n\n\n")
    end
  end
end
