import { nextTick } from 'vue';
import { GlFormGroup } from '@gitlab/ui';
import { mountExtended } from 'helpers/vue_test_utils_helper';
import SectionIdentifiers from 'ee/vulnerabilities/components/new_vulnerability/section_identifiers.vue';

describe('New vulnerability - Section Identifiers', () => {
  let wrapper;

  const createWrapper = () => {
    return mountExtended(SectionIdentifiers, {
      propsData: {
        validationState: { identifiers: [{ identifierCode: false }] },
      },
    });
  };

  beforeEach(() => {
    wrapper = createWrapper();
  });

  afterEach(() => {
    wrapper.destroy();
  });

  const findIdentifierRows = () => wrapper.findAllByTestId('identifier-row');
  const findFormGroup = (index) => wrapper.findAllComponents(GlFormGroup).at(index);
  const findIdentifierCodeInput = () => wrapper.findByLabelText('Identifier code');
  const findIdentifierUrlInput = () => wrapper.findByLabelText('Identifier URL');

  it('does not display a warning when the validation state is emtpy', async () => {
    wrapper.setProps({
      validationState: {
        identifiers: [],
      },
    });

    await nextTick();

    expect(findFormGroup(1).attributes('aria-invalid')).toBeUndefined();
    expect(findFormGroup(0).attributes('aria-invalid')).toBeUndefined();
  });

  it('displays a warning when the validation fails', async () => {
    wrapper.setProps({
      validationState: {
        identifiers: [{ identifierCode: false, identifierUrl: false }],
      },
    });

    await nextTick();

    expect(findFormGroup(0).attributes('aria-invalid')).toBe('true');
    expect(wrapper.findAllByRole('alert').at(0).text()).toBe('Enter the CVE or CWE code');

    expect(findFormGroup(1).attributes('aria-invalid')).toBe('true');
    expect(wrapper.findAllByRole('alert').at(1).text()).toBe('Enter the CVE or CWE identifier URL');
  });

  it('emits change event when input changes', () => {
    const codeInput = findIdentifierCodeInput();
    const urlInput = findIdentifierUrlInput();

    codeInput.setValue('cve-23');
    urlInput.setValue('https://gitlab.com');

    expect(wrapper.emitted('change')[0][0]).toEqual({
      identifiers: [{ name: 'cve-23', url: '' }],
    });

    expect(wrapper.emitted('change')[1][0]).toEqual({
      identifiers: [{ name: 'cve-23', url: 'https://gitlab.com' }],
    });
  });

  it('adds and removes identifier rows', async () => {
    expect(findIdentifierRows()).toHaveLength(1);

    wrapper.findByRole('button', { name: 'Add another identifier' }).trigger('click');
    await nextTick();

    expect(findIdentifierRows()).toHaveLength(2);

    expect(wrapper.emitted('change')[0][0]).toEqual({
      identifiers: [
        { name: '', url: '' },
        { name: '', url: '' },
      ],
    });

    wrapper.findByLabelText('Remove identifier row').trigger('click');
    await nextTick();

    expect(findIdentifierRows()).toHaveLength(1);
    expect(wrapper.findByLabelText('Remove identifier row').exists()).toBe(false);
  });
});
