# frozen_string_literal: true

class Projects::VulnerabilityFeedbackController < Projects::ApplicationController
  before_action :vulnerability_feedback, only: [:update, :destroy]
  before_action :authorize_read_vulnerability_feedback!, only: [:index]
  before_action :authorize_create_vulnerability_feedback!, only: [:create]
  before_action :authorize_update_vulnerability_feedback!, only: [:update]
  before_action :authorize_destroy_vulnerability_feedback!, only: [:destroy]

  skip_before_action :authenticate_user!, only: [:index], raise: false

  respond_to :json

  feature_category :vulnerability_management
  urgency :low

  def count
    render json: { count: vulnerability_feedbacks.count }
  end

  def index
    if Feature.enabled?(:disable_vulnerability_feedback_controller, project)
      render json: []
    else
      # This action is currently still used in the MR security widget.
      # Their use will be removed as part of https://gitlab.com/gitlab-org/gitlab/-/issues/390281
      serializer = Vulnerabilities::FeedbackSerializer.new(current_user: current_user, project: project)
      render json: serializer.represent(optionally_paginate(vulnerability_feedbacks))
    end
  end

  def create
    service = VulnerabilityFeedback::CreateService.new(project, current_user, vulnerability_feedback_params)
    result = service.execute

    response_for(result)
  end

  def update
    service = VulnerabilityFeedbackModule::UpdateService.new(project, current_user, vulnerability_feedback_params)
    result = service.execute(vulnerability_feedback)

    response_for(result)
  end

  def destroy
    service = VulnerabilityFeedback::DestroyService.new(project, current_user, vulnerability_feedback)
    service.execute

    head :no_content
  end

  private

  def response_for(result)
    if result[:status] == :success
      # We will replace this with vulnerability from response after https://gitlab.com/gitlab-org/gitlab/-/issues/390156
      serializeable = if Feature.enabled?(:deprecate_vulnerabilities_feedback, project)
                        result[:vulnerability_feedback].finding&.vulnerability
                      else
                        result[:vulnerability_feedback]
                      end

      render json: serializer.represent(serializeable)
    else
      render json: result[:message], status: :unprocessable_entity
    end
  end

  def vulnerability_feedbacks
    @vulnerability_feedbacks ||= Security::VulnerabilityFeedbacksFinder.new(project, params).execute
  end

  def optionally_paginate(relation)
    return relation unless params[:page] || params[:per_page]

    relation.page(params[:page]).per(params[:per_page])
  end

  def authorize_create_vulnerability_feedback!
    type = vulnerability_feedback_params[:feedback_type]

    return render_403 unless Vulnerabilities::Feedback.feedback_types.key?(type)

    feedback = Vulnerabilities::Feedback.new(project: project, feedback_type: type)
    return render_403 unless can?(current_user, :create_vulnerability_feedback, feedback)
  end

  def authorize_destroy_vulnerability_feedback!
    render_403 unless can?(current_user, :destroy_vulnerability_feedback, vulnerability_feedback)
  end

  def authorize_update_vulnerability_feedback!
    render_403 unless can?(current_user, :update_vulnerability_feedback, vulnerability_feedback)
  end

  def serializer
    # This is needed as we do not want to depend on Vulnerabilities::Feedback in the frontend anymore.
    if Feature.enabled?(:deprecate_vulnerabilities_feedback, project)
      VulnerabilitySerializer.new
    else
      Vulnerabilities::FeedbackSerializer.new(current_user: current_user, project: project)
    end
  end

  def vulnerability_feedback
    @vulnerability_feedback ||= @project.vulnerability_feedback.find(params[:id])
  end

  def vulnerability_feedback_params
    params.require(:vulnerability_feedback).permit(*vulnerability_feedback_params_attributes)
  end

  def vulnerability_feedback_params_attributes
    %i[
      category
      feedback_type
      pipeline_id
      project_fingerprint
      comment
      finding_uuid
    ] + [
      vulnerability_data: vulnerability_data_params_attributes
    ]
  end

  def vulnerability_data_params_attributes
    %i[
      category
      confidence
      count
      cve
      cweid
      desc
      description
      featurename
      featureversion
      fingerprint
      fixedby
      link
      message
      name
      namespace
      otherinfo
      pluginid
      priority
      project_fingerprint
      reference
      riskcode
      riskdesc
      severity
      solution
      sourceid
      target_branch
      title
      tool
      tools
      url
      vulnerability_id
      wascid
    ] + [
      instances: %i[
        param
        method
        uri
      ],
      location: %i[
        file
        start_line
        end_line
        class
        method
        blob_path
      ],
      identifiers: %i[
        type
        name
        value
        url
      ],
      links: %i[
        name
        url
      ],
      remediations: %i[
        diff
      ],
      scanner: %i[
        external_id
        name
        vendor
      ],
      scan: %i[
        type
        status
        start_time
        end_time
      ]
    ]
  end
end
