# frozen_string_literal: true

require 'spec_helper'

RSpec.describe VulnerabilityMergeRequestLinks::CreateService do
  include AccessMatchersGeneric

  let_it_be(:user) { create(:user) }
  let_it_be(:project) { create(:project) }
  let_it_be(:vulnerability) { create(:vulnerability, project: project) }
  let_it_be(:merge_request) { create(:merge_request, source_project: project) }
  let(:params) { { vulnerability: vulnerability, merge_request: merge_request } }
  let(:service) { described_class.new(project: project, current_user: user, params: params) }

  before do
    stub_licensed_features(security_dashboard: true)
  end

  subject(:create_merge_request_link) { service.execute }

  context 'with an authorized user with proper permissions' do
    before do
      project.add_developer(user)
    end

    shared_examples 'new vulnerability-merge_request link created' do
      it 'creates a new vulnerability-merge_request link' do
        expect { create_merge_request_link }.to change(Vulnerabilities::MergeRequestLink, :count).by(1)

        response = create_merge_request_link
        expect(response).to be_success

        merge_request_link = response.payload[:merge_request_link]
        expect(merge_request_link).to be_persisted
        expect(merge_request_link).to have_attributes(vulnerability: vulnerability, merge_request: merge_request)
      end
    end

    context 'with valid params' do
      it_behaves_like 'new vulnerability-merge_request link created'
    end

    context 'with missing vulnerability' do
      let_it_be(:vulnerability) { nil }

      it 'responds with an error' do
        expect { create_merge_request_link }.to raise_error(Gitlab::Access::AccessDeniedError)
      end
    end

    context 'with missing merge_request' do
      let_it_be(:merge_request) { nil }

      it 'responds with an error', :aggregate_failures do
        expect { create_merge_request_link }.not_to change(Vulnerabilities::MergeRequestLink, :count)

        response = create_merge_request_link

        expect(response).to be_error
        expect(response.message).to eq "Merge request can't be blank"
      end
    end

    context 'when a link between the merge_request and vulnerability already exists' do
      before do
        create(:vulnerabilities_merge_request_link, vulnerability: vulnerability, merge_request: merge_request)
      end

      it 'responds with an error about a conflicting data' do
        expect { create_merge_request_link }.not_to change(Vulnerabilities::MergeRequestLink, :count)

        response = create_merge_request_link

        expect(response).to be_error
        expect(response.message).to eq 'Merge request is already linked to this vulnerability'
      end
    end

    context 'when a link between another merge_request and vulnerability already exists' do
      let_it_be(:another_merge_request) do
        create(:merge_request, source_branch: 'testing', source_project: project,
                               target_project: project)
      end

      before do
        create(:vulnerabilities_merge_request_link, vulnerability: vulnerability, merge_request: another_merge_request)
      end

      it_behaves_like 'new vulnerability-merge_request link created'
    end

    context 'when security dashboard feature is disabled' do
      before do
        stub_licensed_features(security_dashboard: false)
      end

      it 'raises an "access denied" error' do
        expect { create_merge_request_link }.to raise_error(Gitlab::Access::AccessDeniedError)
      end
    end
  end

  describe 'permissions' do
    context 'when admin mode enabled', :enable_admin_mode do
      it { expect { create_merge_request_link }.to be_allowed_for(:admin) }
    end

    context 'when admin mode disabled' do
      it { expect { create_merge_request_link }.to be_denied_for(:admin) }
    end

    it { expect { create_merge_request_link }.to be_allowed_for(:owner).of(project) }
    it { expect { create_merge_request_link }.to be_allowed_for(:maintainer).of(project) }
    it { expect { create_merge_request_link }.to be_allowed_for(:developer).of(project) }

    it { expect { create_merge_request_link }.to be_denied_for(:auditor) }
    it { expect { create_merge_request_link }.to be_denied_for(:reporter).of(project) }
    it { expect { create_merge_request_link }.to be_denied_for(:guest).of(project) }
    it { expect { create_merge_request_link }.to be_denied_for(:anonymous) }
  end
end
