# frozen_string_literal: true

require 'spec_helper'

RSpec.describe GitlabSchema.types['VulnerabilityLocation'] do
  let(:vulnerability) { create(:vulnerability, report_type: report_type) }

  it 'returns all possible types' do
    expect(described_class.possible_types).to include(
      Types::VulnerabilityLocation::ClusterImageScanningType,
      Types::VulnerabilityLocation::ContainerScanningType,
      Types::VulnerabilityLocation::DependencyScanningType,
      Types::VulnerabilityLocation::DastType,
      Types::VulnerabilityLocation::SastType,
      Types::VulnerabilityLocation::SecretDetectionType,
      Types::VulnerabilityLocation::CoverageFuzzingType,
      Types::VulnerabilityLocation::GenericType
    )
  end

  describe '#resolve_type' do
    using RSpec::Parameterized::TableSyntax

    where(:report_type, :expected_type) do
      :cluster_image_scanning | Types::VulnerabilityLocation::ClusterImageScanningType
      :container_scanning     | Types::VulnerabilityLocation::ContainerScanningType
      :dependency_scanning    | Types::VulnerabilityLocation::DependencyScanningType
      :dast                   | Types::VulnerabilityLocation::DastType
      :api_fuzzing            | Types::VulnerabilityLocation::DastType
      :sast                   | Types::VulnerabilityLocation::SastType
      :secret_detection       | Types::VulnerabilityLocation::SecretDetectionType
      :coverage_fuzzing       | Types::VulnerabilityLocation::CoverageFuzzingType
      :generic                | Types::VulnerabilityLocation::GenericType
    end

    subject do
      described_class.resolve_type(vulnerability, {})
    end

    with_them do
      specify { expect(subject).to eq(expected_type) }
    end

    context 'when report_type is unknown' do
      # We have to mock this one since :report_type is an enum on the Vulnerability model.
      # Trying to create a vulnerability an invalid report_type causes an ArgumentError.
      let(:vulnerability) { { report_type: :unknown_type } }

      it 'raises an error' do
        expect { subject }.to raise_error(Types::VulnerabilityLocationType::UnexpectedReportType)
      end
    end
  end
end
