# frozen_string_literal: true

require 'spec_helper'

RSpec.describe GitlabSchema.types['VulnerabilityLocationCoverageFuzzing'] do
  let_it_be(:project) { create(:project) }
  let_it_be(:user) { create(:user) }
  let_it_be(:pipeline) { create(:ee_ci_pipeline, :with_coverage_fuzzing_report, project: project) }

  before do
    stub_licensed_features(coverage_fuzzing: true, security_dashboard: true)

    project.add_developer(user)
  end

  it do
    expect(described_class).to have_graphql_fields(
      :vulnerable_class,
      :end_line,
      :file,
      :vulnerable_method,
      :start_line,
      :blob_path,
      :crash_type,
      :crash_address,
      :stacktrace_snippet
    )
  end

  subject { GitlabSchema.execute(query, context: { current_user: user }).as_json }

  specify { expect(described_class.graphql_name).to eq('VulnerabilityLocationCoverageFuzzing') }

  describe 'checking field contents' do
    let(:query) do
      %(
        query {
          project(fullPath: "#{project.full_path}") {
            pipeline(iid: "#{pipeline.iid}") {
              securityReportFindings {
                nodes {
                  location {
                    ... on VulnerabilityLocationCoverageFuzzing {
                      crashType
                      crashAddress
                      stacktraceSnippet
                    }
                  }
                }
              }
            }
          }
        }
      )
    end

    it 'checks the contents of the VulnerabilityLocationCoverageFuzzing fields' do
      vulnerability = subject.dig('data', 'project', 'pipeline', 'securityReportFindings', 'nodes').first

      expect(vulnerability['location']['crashType']).to eq("Heap-buffer-overflow\nREAD 1")
      expect(vulnerability['location']['crashAddress']).to eq("0x602000001573")
      expect(vulnerability['location']['stacktraceSnippet']).to eq("INFO: Seed: 3415817494\nINFO: Loaded 1 modules")
    end
  end
end
