# frozen_string_literal: true

require 'spec_helper'

RSpec.describe Security::VulnerabilityFeedbacksFinder do
  let_it_be(:group)   { create(:group) }
  let_it_be(:project) { create(:project, :public, namespace: group) }

  let_it_be(:vuln_feedback_1) { create(:vulnerability_feedback, :dismissal, :sast, project: project ) }
  let_it_be(:vuln_feedback_2) { create(:vulnerability_feedback, :issue, :sast, project: project) }
  let_it_be(:vuln_feedback_3) { create(:vulnerability_feedback, :dismissal, :dast, project: project) }
  let_it_be(:vuln_feedback_4) { create(:vulnerability_feedback, :dismissal, :dependency_scanning, project: project) }
  let_it_be(:vuln_feedback_5) do
    create(:vulnerability_feedback, :merge_request, :dependency_scanning, project: project)
  end

  let_it_be(:filters) { {} }
  let_it_be(:vulnerable) { project }

  subject { described_class.new(vulnerable, filters).execute }

  it 'returns vulnerability feedbacks of a project' do
    expect(subject).to match_array(project.vulnerability_feedback)
  end

  context 'when not given a parameter' do
    subject { described_class.new(project).execute }

    it 'does not filter the vulnerability feedbacks' do
      expect(subject).to match_array(project.vulnerability_feedback)
    end
  end

  context 'when filtered by feedback type' do
    let(:filters) { { feedback_type: %w[dismissal] } }

    it 'only returns vulnerability feedbacks matching the given feedback types' do
      is_expected.to contain_exactly(vuln_feedback_1, vuln_feedback_3, vuln_feedback_4)
    end
  end

  context 'when filtered by category' do
    let(:filters) { { category: %w[sast dependency_scanning] } }

    it 'only returns vulnerability feedbacks matching the given categories' do
      is_expected.to contain_exactly(vuln_feedback_1, vuln_feedback_2, vuln_feedback_4, vuln_feedback_5)
    end
  end

  context 'when filtered by more than one property' do
    let(:filters) { { feedback_type: %w[dismissal], category: %w[dependency_scanning] } }

    it 'only returns vulnerabilities matching all of the given filters' do
      is_expected.to contain_exactly(vuln_feedback_4)
    end
  end
end
