# frozen_string_literal: true

require 'spec_helper'

require_migration!

RSpec.describe AddUpdateVulnerabilityReadsLocationTrigger, feature_category: :vulnerability_management do
  let(:migration) { described_class.new }
  let(:vulnerability_reads) { table(:vulnerability_reads) }
  let(:issue_links) { table(:vulnerability_issue_links) }
  let(:vulnerabilities) { table(:vulnerabilities) }
  let(:vulnerabilities_findings) { table(:vulnerability_occurrences) }

  let(:namespace) { table(:namespaces).create!(name: 'user', path: 'user') }
  let(:user) { table(:users).create!(id: 13, email: 'author@example.com', username: 'author', projects_limit: 10) }
  let(:project) { table(:projects).create!(id: 123, namespace_id: namespace.id) }
  let(:issue) { table(:issues).create!(description: '1234', state_id: 1, project_id: project.id) }
  let(:scanner) { table(:vulnerability_scanners).create!(project_id: project.id, external_id: 'test 1', name: 'test scanner 1') }

  let(:vulnerability) do
    create_vulnerability!(
      project_id: project.id,
      report_type: 7,
      author_id: user.id
    )
  end

  let(:identifier) do
    table(:vulnerability_identifiers).create!(
      project_id: project.id,
      external_type: 'uuid-v5',
      external_id: 'uuid-v5',
      fingerprint: '7e394d1b1eb461a7406d7b1e08f057a1cf11287a',
      name: 'Identifier for UUIDv5')
  end

  describe '#up' do
    before do
      migrate!
    end

    describe 'UPDATE trigger' do
      context 'when image is updated' do
        it 'updates location_image in vulnerability_reads' do
          finding = create_finding!(
            vulnerability_id: vulnerability.id,
            project_id: project.id,
            scanner_id: scanner.id,
            report_type: 7,
            location: { "image" => "alpine:3.4" },
            primary_identifier_id: identifier.id
          )

          expect do
            finding.update!(location: { "image" => "alpine:4", "kubernetes_resource" => { "agent_id" => "1234" } })
          end.to change { vulnerability_reads.first.location_image }.from("alpine:3.4").to("alpine:4")
        end
      end

      context 'when image is not updated' do
        it 'updates location_image in vulnerability_reads' do
          finding = create_finding!(
            vulnerability_id: vulnerability.id,
            project_id: project.id,
            scanner_id: scanner.id,
            report_type: 7,
            location: { "image" => "alpine:3.4", "kubernetes_resource" => { "agent_id" => "1234" } },
            primary_identifier_id: identifier.id
          )

          expect do
            finding.update!(project_fingerprint: "123qweasdzx")
          end.not_to change { vulnerability_reads.first.location_image }
        end
      end
    end
  end

  describe '#down' do
    before do
      migration.up
      migration.down
    end

    it 'drops the trigger' do
      finding = create_finding!(
        vulnerability_id: vulnerability.id,
        project_id: project.id,
        scanner_id: scanner.id,
        primary_identifier_id: identifier.id
      )

      expect do
        finding.update!(location: '{"image":"alpine:4"}')
      end.not_to change { vulnerability_reads.first.location_image }
    end
  end

  private

  def create_vulnerability!(project_id:, author_id:, title: 'test', severity: 7, confidence: 7, report_type: 0)
    vulnerabilities.create!(
      project_id: project_id,
      author_id: author_id,
      title: title,
      severity: severity,
      confidence: confidence,
      report_type: report_type
    )
  end

  # rubocop:disable Metrics/ParameterLists
  def create_finding!(
    project_id:, scanner_id:, primary_identifier_id:, vulnerability_id: nil,
    name: "test", severity: 7, confidence: 7, report_type: 0,
    project_fingerprint: '123qweasdzxc', location: { "image" => "alpine:3.4" }, location_fingerprint: 'test',
    metadata_version: 'test', raw_metadata: 'test', uuid: SecureRandom.uuid)
    vulnerabilities_findings.create!(
      vulnerability_id: vulnerability_id,
      project_id: project_id,
      name: name,
      severity: severity,
      confidence: confidence,
      report_type: report_type,
      project_fingerprint: project_fingerprint,
      scanner_id: scanner_id,
      primary_identifier_id: primary_identifier_id,
      location: location,
      location_fingerprint: location_fingerprint,
      metadata_version: metadata_version,
      raw_metadata: raw_metadata,
      uuid: uuid
    )
  end
  # rubocop:enable Metrics/ParameterLists
end
