import { shallowMount, mount } from '@vue/test-utils';
import { nextTick } from 'vue';
import BlobHeader from '~/blob/components/blob_header.vue';
import DefaultActions from '~/blob/components/blob_header_default_actions.vue';
import BlobFilepath from '~/blob/components/blob_header_filepath.vue';
import ViewerSwitcher from '~/blob/components/blob_header_viewer_switcher.vue';
import TableContents from '~/blob/components/table_contents.vue';

import { Blob } from './mock_data';

describe('Blob Header Default Actions', () => {
  let wrapper;

  function createComponent(blobProps = {}, options = {}, propsData = {}, shouldMount = false) {
    const method = shouldMount ? mount : shallowMount;
    const blobHash = 'foo-bar';
    wrapper = method.call(this, BlobHeader, {
      provide: {
        blobHash,
      },
      propsData: {
        blob: { ...Blob, ...blobProps },
        ...propsData,
      },
      ...options,
    });
  }

  afterEach(() => {
    wrapper.destroy();
  });

  describe('rendering', () => {
    const findDefaultActions = () => wrapper.findComponent(DefaultActions);

    const slots = {
      prepend: 'Foo Prepend',
      actions: 'Actions Bar',
    };

    it('matches the snapshot', () => {
      createComponent();
      expect(wrapper.element).toMatchSnapshot();
    });

    it('renders all components', () => {
      createComponent();
      expect(wrapper.findComponent(TableContents).exists()).toBe(true);
      expect(wrapper.findComponent(ViewerSwitcher).exists()).toBe(true);
      expect(findDefaultActions().exists()).toBe(true);
      expect(wrapper.findComponent(BlobFilepath).exists()).toBe(true);
    });

    it('does not render viewer switcher if the blob has only the simple viewer', () => {
      createComponent({
        richViewer: null,
      });
      expect(wrapper.findComponent(ViewerSwitcher).exists()).toBe(false);
    });

    it('does not render viewer switcher if a corresponding prop is passed', () => {
      createComponent(
        {},
        {},
        {
          hideViewerSwitcher: true,
        },
      );
      expect(wrapper.findComponent(ViewerSwitcher).exists()).toBe(false);
    });

    it('does not render default actions is corresponding prop is passed', () => {
      createComponent(
        {},
        {},
        {
          hideDefaultActions: true,
        },
      );
      expect(wrapper.findComponent(DefaultActions).exists()).toBe(false);
    });

    Object.keys(slots).forEach((slot) => {
      it('renders the slots', () => {
        const slotContent = slots[slot];
        createComponent(
          {},
          {
            scopedSlots: {
              [slot]: `<span>${slotContent}</span>`,
            },
          },
          {},
          true,
        );
        expect(wrapper.text()).toContain(slotContent);
      });
    });

    it('passes information about render error down to default actions', () => {
      createComponent(
        {},
        {},
        {
          hasRenderError: true,
        },
      );
      expect(findDefaultActions().props('hasRenderError')).toBe(true);
    });

    it('passes the correct isBinary value to default actions when viewing a binary file', () => {
      createComponent({}, {}, { isBinary: true });

      expect(findDefaultActions().props('isBinary')).toBe(true);
    });
  });

  describe('functionality', () => {
    const newViewer = 'Foo Bar';
    const activeViewerType = 'Alpha Beta';

    const factory = (hideViewerSwitcher = false) => {
      createComponent(
        {},
        {},
        {
          activeViewerType,
          hideViewerSwitcher,
        },
      );
    };

    it('by default sets viewer data based on activeViewerType', () => {
      factory();
      expect(wrapper.vm.viewer).toBe(activeViewerType);
    });

    it('sets viewer to null if the viewer switcher should be hidden', () => {
      factory(true);
      expect(wrapper.vm.viewer).toBe(null);
    });

    it('watches the changes in viewer data and emits event when the change is registered', async () => {
      factory();
      jest.spyOn(wrapper.vm, '$emit');
      wrapper.vm.viewer = newViewer;

      await nextTick();
      expect(wrapper.vm.$emit).toHaveBeenCalledWith('viewer-changed', newViewer);
    });

    it('does not emit event if the switcher is not rendered', async () => {
      factory(true);

      expect(wrapper.vm.showViewerSwitcher).toBe(false);
      jest.spyOn(wrapper.vm, '$emit');
      wrapper.vm.viewer = newViewer;

      await nextTick();
      expect(wrapper.vm.$emit).not.toHaveBeenCalled();
    });

    it('sets different icons depending on the blob file type', async () => {
      factory();
      expect(wrapper.vm.blobSwitcherDocIcon).toBe('document');
      await wrapper.setProps({
        blob: {
          ...Blob,
          richViewer: {
            ...Blob.richViewer,
            fileType: 'csv',
          },
        },
      });
      expect(wrapper.vm.blobSwitcherDocIcon).toBe('table');
    });
  });
});
