import { GlDropdown } from '@gitlab/ui';
import { nextTick } from 'vue';
import { shallowMountExtended } from 'helpers/vue_test_utils_helper';
import VulnerabilityStateDropdown from 'ee/vulnerabilities/components/vulnerability_state_dropdown.vue';
import { VULNERABILITY_STATE_OBJECTS } from 'ee/vulnerabilities/constants';

const vulnerabilityStateObjects = Object.values(VULNERABILITY_STATE_OBJECTS);

describe('Vulnerability state dropdown component', () => {
  let wrapper;

  const createWrapper = (initialState = vulnerabilityStateObjects[0].state) => {
    // Create a dropdown that by default has the first vulnerability state selected.
    wrapper = shallowMountExtended(VulnerabilityStateDropdown, {
      propsData: { initialState },
      stubs: { GlDropdown },
    });

    // Mock out this function, it's called by some methods in the component.
    wrapper.vm.$refs.dropdown.hide = jest.fn();
  };

  const isSelected = (item) => item.find('.selected-icon').exists(); // Item is selected if there's a checkmark icon.
  const isDisabled = (item) => item.attributes('disabled') === 'true';
  const dropdownItems = () => wrapper.findAll('.dropdown-item');
  const firstUnselectedItem = () => wrapper.find('.dropdown-item:not(.selected)');
  const selectedItem = () => wrapper.find('.dropdown-item.selected');
  const saveButton = () => wrapper.findComponent({ ref: 'save-button' });
  const cancelButton = () => wrapper.findComponent({ ref: 'cancel-button' });
  const innerDropdown = () => wrapper.findComponent({ ref: 'dropdown' });
  const dropdownItemFor = (state) => wrapper.findByTestId(state);

  afterEach(() => wrapper.destroy());

  describe('tests that need to manually create the wrapper', () => {
    it.each(vulnerabilityStateObjects)(
      'dropdown is created with the %s state already selected',
      ({ state }) => {
        createWrapper(state);

        expect(isSelected(dropdownItemFor(state))).toBe(true); // Check that the dropdown item is selected.
      },
    );

    it('if an unknown state is passed in, nothing will be selected by default', () => {
      createWrapper('some unknown state');

      dropdownItems().wrappers.forEach((dropdownItem) => {
        expect(isSelected(dropdownItem)).toBe(false);
      });
    });

    it.each(vulnerabilityStateObjects)(
      `when the %s dropdown item is clicked, it's the only one that's selected`,
      async ({ state }) => {
        // Start off with an unknown state so we can click through each item and see it change.
        createWrapper('some unknown state');
        const dropdownItem = dropdownItemFor(state);

        dropdownItem.trigger('click');

        await nextTick();
        dropdownItems().wrappers.forEach((item) => {
          expect(isSelected(item)).toBe(item.attributes('data-testid') === state);
        });
      },
    );
  });

  describe('tests that use the default wrapper', () => {
    beforeEach(() => createWrapper());

    it('the save button should be enabled/disabled based on if the selected item has changed or not', async () => {
      const originalItem = selectedItem();

      expect(isDisabled(saveButton())).toBe(true); // Check that the save button starts off as disabled.
      firstUnselectedItem().trigger('click'); // Click on an unselected item.

      await nextTick();
      expect(isDisabled(saveButton())).toBe(false); // Check that the save button has been enabled.
      originalItem.trigger('click'); // Re-select the original item.

      await nextTick();
      expect(isDisabled(saveButton())).toBe(true); // Check that the save button has once again been disabled.
    });

    it('clicking on the save button will close the dropdown and fire a change event', async () => {
      createWrapper();

      expect(isDisabled(saveButton())).toBe(true); // Check that the save button starts off disabled.
      firstUnselectedItem().trigger('click'); // Click on an unselected item.

      await nextTick();
      saveButton().vm.$emit('click'); // Click on the save button.
      const changeEvent = wrapper.emitted('change');
      expect(wrapper.vm.$refs.dropdown.hide).toHaveBeenCalledTimes(1); // Check that the dropdown hide function was called.
      expect(changeEvent).toHaveLength(1); // Check that a change event was emitted.
      expect(changeEvent[0][0]).toEqual(expect.any(Object)); // Check that the change event has been emitted with something as its first parameter.
    });

    it('clicking on the cancel button will close the dropdown without emitting any events', async () => {
      expect(isDisabled(saveButton())).toBe(true); // Check that the save button starts out disabled.
      firstUnselectedItem().trigger('click'); // Click on an unselected item.

      await nextTick();
      expect(isDisabled(saveButton())).toBe(false); // Check that the save button is enabled.
      cancelButton().vm.$emit('click'); // Click on the cancel button.
      expect(Object.keys(wrapper.emitted())).toHaveLength(0); // Check that no events have been emitted.
      expect(wrapper.vm.$refs.dropdown.hide).toHaveBeenCalledTimes(1); // Check that hide was called on the inner dropdown.
    });

    it('when the dropdown is closed, the selected item resets back to the initial item', async () => {
      const initialSelectedItem = selectedItem();

      firstUnselectedItem().trigger('click'); // Click on an unselected item.

      await nextTick();
      expect(selectedItem().element).not.toBe(initialSelectedItem.element); // Check that the selected item actually changed.
      innerDropdown().vm.$emit('hide'); // Emit the dropdown hide event.

      await nextTick();
      expect(selectedItem().element).toBe(initialSelectedItem.element); // Check that the selected item has been reset back to the initial item.
    });

    it('when the parent component changes the state, the dropdown will update its selected and initial item', async () => {
      const stateObject = vulnerabilityStateObjects[1];

      wrapper.setProps({ initialState: stateObject.state }); // Change the state.

      await nextTick();
      expect(innerDropdown().props('text')).toBe(stateObject.buttonText); // Check that the dropdown button's value matches the initial state.
      expect(selectedItem().text()).toMatch(stateObject.dropdownText); // Check that the selected item is the initial state.
      expect(isDisabled(saveButton())).toBe(true); // Check that the save button is disabled.
    });
  });
});
