# frozen_string_literal: true

require 'spec_helper'

describe Gitlab::Vulnerabilities::ContainerScanningVulnerability do
  let(:data) do
    {
      featurename: 'foo',
      featureversion: '1.2.3',
      vulnerability: 'CVE-2018-777',
      namespace: 'debian:9',
      link: 'https://security-tracker.debian.org/tracker/CVE-2018-777',
      severity: 'high',
      fixedby: '1.4',
      confidence: 'low',
      identifiers: '42',
      links: 'link'
    }
  end

  describe 'getters' do
    let(:vulnerability) { described_class.new(data) }

    where(:getter) do
      %i[severity confidence identifiers links]
    end

    with_them do
      it 'returns right value' do
        expect(vulnerability.public_send(getter)).to eq(data[getter])
      end
    end
  end

  describe '#title' do
    subject { described_class.new(data).title }

    context 'when there is a name' do
      before do
        data[:name] = 'Foo is affected by CVE-2018-777'
      end
      it 'returns the provided name' do
        is_expected.to eq('Foo is affected by CVE-2018-777')
      end
    end

    context 'when there is no featurename' do
      before do
        data[:featurename] = ''
      end

      it 'formats title using the vulnerability only' do
        is_expected.to eq('CVE-2018-777')
      end
    end

    context 'when there is a featurename' do
      it 'formats title using the featurename' do
        is_expected.to eq('CVE-2018-777 in foo')
      end
    end
  end

  describe '#description' do
    subject { described_class.new(data).description }

    context 'when there is a description' do
      before do
        data[:description] = 'SSE2-optimized memmove implementation problem.'
      end

      it 'returns the provided description' do
        is_expected.to eq('SSE2-optimized memmove implementation problem.')
      end
    end
    context 'when there is no featurename' do
      before do
        data[:featurename] = ''
      end

      it 'formats description using the namespace' do
        is_expected.to eq('**debian:9** is affected by CVE-2018-777')
      end
    end

    context 'when there is no featureversion' do
      before do
        data[:featureversion] = ''
      end

      it 'formats description using the featurename only' do
        is_expected.to eq('**foo** is affected by CVE-2018-777')
      end
    end

    context 'when featurename and featureversion are present' do
      it 'formats description using the featurename and featureversion' do
        is_expected.to eq('**foo** `1.2.3` is affected by CVE-2018-777')
      end
    end
  end

  describe '#solution' do
    subject { described_class.new(data).solution }

    context 'when there is a solution' do
      before do
        data[:solution] = 'Do something about it!'
      end

      it 'returns the provided solution' do
        is_expected.to eq('Do something about it!')
      end
    end

    context 'when there is no fixedby' do
      before do
        data[:fixedby] = ''
      end

      it 'returns nil' do
        is_expected.to be_nil
      end
    end

    context 'when there is no featurename' do
      before do
        data[:featurename] = ''
      end

      it 'formats solution using the fixedby only' do
        is_expected.to eq('Upgrade to `1.4`')
      end
    end

    context 'when there is no featureversion' do
      before do
        data[:featureversion] = ''
      end

      it 'formats solution using the featurename only' do
        is_expected.to eq('Upgrade **foo** to `1.4`')
      end
    end

    context 'when featurename and featureversion are present' do
      it 'formats solution using the featurename and featureversion' do
        is_expected.to eq('Upgrade **foo** from `1.2.3` to `1.4`')
      end
    end
  end
end
