import Vue from 'vue';
import component from 'ee/security_dashboard/components/vulnerability_action_buttons.vue';

import createStore from 'ee/security_dashboard/store';
import { mountComponentWithStore } from 'spec/helpers/vue_mount_component_helper';
import { resetStore } from '../helpers';
import mockDataVulnerabilities from '../store/vulnerabilities/data/mock_data_vulnerabilities.json';

describe('Security Dashboard Action Buttons', () => {
  const Component = Vue.extend(component);
  let vm;
  let store;
  let props;

  beforeEach(() => {
    store = createStore();
  });

  afterEach(() => {
    vm.$destroy();
    resetStore(store);
  });

  describe('with a fresh vulnerability', () => {
    beforeEach(() => {
      props = {
        vulnerability: mockDataVulnerabilities[0],
        canCreateIssue: true,
        canDismissVulnerability: true,
      };
      vm = mountComponentWithStore(Component, { store, props });
      spyOn(vm.$store, 'dispatch').and.returnValue(Promise.resolve());
    });

    afterEach(() => {
      vm.$destroy();
    });

    it('should render three buttons in a button group', () => {
      expect(vm.$el.querySelectorAll('.btn-group .btn')).toHaveLength(3);
    });

    describe('More Info Button', () => {
      let button;

      beforeEach(() => {
        button = vm.$el.querySelector('.js-more-info');
      });

      it('should render the More info button', () => {
        expect(button).not.toBeNull();
      });

      it('should emit an `openModal` event when clicked', () => {
        button.click();

        expect(vm.$store.dispatch).toHaveBeenCalledWith('vulnerabilities/openModal', {
          vulnerability: mockDataVulnerabilities[0],
        });
      });
    });

    describe('Create Issue Button', () => {
      let button;

      beforeEach(() => {
        button = vm.$el.querySelector('.js-create-issue');
      });

      it('should render the create issue button', () => {
        expect(button).not.toBeNull();
      });

      it('should emit an `createIssue` event when clicked', () => {
        button.click();

        expect(vm.$store.dispatch).toHaveBeenCalledWith('vulnerabilities/createIssue', {
          vulnerability: mockDataVulnerabilities[0],
          flashError: true,
        });
      });
    });

    describe('Dismiss Vulnerability Button', () => {
      let button;

      beforeEach(() => {
        button = vm.$el.querySelector('.js-dismiss-vulnerability');
      });

      it('should render the dismiss vulnerability button', () => {
        expect(button).not.toBeNull();
      });

      it('should emit an `dismissVulnerability` event when clicked', () => {
        button.click();

        expect(vm.$store.dispatch).toHaveBeenCalledWith('vulnerabilities/dismissVulnerability', {
          vulnerability: mockDataVulnerabilities[0],
          flashError: true,
        });
      });
    });
  });

  describe('with a vulnerbility that has an issue', () => {
    beforeEach(() => {
      props = {
        vulnerability: mockDataVulnerabilities[3],
      };
      vm = mountComponentWithStore(Component, { store, props });
    });

    afterEach(() => {
      vm.$destroy();
    });

    it('should only render one button', () => {
      expect(vm.$el.querySelectorAll('.btn')).toHaveLength(1);
    });

    it('should not render the create issue button', () => {
      expect(vm.$el.querySelector('.js-create-issue')).toBeNull();
    });
  });

  describe('with a vulnerability that has been dismissed', () => {
    beforeEach(() => {
      props = {
        vulnerability: mockDataVulnerabilities[2],
        canDismissVulnerability: true,
        isDismissed: true,
      };
      vm = mountComponentWithStore(Component, { store, props });
    });

    afterEach(() => {
      vm.$destroy();
    });

    it('should render two buttons in a button group', () => {
      expect(vm.$el.querySelectorAll('.btn-group .btn')).toHaveLength(2);
    });

    it('should not render the dismiss vulnerability button', () => {
      expect(vm.$el.querySelector('.js-dismiss-vulnerability')).toBeNull();
    });

    it('should render the undo dismiss button', () => {
      expect(vm.$el.querySelector('.js-undo-dismiss')).not.toBeNull();
    });
  });
});
