# frozen_string_literal: true

require "spec_helper"

describe EE::RunnersHelper do
  describe '.ci_usage_warning_message' do
    let(:project) { create(:project, namespace: namespace) }
    let(:minutes_used) { 0 }
    let(:user) { create(:user) }

    let(:namespace) do
      create(:group, shared_runners_minutes_limit: 100)
    end

    let!(:statistics) do
      create(:namespace_statistics, namespace: namespace, shared_runners_seconds: minutes_used * 60)
    end

    before do
      allow(::Gitlab).to receive(:com?).and_return(true)
      allow(helper).to receive(:current_user).and_return(user)
      allow(helper).to receive(:can?).with(user, :admin_project, project) { false }

      stub_const("EE::Namespace::CI_USAGE_ALERT_LEVELS", [50])
    end

    subject { helper.ci_usage_warning_message(namespace, project) }

    context 'when CI minutes quota is above the warning limits' do
      let(:minutes_used) { 40 }

      it 'does not return a message' do
        expect(subject).to be_nil
      end
    end

    context 'when current user is an owner' do
      before do
        allow(helper).to receive(:can?).with(user, :admin_project, project) { true }
      end

      context 'when usage has reached first level of notification' do
        before do
          namespace.update_attribute(:last_ci_minutes_usage_notification_level, 50)
        end

        it 'shows the partial usage message' do
          expect(subject).to match("#{namespace.name} has less than 50% of CI minutes available.")
          expect(subject).to match('to purchase more minutes')
        end
      end

      context 'when usage is above the quota' do
        let(:minutes_used) { 120 }

        it 'shows the total usage message' do
          expect(subject).to match("#{namespace.name} has exceeded its pipeline minutes quota.")
          expect(subject).to match('to purchase more minutes')
        end
      end
    end

    context 'when current user is not an owner' do
      context 'when usage has reached first level of notification' do
        before do
          namespace.update_attribute(:last_ci_minutes_usage_notification_level, 50)
        end

        it 'shows the partial usage message without the purchase link' do
          expect(subject).to match("#{namespace.name} has less than 50% of CI minutes available.")
          expect(subject).not_to match('to purchase more minutes')
        end
      end

      context 'when usage is above the quota' do
        let(:minutes_used) { 120 }

        it 'shows the total usage message without the purchase link' do
          expect(subject).to match("#{namespace.name} has exceeded its pipeline minutes quota.")
          expect(subject).not_to match('to purchase more minutes')
        end
      end
    end
  end
end
