class Projects::VulnerabilityFeedbackController < Projects::ApplicationController
  before_action :vulnerability_feedback, only: [:destroy]
  before_action :authorize_read_vulnerability_feedback!, only: [:index]
  before_action :authorize_admin_vulnerability_feedback!, only: [:create, :destroy]
  skip_before_action :authenticate_user!, only: [:index], raise: false

  respond_to :json

  def index
    # TODO: Move to finder or list service
    @vulnerability_feedback = @project.vulnerability_feedback.with_associations
    if params[:category].present?
      @vulnerability_feedback = @vulnerability_feedback
        .where(category: VulnerabilityFeedback.categories[params[:category]])
    end

    if params[:feedback_type].present?
      @vulnerability_feedback = @vulnerability_feedback
        .where(feedback_type: VulnerabilityFeedback.feedback_types[params[:feedback_type]])
    end

    render json: serializer.represent(@vulnerability_feedback)
  end

  def create
    service = VulnerabilityFeedbackModule::CreateService.new(project, current_user, vulnerability_feedback_params)
    result = service.execute

    if result[:status] == :success
      render json: serializer.represent(result[:vulnerability_feedback])
    else
      render json: result[:message], status: :unprocessable_entity
    end
  end

  def destroy
    service = VulnerabilityFeedbackModule::DestroyService.new(@vulnerability_feedback)
    service.execute

    head :no_content
  end

  private

  def authorize_admin_vulnerability_feedback!
    render_403 unless can?(current_user, :admin_vulnerability_feedback, project)
  end

  def serializer
    VulnerabilityFeedbackSerializer.new(current_user: current_user, project: project)
  end

  def vulnerability_feedback
    @vulnerability_feedback ||= @project.vulnerability_feedback.find(params[:id])
  end

  def vulnerability_feedback_params
    params.require(:vulnerability_feedback).permit(*vulnerability_feedback_params_attributes)
  end

  def vulnerability_feedback_params_attributes
    %i[
      category
      feedback_type
      pipeline_id
      project_fingerprint
      ] + [
        vulnerability_data: vulnerability_data_params_attributes
      ]
  end

  def vulnerability_data_params_attributes
    %i[
      category
      confidence
      count
      cve
      cweid
      desc
      description
      featurename
      featureversion
      file
      fingerprint
      fixedby
      line
      link
      message
      name
      namespace
      otherinfo
      pluginid
      priority
      project_fingerprint
      reference
      riskcode
      riskdesc
      severity
      solution
      sourceid
      title
      tool
      tools
      url
      wascid
    ] + [
      instances: %i[
        param
        method
        uri
      ],
      location: %i[
        file
        start_line
        end_line
        class
        method
      ],
      identifiers: %i[
        type
        name
        value
        url
      ],
      links: %i[
        name
        url
      ]
    ]
  end
end
