# Geo database replication (GitLab Omnibus)

NOTE: **Note:**
This is the documentation for the Omnibus GitLab packages. For installations
from source, follow the
[**database replication for installations from source**][database-source] guide.

NOTE: **Note:**
If your GitLab installation uses external PostgreSQL, the Omnibus roles
will not be able to perform all necessary configuration steps. Refer to the
section on [External PostreSQL][external postgresql] for additional instructions.

NOTE: **Note:**
The stages of the setup process must be completed in the documented order.
Before attempting the steps in this stage, [complete all prior stages][toc].

This document describes the minimal steps you have to take in order to
replicate your primary GitLab database to a secondary node's database. You may
have to change some values according to your database setup, how big it is, etc.

You are encouraged to first read through all the steps before executing them
in your testing/production environment.


## PostgreSQL replication

The GitLab primary node where the write operations happen will connect to
the primary database server, and the secondary nodes which are read-only will
connect to the secondary database servers (which are also read-only).

NOTE: **Note:**
In database documentation you may see "primary" being referenced as "master"
and "secondary" as either "slave" or "standby" server (read-only).

We recommend using [PostgreSQL replication slots][replication-slots-article]
to ensure that the primary retains all the data necessary for the secondaries to
recover. See below for more details.

The following guide assumes that:

- You are using Omnibus and therefore you are using PostgreSQL 9.6 or later
  which includes the  [`pg_basebackup` tool][pgback] and improved
  [Foreign Data Wrapper][FDW] support.
- You have a primary node already set up (the GitLab server you are
  replicating from), running Omnibus' PostgreSQL (or equivalent version), and
  you have a new secondary server set up with the same versions of the OS,
  PostgreSQL, and GitLab on all nodes.
- The IP of the primary server for our examples will be `1.2.3.4`, whereas the
  secondary's IP will be `5.6.7.8`. Note that the primary and secondary servers
  **must** be able to communicate over these addresses. More on this in the
  guide below.


### Step 1. Configure the primary server

1. SSH into your GitLab **primary** server and login as root:

    ```bash
    sudo -i
    ```

1. Execute the command below to define the node as primary Geo node:

    ```bash
    gitlab-ctl set-geo-primary-node
    ```

    This command will use your defined `external_url` in `/etc/gitlab/gitlab.rb`.

1. GitLab 10.4 and up only: Do the following to make sure the `gitlab` database user has a password defined

    Generate a MD5 hash of the desired password:

    ```bash
    gitlab-ctl pg-password-md5 gitlab
    # Enter password: mypassword
    # Confirm password: mypassword
    # fca0b89a972d69f00eb3ec98a5838484
    ```

    Edit `/etc/gitlab/gitlab.rb`:

    ```ruby
    # Fill with the hash generated by `gitlab-ctl pg-password-md5 gitlab`
    postgresql['sql_user_password'] = 'fca0b89a972d69f00eb3ec98a5838484'

    # Every node that runs Unicorn or Sidekiq needs to have the database
    # password specified as below. If you have a high-availability setup, this
    # must be present in all application nodes.
    gitlab_rails['db_password'] = 'mypassword'
    ```

1. Omnibus GitLab already has a [replication user]
   called `gitlab_replicator`. You must set the password for this user manually.
   You will be prompted to enter a password:

    ```bash
    gitlab-ctl set-replication-password
    ```

    This command will also read the `postgresql['sql_replication_user']` Omnibus
    setting in case you have changed `gitlab_replicator` username to something
    else.
    
    If you are using an external database not managed by Omnibus GitLab, you need
    to create the replicator user and define a password to it manually.
    Check [How to create replication user][database-source-primary] documentation. 

1. Configure PostgreSQL to listen on network interfaces

    For security reasons, PostgreSQL does not listen on any network interfaces
    by default. However, Geo requires the secondary to be able to
    connect to the primary's database. For this reason, we need the address of
    each node. Note: For external PostgreSQL instances, see [additional instructions][external postgresql].

    If you are using a cloud provider, you can lookup the addresses for each
    Geo node through your cloud provider's management console.

    To lookup the address of a Geo node, SSH in to the Geo node and execute:

    ```bash
    ##
    ## Private address
    ##
    ip route get 255.255.255.255 | awk '{print "Private address:", $NF; exit}'

    ##
    ## Public address
    ##
    echo "External address: $(curl --silent ipinfo.io/ip)"
    ```

    In most cases, the following addresses will be used to configure GitLab
    Geo:

    | Configuration                           | Address                                     |
    |-----------------------------------------|---------------------------------------------|
    | `postgresql['listen_address']`          | Primary's public or VPC private address     |
    | `postgresql['md5_auth_cidr_addresses']` | Secondary's public or VPC private addresses |

    If you are using Google Cloud Platform, SoftLayer, or any other vendor that
    provides a virtual private cloud (VPC) you can use the secondary's private
    address (corresponds to "internal address" for Google Cloud Platform) for
    `postgresql['md5_auth_cidr_addresses']` and `postgresql['listen_address']`.

    The `listen_address` option opens PostgreSQL up to network connections
    with the interface corresponding to the given address. See [the PostgreSQL
    documentation][pg-docs-runtime-conn] for more details.

    Depending on your network configuration, the suggested addresses may not
    be correct. If your primary and secondary connect over a local
    area network, or a virtual network connecting availability zones like
    [Amazon's VPC](https://aws.amazon.com/vpc/) or [Google's VPC](https://cloud.google.com/vpc/)
    you should use the secondary's private address for `postgresql['md5_auth_cidr_addresses']`.

    Edit `/etc/gitlab/gitlab.rb` and add the following, replacing the IP
    addresses with addresses appropriate to your network configuration:

    ```ruby
    ##
    ## Geo Primary role
    ## - configure dependent flags automatically to enable Geo
    ##
    geo_primary_role['enable'] = true

    ##
    ## Primary address
    ## - replace '1.2.3.4' with the primary public or VPC address
    ##
    postgresql['listen_address'] = '1.2.3.4'
    
    ##
    # Primary and Secondary addresses
    # - replace '1.2.3.4' with the primary public or VPC address
    # - replace '5.6.7.8' with the secondary public or VPC address
    ##
    postgresql['md5_auth_cidr_addresses'] = ['1.2.3.4/32','5.6.7.8/32']

    ##
    ## Replication settings
    ## - set this to be the number of Geo secondary nodes you have
    ##
    postgresql['max_replication_slots'] = 1
    # postgresql['max_wal_senders'] = 10
    # postgresql['wal_keep_segments'] = 10

    ##
    ## Disable automatic database migrations temporarily
    ## (until PostgreSQL is restarted and listening on the private address).
    ##
    gitlab_rails['auto_migrate'] = false
    ```

1. Optional: If you want to add another secondary, the relevant setting would look like:

    ```ruby
    postgresql['md5_auth_cidr_addresses'] = ['1.2.3.4/32', '5.6.7.8/32','9.10.11.12/32']
    ```

    You may also want to edit the `wal_keep_segments` and `max_wal_senders` to
    match your database replication requirements. Consult the [PostgreSQL -
    Replication documentation][pg-docs-runtime-replication]
    for more information.

1. Save the file and reconfigure GitLab for the database listen changes and
   the replication slot changes to be applied:

    ```bash
    gitlab-ctl reconfigure
    ```

    Restart PostgreSQL for its changes to take effect:

    ```bash
    gitlab-ctl restart postgresql
    ```

1. Re-enable migrations now that PostgreSQL is restarted and listening on the
   private address.

    Edit `/etc/gitlab/gitlab.rb` and **change** the configuration to `true`:

    ```ruby
    gitlab_rails['auto_migrate'] = true
    ```

    Save the file and reconfigure GitLab:

    ```bash
    gitlab-ctl reconfigure
    ```

1. Now that the PostgreSQL server is set up to accept remote connections, run
   `netstat -plnt | grep 5432` to make sure that PostgreSQL is listening on port
   `5432` to the primary server's private address.

1. A certificate was automatically generated when GitLab was reconfigured. This
   will be used automatically to protect your PostgreSQL traffic from
   eavesdroppers, but to protect against active ("man-in-the-middle") attackers,
   the secondary needs a copy of the certificate. Make a copy of the PostgreSQL
    `server.crt` file on the primary node by running this command:

    ```bash
    cat ~gitlab-psql/data/server.crt
    ```

    Copy the output into a clipboard or into a local file. You
    will need it when setting up the secondary! The certificate is not sensitive
    data.

### Step 2. Configure the secondary server

1. SSH into your GitLab **secondary** server and login as root:

    ```
    sudo -i
    ```

1. Stop application server and Sidekiq

    ```
    gitlab-ctl stop unicorn
    gitlab-ctl stop sidekiq
    ```

    NOTE: **Note**: 
    This step is important so we don't try to execute anything before the node is fully configured. 

1. [Check TCP connectivity][rake-maintenance] to the primary's PostgreSQL server:

    ```bash
    gitlab-rake gitlab:tcp_check[1.2.3.4,5432]
    ```

    NOTE: **Note**: 
    If this step fails, you may be using the wrong IP address, or a firewall may
    be preventing access to the server. Check the IP address, paying close
    attention to the difference between public and private addresses and ensure
    that, if a firewall is present, the secondary is permitted to connect to the
    primary on port 5432.

1. Create a file `server.crt` in the secondary server, with the content you got on the last step of the primary setup:

    ```
    editor server.crt
    ```


1. Set up PostgreSQL TLS verification on the secondary

    Install the `server.crt` file:

    ```bash
    install -D -o gitlab-psql -g gitlab-psql -m 0400 -T server.crt ~gitlab-psql/.postgresql/root.crt
    ```

    PostgreSQL will now only recognize that exact certificate when verifying TLS
    connections. The certificate can only be replicated by someone with access
    to the private key, which is **only** present on the primary node.

1. Test that the `gitlab-psql` user can connect to the primary's database:

    ```bash
    sudo -u gitlab-psql /opt/gitlab/embedded/bin/psql --list -U gitlab_replicator -d "dbname=gitlabhq_production sslmode=verify-ca" -W -h 1.2.3.4
    ```

    When prompted enter the password you set in the first step for the
    `gitlab_replicator` user. If all worked correctly, you should see
    the list of primary's databases.

    A failure to connect here indicates that the TLS configuration is incorrect.
    Ensure that the contents of `~gitlab-psql/data/server.crt` on the primary
    match the contents of `~gitlab-psql/.postgresql/root.crt` on the secondary.

1. Configure PostreSQL to enable FDW support

    This step is similar to how we configured the primary instance.
    We need to enable this, to enable FDW support, even if using a single node.

    Edit `/etc/gitlab/gitlab.rb` and add the following, replacing the IP
    addresses with addresses appropriate to your network configuration:

    ```ruby
    ##
    ## Geo Secondary role
    ## - configure dependent flags automatically to enable Geo
    ##
    geo_secondary_role['enable'] = true 
  
    ##
    ## Secondary address
    ## - replace '5.6.7.8' with the secondary public or VPC address
    ##
    postgresql['listen_address'] = '5.6.7.8'
    postgresql['md5_auth_cidr_addresses'] = ['5.6.7.8/32']

    ##
    ## Database credentials password (defined previously in primary node)
    ## - replicate same values here as defined in primary node 
    ##
    postgresql['sql_user_password'] = 'fca0b89a972d69f00eb3ec98a5838484'
    gitlab_rails['db_password'] = 'mypassword'

    ##
    ## Enable FDW support for the Geo Tracking Database (improves performance)
    ##
    geo_secondary['db_fdw'] = true
    ```

    For external PostgreSQL instances, [see additional instructions][external postgresql].
    If you bring a former primary back online to serve as a secondary then you also need to remove `geo_primary_role['enable'] = true`.

1. Reconfigure GitLab for the changes to take effect:

    ```bash
    gitlab-ctl reconfigure
    ```

1. Restart PostgreSQL for the IP change to take effect and Reconfigure again:

    ```bash
    gitlab-ctl restart postgresql
    gitlab-ctl reconfigure
    ```
    
    This last reconfigure will provision the FDW configuration and enable it.

### Step 3. Initiate the replication process

Below we provide a script that connects the database on the secondary node to
the database on the primary node, replicates the database, and creates the
needed files for streaming replication.

The directories used are the defaults that are set up in Omnibus. If you have
changed any defaults or are using a source installation, configure it as you
see fit replacing the directories and paths.

CAUTION: **Warning:**
Make sure to run this on the **secondary** server as it removes all PostgreSQL's
data before running `pg_basebackup`.

1. SSH into your GitLab **secondary** server and login as root:

    ```
    sudo -i
    ```

1. Choose a database-friendly name to use for your secondary to
   use as the replication slot name. For example, if your domain is
   `secondary.geo.example.com`, you may use `secondary_example` as the slot
   name as shown in the commands below.

1. Execute the command below to start a backup/restore and begin the replication
    CAUTION: **Warning:** Each Geo secondary must have its own unique replication slot name.
    Using the same slot name between two secondaries will break PostgreSQL replication.

    ```bash
    gitlab-ctl replicate-geo-database --slot-name=secondary_example --host=1.2.3.4
    ```

    When prompted, enter the _plaintext_ password you set up for the `gitlab_replicator`
    user in the first step.

    This command also takes a number of additional options. You can use `--help`
    to list them all, but here are a couple of tips:
       - If PostgreSQL is listening on a non-standard port, add `--port=` as well.
       - If your database is too large to be transferred in 30 minutes, you will need
         to increase the timeout, e.g., `--backup-timeout=3600` if you expect the
         initial replication to take under an hour.
       - Pass `--sslmode=disable` to skip PostgreSQL TLS authentication altogether
         (e.g., you know the network path is secure, or you are using a site-to-site
         VPN). This is **not** safe over the public Internet!
       - You can read more details about each `sslmode` in the
         [PostgreSQL documentation][pg-docs-ssl];
         the instructions above are carefully written to ensure protection against
         both passive eavesdroppers and active "man-in-the-middle" attackers.
       - Change the `--slot-name` to the name of the replication slot
         to be used on the primary database. The script will attempt to create the
         replication slot automatically if it does not exist.
       - If you're repurposing an old server into a Geo secondary, you'll need to
         add `--force` to the command line.
       - When not in a production machine you can disable backup step if you
         really sure this is what you want by adding `--skip-backup`

1. Verify that the secondary is configured correctly and that the primary is
   reachable:

    ```
    gitlab-rake gitlab:geo:check
    ```

The replication process is now complete.

### External PostgreSQL instances

For installations using external PostgreSQL instances, the `geo_primary_role`
and `geo_secondary_role` includes configuration changes that must be applied
manually.

The `geo_primary_role` makes configuration changes to `pg_hba.conf` and
`postgresql.conf` on the primary:

```
##
## Geo Primary
## - pg_hba.conf
##
host    replication gitlab_replicator <trusted secondary IP>/32     md5
```

```
##
## Geo Primary Role
## - postgresql.conf
##
sql_replication_user = gitlab_replicator
wal_level = hot_standby
max_wal_senders = 10
wal_keep_segments = 50
max_replication_slots = 1 # number of secondary instances
hot_standby = on
```

The `geo_secondary_role` makes configuration changes to `postgresql.conf` and
enables the Geo Log Cursor (`geo_logcursor`) and secondary tracking database
on the secondary. The PostgreSQL settings for this database it adds to
the default settings:

```
##
## Geo Secondary Role
## - postgresql.conf
##
wal_level = hot_standby
max_wal_senders = 10
wal_keep_segments = 10
hot_standby = on
```

#### Tracking Database for the Secondary nodes

NOTE: **Note**:
You only need to follow the steps below if you are not using the managed
PostgreSQL from a Omnibus GitLab package.

Geo secondary nodes use a tracking database to keep track of replication
status and recover automatically from some replication issues. 

This is a separate PostgreSQL installation that can be configured to use
FDW to connect with the secondary database for improved performance.

To enable an external PostgreSQL instance as tracking database, follow
the instructions below:

1. Edit `/etc/gitlab/gitlab.rb` with the connection params and credentials

    ```ruby
    # note this is shared between both databases, 
    # make sure you define the same password in both
    gitlab_rails['db_password'] = 'mypassword'
    
    geo_secondary['db_host'] = '2.3.4.5' # change to the correct public IP
    geo_secondary['db_port'] = 5431      # change to the correct port
    geo_secondary['db_fdw'] = true       # enable FDW
    geo_postgresql['enable'] = false     # don't use internal managed instance
    ```

1. Reconfigure GitLab for the changes to take effect:

    ```bash
    gitlab-ctl reconfigure
    ```
    
1. Run the tracking database migrations:

    ```bash
    gitlab-rake geo:db:migrate
    ```

1. Configure the [PostgreSQL FDW][FDW] connection and credentials:

    Save the script below in a file, ex. `/tmp/geo_fdw.sh` and modify the connection
    params to match your environment. Execute it to setup the FDW connection.
    
    ```bash
    #!/bin/bash
  
    # Secondary Database connection params:
    DB_HOST="5.6.7.8" # change to the public IP or VPC private IP
    DB_NAME="gitlabhq_production"
    DB_USER="gitlab"
    DB_PORT="5432"
    
    # Tracking Database connection params:
    GEO_DB_HOST="2.3.4.5" # change to the public IP or VPC private IP
    GEO_DB_NAME="gitlabhq_geo_production"
    GEO_DB_USER="gitlab_geo"
    GEO_DB_PORT="5432"
 
    query_exec () {
      gitlab-psql -h $GEO_DB_HOST -d $GEO_DB_NAME -p $GEO_DB_PORT -c "${1}"
    }
 
    query_exec "CREATE EXTENSION postgres_fdw;"
    query_exec "CREATE SERVER gitlab_secondary FOREIGN DATA WRAPPER postgres_fdw OPTIONS (host '${DB_HOST}', dbname '${DB_NAME}', port '${DB_PORT}');"
    query_exec "CREATE USER MAPPING FOR ${GEO_DB_USER} SERVER gitlab_secondary OPTIONS (user '${DB_USER}');"
    query_exec "CREATE SCHEMA gitlab_secondary;"
    query_exec "GRANT USAGE ON FOREIGN SERVER gitlab_secondary TO ${GEO_DB_USER};"
    ```
    
    NOTE: **Note:** The script template above uses `gitlab-psql` as it's intended to be executed from the Geo machine,
    but you can change it to `psql` and run it from any machine that has access to the database.
    
1. Restart GitLab

    ```bash
    gitlab-ctl restart
    ```

## PGBouncer support (optional)

[PGBouncer](http://pgbouncer.github.io/) may be used with GitLab Geo to pool
PostgreSQL connections. We recommend using PGBouncer if you use GitLab in a
high-availability configuration with a cluster of nodes supporting a Geo
primary and another cluster of nodes supporting a Geo secondary. For more
information, see the [Omnibus HA](https://docs.gitlab.com/ee/administration/high_availability/database.html#configure-using-omnibus-for-high-availability)
documentation.

For a Geo secondary to work properly with PGBouncer in front of the database,
it will need a separate read-only user to make [PostgreSQL FDW queries][FDW]
work:

1. On the primary Geo database, enter the PostgreSQL on the console as an
admin user. If you are using an Omnibus-managed database, log onto the primary
node that is running the PostgreSQL database:

    ```bash
     sudo -u gitlab-psql /opt/gitlab/embedded/bin/psql -h /var/opt/gitlab/postgresql gitlabhq_production
    ```

2. Then create the read-only user:

    ```sql
    -- NOTE: Use the password defined earlier
    CREATE USER gitlab_geo_fdw WITH password 'mypassword';
    GRANT CONNECT ON DATABASE gitlabhq_production to gitlab_geo_fdw;
    GRANT USAGE ON SCHEMA public TO gitlab_geo_fdw;
    GRANT SELECT ON ALL TABLES IN SCHEMA public TO gitlab_geo_fdw;
    GRANT SELECT ON ALL SEQUENCES IN SCHEMA public TO gitlab_geo_fdw;

    -- Tables created by "gitlab" should be made read-only for "gitlab_geo_fdw"
    -- automatically.
    ALTER DEFAULT PRIVILEGES FOR USER gitlab IN SCHEMA public GRANT SELECT ON TABLES TO gitlab_geo_fdw;
    ALTER DEFAULT PRIVILEGES FOR USER gitlab IN SCHEMA public GRANT SELECT ON SEQUENCES TO gitlab_geo_fdw;
    ```

3. On the Geo secondary nodes, change `/etc/gitlab/gitlab.rb`:

    ```
    geo_postgresql['fdw_external_user'] = 'gitlab_geo_fdw'
    ```

4. Save the file and reconfigure GitLab for the changes to be applied:

    ```bash
    gitlab-ctl reconfigure
    ```

## MySQL replication

MySQL replication is not supported for Geo.

## Troubleshooting

Read the [troubleshooting document](troubleshooting.md).

[replication-slots-article]: https://medium.com/@tk512/replication-slots-in-postgresql-b4b03d277c75
[pgback]: http://www.postgresql.org/docs/9.2/static/app-pgbasebackup.html
[replication user]:https://wiki.postgresql.org/wiki/Streaming_Replication
[external postgresql]: #external-postgresql-instances
[tracking]: database_source.md#enable-tracking-database-on-the-secondary-server
[FDW]: https://www.postgresql.org/docs/9.6/static/postgres-fdw.html
[toc]: index.md#using-omnibus-gitlab
[database-source]: database_source.md
[database-source-primary]: database_source.md#step-1-configure-the-primary-server
[rake-maintenance]: ../../raketasks/maintenance.md
[pg-docs-ssl]: https://www.postgresql.org/docs/9.6/static/libpq-ssl.html#LIBPQ-SSL-PROTECTION
[pg-docs-runtime-conn]: https://www.postgresql.org/docs/9.6/static/runtime-config-connection.html
[pg-docs-runtime-replication]: https://www.postgresql.org/docs/9.6/static/runtime-config-replication.html
