# frozen_string_literal: true

require 'spec_helper'

require_migration!

RSpec.describe AddInsertOrUpdateVulnerabilityReadsTrigger, feature_category: :vulnerability_management do
  let(:migration) { described_class.new }
  let(:vulnerabilities) { table(:vulnerabilities) }
  let(:vulnerability_reads) { table(:vulnerability_reads) }
  let(:vulnerabilities_findings) { table(:vulnerability_occurrences) }
  let(:namespace) { table(:namespaces).create!(name: 'user', path: 'user') }
  let(:user) { table(:users).create!(id: 13, email: 'author@example.com', username: 'author', projects_limit: 10) }
  let(:project) { table(:projects).create!(id: 123, namespace_id: namespace.id) }
  let(:scanner) { table(:vulnerability_scanners).create!(project_id: project.id, external_id: 'test 1', name: 'test scanner 1') }

  let(:vulnerability) do
    create_vulnerability!(
      project_id: project.id,
      author_id: user.id
    )
  end

  let(:vulnerability2) do
    create_vulnerability!(
      project_id: project.id,
      author_id: user.id
    )
  end

  let(:identifier) do
    table(:vulnerability_identifiers).create!(
      project_id: project.id,
      external_type: 'uuid-v5',
      external_id: 'uuid-v5',
      fingerprint: '7e394d1b1eb461a7406d7b1e08f057a1cf11287a',
      name: 'Identifier for UUIDv5')
  end

  let(:finding) do
    create_finding!(
      project_id: project.id,
      scanner_id: scanner.id,
      primary_identifier_id: identifier.id
    )
  end

  describe '#up' do
    before do
      migrate!
    end

    describe 'UPDATE trigger' do
      context 'when vulnerability_id is updated' do
        it 'creates a new vulnerability_reads row' do
          expect do
            finding.update!(vulnerability_id: vulnerability.id)
          end.to change { vulnerability_reads.count }.from(0).to(1)
        end
      end

      context 'when vulnerability_id is not updated' do
        it 'does not create a new vulnerability_reads row' do
          finding.update!(vulnerability_id: nil)

          expect do
            finding.update!(location: '')
          end.not_to change { vulnerability_reads.count }
        end
      end
    end

    describe 'INSERT trigger' do
      context 'when vulnerability_id is set' do
        it 'creates a new vulnerability_reads row' do
          expect do
            create_finding!(
              vulnerability_id: vulnerability2.id,
              project_id: project.id,
              scanner_id: scanner.id,
              primary_identifier_id: identifier.id
            )
          end.to change { vulnerability_reads.count }.from(0).to(1)
        end
      end

      context 'when vulnerability_id is not set' do
        it 'does not create a new vulnerability_reads row' do
          expect do
            create_finding!(
              project_id: project.id,
              scanner_id: scanner.id,
              primary_identifier_id: identifier.id
            )
          end.not_to change { vulnerability_reads.count }
        end
      end
    end
  end

  describe '#down' do
    before do
      migration.up
      migration.down
    end

    it 'drops the trigger' do
      expect do
        finding.update!(vulnerability_id: vulnerability.id)
      end.not_to change { vulnerability_reads.count }
    end
  end

  private

  def create_vulnerability!(project_id:, author_id:, title: 'test', severity: 7, confidence: 7, report_type: 0)
    vulnerabilities.create!(
      project_id: project_id,
      author_id: author_id,
      title: title,
      severity: severity,
      confidence: confidence,
      report_type: report_type
    )
  end

  # rubocop:disable Metrics/ParameterLists
  def create_finding!(
    project_id:, scanner_id:, primary_identifier_id:, vulnerability_id: nil,
    name: "test", severity: 7, confidence: 7, report_type: 0,
    project_fingerprint: '123qweasdzxc', location: { "image" => "alpine:3.4" }, location_fingerprint: 'test',
    metadata_version: 'test', raw_metadata: 'test', uuid: SecureRandom.uuid)
    vulnerabilities_findings.create!(
      vulnerability_id: vulnerability_id,
      project_id: project_id,
      name: name,
      severity: severity,
      confidence: confidence,
      report_type: report_type,
      project_fingerprint: project_fingerprint,
      scanner_id: scanner_id,
      primary_identifier_id: primary_identifier_id,
      location: location,
      location_fingerprint: location_fingerprint,
      metadata_version: metadata_version,
      raw_metadata: raw_metadata,
      uuid: uuid
    )
  end
  # rubocop:enable Metrics/ParameterLists
end
