# frozen_string_literal: true

require 'spec_helper'

RSpec.describe GitlabSchema.types['VulnerabilityEvidenceSource'] do
  let_it_be(:project) { create(:project) }
  let(:fields) do
    %i[identifier name url]
  end

  let_it_be(:user) { create(:user) }
  let_it_be(:pipeline) { create(:ee_ci_pipeline, :with_api_fuzzing_report, project: project) }

  before do
    stub_licensed_features(api_fuzzing: true, security_dashboard: true)

    project.add_developer(user)
  end

  subject { GitlabSchema.execute(query, context: { current_user: user }).as_json }

  specify { expect(described_class.graphql_name).to eq('VulnerabilityEvidenceSource') }

  it { expect(described_class).to have_graphql_fields(fields) }

  describe 'checking field contents' do
    let(:query) do
      %(
        query {
          project(fullPath: "#{project.full_path}") {
            pipeline(iid: "#{pipeline.iid}") {
              securityReportFindings {
                nodes {
                  title
                  evidence {
                    source {
                      identifier
                      name
                      url
                    }
                  }
                }
              }
            }
          }
        }
      )
    end

    it 'checks the contents of the evidence.source field' do
      vulnerabilities = subject.dig('data', 'project', 'pipeline', 'securityReportFindings', 'nodes')

      vulnerability = vulnerabilities.find { |v| v['title'] == "CORS misconfiguration at 'http://127.0.0.1:7777/api/users'" }

      source = vulnerability['evidence']['source']

      expect(source['name']).to eq("CORS - Bad 'Origin' value")
      expect(source['identifier']).to eq("assert:CORS - Bad 'Origin' value")
      expect(source['url']).to eq("https://developer.mozilla.org/en-US/docs/Web/HTTP/CORS/Errors/CORSMissingAllowOrigin")
    end
  end
end
